﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.Net;
using System.Net.Sockets;
using System.Threading;

namespace NintendoWare.Generic.Preview.MCS.Sync
{
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Sync;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Tool;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Viewer;
    using NintendoWare.SoundMaker.Preview.MCS;

    /// <summary>
    /// ビューア接続
    /// <para>
    /// ビューアからツールに送られるパケットを処理する接続です。
    /// </para>
    /// </summary>
    public class MCSSyncConnection : MCSConnection
    {
        static MCSSyncConnection()
        {
            SyncPacket.Factories.Add(new SyncPacketFactory());
            SyncPacket.Factories.Add(new QueryInvalidItemPacketFactory());
        }

        // 読み込み間隔
        private const int Interval = 100;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MCSSyncConnection()
        {
        }

        /// <summary>
        /// チャンネルの取得。
        /// </summary>
        public override uint Channel
        {
            get { return _baseChannel + 0x2; }
        }

        /// <summary>
        /// 接続
        /// </summary>
        public override bool Connect(IPEndPoint ipEndPoint, string hostName)
        {
            bool result = base.Connect(ipEndPoint, hostName);
            if (result)
            {
#if USE_TIMEOUT
                Client.ReceiveTimeout = Interval;
                Client.SendTimeout = Interval;
#endif

                SreSyncPacket.ResetTimeOut();
            }
            return result;
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// メインループ。
        /// </summary>
        public override void MainLoop()
        {
#if USE_TIMEOUT
            // タイムアウトを利用した読み込み待ち
            MCSPacketHeader header = null;
            try
            {
                header = new MCSPacketHeader(Reader);
                Debug.Assert(!header.Error);
            }
            catch(IOException exception)
            {
                Exception inner = exception.InnerException;
                if((inner != null) && (inner is SocketException)){ return; }
                throw exception;
            }
#else

            // ネットワークストリームのDataAvailableを利用した読み込み待ち
            if (!((NetworkStream)Reader.BaseStream).DataAvailable)
            {
                Thread.Sleep(Interval);

                if (SreSyncPacket.DoTimeOut != false)
                {
                    throw new TimeoutException("Sync connection time out");
                }

                return;
            }
            CommPacketHeader header = new SyncPacketHeader(Reader);
            Debug.Assert(!header.Error, "Header is null");
#endif

            SyncPacket packet = SyncPacket.Create(header);
            if (packet != null)
            {
                packet.Read(Reader, header);
                //ShowMessage( "  Sync ← " + packet );
                if (packet.Reply)
                {
                    CommPacketHeader replyHeader = packet.Write(Writer);
                    Writer.Flush();
                    //ShowMessage( "       → " + replyHeader );
                }
            }
        }
    }
}
