﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;

namespace NintendoWare.Preview.HIO.Sound.Tool
{
  using Communications;
  using NintendoWare.SoundMaker.Preview;
  using NW4R.ProtocolSound;

    public class GetSeqVariablesPacket : HIOToolSoundPacket
    {
        private SeqVariableContainerCollectionGeneric _variableSets;

        public GetSeqVariablesPacket(SeqVariableContainerCollectionGeneric variableSets)
        {
            Debug.Assert( null != variableSets, "VariableSets is null");
            _variableSets = variableSets;
        }

        #region ** プロパティのオーバーライド

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_GET_SEQVARIABLES; }
        }

        /// <summary>
        /// データサイズ
        /// </summary>
        public override ushort Size
        {
            get { return (ushort)( ToolSoundItemContainer.StructSize + ItemsSize ); }
        }

        private ushort ItemsSize
        {
            get
            {
                int size = 0;

                foreach( SeqVariableContainerGeneric item in _variableSets ) {
                    size += item.GetWritableSize( true );
                }

                Debug.Assert( ushort.MaxValue >= size, "Size exceeds the MaxValue");
                Debug.Assert( ushort.MinValue <= size, "Size below the MinValue");

                return (ushort)size;
            }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// パケットを作成して送信します。
        /// </summary>
        /// <param name="effectParam">エフェクトパラメータ</param>
        public static void Send(SeqVariableContainerCollectionGeneric variables)
        {
            //if( !MCSManager.IsConnected ) { return; }
            if( CommManager.Instance.IsConnected == false )
            {
                return;
            }
            SendPacket( new GetSeqVariablesPacket( variables ) );
        }

        #endregion

        #region ** メソッドのオーバーライド

        /// <summary>
        /// 送信パケットの圧縮
        /// </summary>
        public override bool Compress(HIOToolPacket nextPacket)
        {
            GetSeqVariablesPacket next = nextPacket as GetSeqVariablesPacket;
            if( null == next ) { return false; }

            if( _variableSets.Count != next._variableSets.Count ) { return false; }

            for( int i=0; i<_variableSets.Count; i++ ) {

                if( _variableSets[ i ].ID != next._variableSets[ i ].ID ) { return false; }

            }

            return true;
        }

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        public override void Read(ProtocolSoundReader reader, HIOPacketHeader header)
        {
            //Validator.Assert( false );
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public override HIOPacketHeader Write(ProtocolSoundWriter writer)
        {
            Debug.Assert( null != writer, "Writer is null");

            MemoryStream        memStream = new MemoryStream();
            ProtocolSoundWriter memWriter = new ProtocolSoundWriter( memStream );


            // パケットヘッダを出力する
            HIOPacketHeader header = CreateHeader();
            header.Write( memWriter );

            Debug.Assert( null != _variableSets, "VariableSets is null");
            Debug.Assert( _variableSets.Count > 0, "VarialbeSets below the zero");


            // シーケンス変数取得用ヘッダを出力する
            ToolSoundItemContainer container = new ToolSoundItemContainer();

            container.offsetForData = ToolSoundItemContainer.StructSize;
            container.itemCount     = (uint)_variableSets.Count;
            container.itemSize      = (uint)_variableSets[ 0 ].GetWritableSize( true );

            container.Write( memWriter );

            // シーケンス変数取得パラメータを出力する
            foreach( SeqVariableContainerGeneric item in _variableSets ) {
                item.Write( memWriter, true );
            }

            // この１行は仮実装
            // Send用ToolPacket, Recv用ViewerPacketとして１つのクラスにまとめる
            NintendoWare.Preview.HIO.Sound.Viewer.GetSeqVariablesPacket.Received = false;

            writer.Write( memStream.GetBuffer(), 0, (int)memStream.Length );

            return header;
        }

        #endregion
    }

    public class SetSeqVariablesPacket : HIOToolSoundPacket
    {
        private SeqVariableContainerCollectionGeneric _variableSets;

        public SetSeqVariablesPacket(SeqVariableContainerCollectionGeneric variableSets)
        {
            Debug.Assert( null != variableSets, "VariableSets is null" );
            _variableSets = variableSets;
        }

        #region ** プロパティのオーバーライド

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_SET_SEQVARIABLES; }
        }

        /// <summary>
        /// データサイズ
        /// </summary>
        public override ushort Size
        {
            get { return (ushort)( ToolSoundItemContainer.StructSize + ItemsSize ); }
        }

        private ushort ItemsSize
        {
            get
            {
                int size = 0;

                foreach( SeqVariableContainerGeneric item in _variableSets ) {
                    size += item.GetWritableSize( false );
                }

                Debug.Assert( ushort.MaxValue >= size, "Size exceeds the MaxValue");
                Debug.Assert( ushort.MinValue <= size, "Size below the MinValue");

                return (ushort)size;
            }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// パケットを作成して送信します。
        /// </summary>
        /// <param name="effectParam">エフェクトパラメータ</param>
        public static void Send(SeqVariableContainerCollectionGeneric variables)
        {
            //if( !MCSManager.IsConnected ) { return; }
          if (CommManager.Instance.IsConnected == false)
          {
            return;
          }
            SendPacket( new SetSeqVariablesPacket( variables ) );
        }

        #endregion

        #region ** メソッドのオーバーライド

        /// <summary>
        /// 送信パケットの圧縮
        /// </summary>
        public override bool Compress(HIOToolPacket nextPacket)
        {
            return false;
        }

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        public override void Read(ProtocolSoundReader reader, HIOPacketHeader header)
        {
            // TOOL_SOUND_SET_SEQVARIABLES は Write のみ
            Debug.Assert( false, "Not override Read method");
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public override HIOPacketHeader Write(ProtocolSoundWriter writer)
        {
            Debug.Assert( null != writer, "Writer is null");

            MemoryStream        memStream = new MemoryStream();
            ProtocolSoundWriter memWriter = new ProtocolSoundWriter( memStream );


            // パケットヘッダを出力する
            HIOPacketHeader header = CreateHeader();
            header.Write( memWriter );

            Debug.Assert( null != _variableSets, "VariableSets is null");
            Debug.Assert( _variableSets.Count > 0, "VariableSets below the zero");


            // シーケンス変数取得用ヘッダを出力する
            ToolSoundItemContainer container = new ToolSoundItemContainer();

            container.offsetForData = ToolSoundItemContainer.StructSize;
            container.itemCount     = (uint)_variableSets.Count;
            container.itemSize      = (uint)_variableSets[ 0 ].GetWritableSize( false );

            container.Write( memWriter );

            // シーケンス変数取得パラメータを出力する
            foreach( SeqVariableContainerGeneric item in _variableSets ) {
                item.Write( memWriter, false );
            }


            writer.Write( memStream.GetBuffer(), 0, (int)memStream.Length );

            return header;
        }

        #endregion
    }
}
