﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.Preview.HIO
{
    using System.Collections;
    using System.Diagnostics;
    using System.Threading;

    /// <summary>
    /// ツール接続
    /// <para>
    /// ツールからビューアに送るパケットを処理する接続です。
    /// </para>
    /// </summary>
    public class HIOToolConnection : HIOConnection
    {
        #region ** 定数

        //---------------------------------------------------------------------
        // 定数
        //---------------------------------------------------------------------
        // パケットがない時のインターバル
        private const int IntervalNothingPacket = 50;
        // パケットがある時のインターバル
        private const int IntervalExistPacket = 50;

        private const int SoundChannel = 8;

        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public HIOToolConnection() { }

        /// <summary>
        /// チャンネルの取得
        /// </summary>
        public override int Channel
        {
            get { return SoundChannel; }
        }

        //---------------------------------------------------------------------
        // 接続関係
        //---------------------------------------------------------------------
        /// <summary>
        /// 接続
        /// </summary>
        public override bool Connect()
        {
            Debug.Assert(_packetQueue.Count == 0, "Packet queue count is not zero");
            bool result = base.Connect();
            if (result)
            {
                // リセット
                HIOResetPacket.Send();
            }
            return result;
        }

        /// <summary>
        /// 接続を閉じる
        /// </summary>
        public override void Disconnect(int timeout)
        {
            DisconnectingFlag = true;
            base.Disconnect(timeout);
            lock (_packetQueue) { _packetQueue.Clear(); }
        }

        //---------------------------------------------------------------------
        // パケット関係
        //---------------------------------------------------------------------
        /// <summary>
        /// パケット送信
        /// </summary>
        public void SendPacket(HIOToolPacket packet)
        {
            // キューにパケットを追加する
            lock (_packetQueue)
            {
                if (_packetQueue.Count > 512) return;

                // パケット圧縮
                if (_packetQueue.Count > 0)
                {
                    if (_lastPacket.Compress(packet)) { return; }
                }
                _packetQueue.Enqueue(packet);
                _lastPacket = packet;

                _queuingEvent.Set();

                //              Debug.WriteLine( string.Format( "+ Queue : {0:C}", _packetQueue.Count ) );
            }

            //ShowMessage("Tool Packet Queue {0}", _packetQueue.Count);
        }

        /// <summary>
        /// パケット数の取得
        /// </summary>
        public int GetPacketCount()
        {
            lock (_packetQueue) { return _packetQueue.Count; }
        }

        /// <summary>
        /// パケット取得
        /// </summary>
        public HIOToolPacket GetPacket()
        {
            // キューからパケットを取り出す
            HIOToolPacket packet = null;
            lock (_packetQueue)
            {
                if (_packetQueue.Count > 0)
                {
                    packet = (HIOToolPacket)_packetQueue.Dequeue();
                    _queuingEvent.Reset();

                    //                  Debug.WriteLine( string.Format( "- Dequeue : {0:N}", _packetQueue.Count ) );
                }
            }
            return packet;
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// メインループ
        /// </summary>
        public override void MainLoop()
        {
            HIOToolPacket packet = GetPacket();
            if (packet == null)
            {
                Stopwatch stopwatch = new Stopwatch();
                stopwatch.Start();
                _queuingEvent.WaitOne(IntervalNothingPacket, false);
                stopwatch.Stop();

                //              Debug.WriteLine( string.Format( "* WaitTime : {0:N}", stopwatch.ElapsedMilliseconds ) );
                return;
            }
            // パケットの書き込み
            HIOPacketHeader packetHeader = packet.Write(this.Writer);
            this.Writer.Flush();
            this.ShowMessage("  Tool → " + packet.ToString());

            if( packet.Reply != false )
            {
                HIOPacketHeader replyHeader = new HIOPacketHeader();
                replyHeader.Read(this.Reader);
                this.ShowMessage("       ← " + replyHeader.ToString());

                //if (replyHeader.IsSucceeded)
                if( replyHeader.Error == false )
                {
                    // 返信処理
                    //Debug.Assert(packet.MessageCategory == replyHeader.MessageCategory);
                    packet.Read(Reader, replyHeader);
                }
                else
                {
#if false
                    // TODO : 未実装
                    // エラー処理
                    HIOErrorPacket errorPacket =
                        HIOErrorPacket.Create(replyHeader, packet);
                    if (errorPacket != null)
                    {
                        errorPacket.Read(Reader, replyHeader);
                    }
                    else
                    {
                        // 未知のエラーパケットはメッセージを出して読み飛ばす
                        ShowMessage(
                                    NW4F_SoundMaker.AnimeSoundApplication.MessageService.Message("HIOToolConnection_UnknownPacket"),
                                    replyHeader.ToString());
                        Reader.ReadBytes(replyHeader.Size);
                    }
#endif
                }
            }
        }

        #region ** フィールド

        //---------------------------------------------------------------------
        // メンバ
        //---------------------------------------------------------------------
        // パケットキュー
        private Queue _packetQueue = new Queue();

        // 最後のパケット
        private HIOToolPacket _lastPacket = null;

        // キューイベント
        private ManualResetEvent _queuingEvent = new ManualResetEvent(false);

        #endregion
    }
}
