﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Preview.HIO
{
    using System;
    using System.Diagnostics;
    using System.Windows.Forms;
    using Communications;

    /// <summary>
    /// マネージャ
    /// </summary>
    public class HIOManager : CommManager
    {
        /// ツール接続(送信)
        private HIOToolConnection _toolConnection;

        /// ビューア接続(受信)
        //private MCSViewerConnection _viewerConnection;

        /// 接続状態
        private ConnectionState _connectionState = ConnectionState.Disconnected;

        /// Lock
        private Object _lock = new Object();

        /// <summary>
        /// 初期化
        /// </summary>
        /// <param name="console">MCSのメッセージ出力先となるコンソール</param>
        public override void Initialize(Control invokeControl, ICommConsole console)
        {
            base.Initialize(invokeControl, console);

            _toolConnection = new HIOToolConnection();
            //_viewerConnection = new MCSViewerConnection();
        }

        //---------------------------------------------------------------------
        // 接続関係
        //---------------------------------------------------------------------
        /// <summary>
        /// 接続
        /// </summary>
        public override bool Connect()
        {
            SetState(ConnectionState.Connecting);

            // ツール接続の確立
            if (_toolConnection.Connect() == false)
            {
                Disconnect();
                return false;
            }

#if false
            // ビューア接続の確立
            if(!_viewerConnection.Connect(ipEndPoint, _hostName))
            {
                Disconnect();
                return false;
            }
#endif

            SetState(ConnectionState.Connected);
            return true;
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// 接続中か
        /// </summary>
        public override bool IsConnected
        {
            get
            {
#if false
                //Debug.Assert(_console != null);
                // 一つでもつながっていれば接続中
                bool result = false;
                result |= _toolConnection.IsConnected;
                result |= _viewerConnection.IsConnected;
                result |= _pingConnection.IsConnected;
                return result;
#else
                return (ConnectionState.Disconnected != _connectionState);
#endif
            }
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// 接続を閉じる
        /// </summary>
        public override void Disconnect()
        {
            if (InvokeControl != null && InvokeControl.InvokeRequired)
            {
                InvokeControl.BeginInvoke(new MethodInvoker(OnDisconnect));
                return;
            }

            OnDisconnect();
        }

        /// <summary>
        /// 強制切断する
        /// </summary>
        public override void Terminate()
        {
            if (InvokeControl != null && InvokeControl.InvokeRequired)
            {
                InvokeControl.BeginInvoke(new MethodInvoker(OnTerminate));
                return;
            }

            OnDisconnect();
        }

        ///
        public override ConnectionState GetState()
        {
            return _connectionState;
        }

        ///
        public override void SetState(ConnectionState state)
        {
            if (InvokeControl != null && InvokeControl.InvokeRequired)
            {
                InvokeControl.Invoke(new SetStateDelegate(OnSetState), state);
                return;
            }

            OnSetState(state);
        }

        //---------------------------------------------------------------------
        // その他公開メンバ
        //---------------------------------------------------------------------

        /// <summary>
        /// パケット送信
        /// </summary>
        public override void SendPacket(CommPacket packet)
        {
            if (!_toolConnection.IsConnected) { return; }
            _toolConnection.SendPacket(packet as HIOToolPacket);
        }

        /// <summary>
        /// 終了待ち
        /// </summary>
        public override bool IsWantToExit
        {
            get
            {
                return _toolConnection.IsWantToExit;
            }
        }

        //---------------------------------------------------------------------
        // 非公開メンバ
        //---------------------------------------------------------------------
        /// <summary>
        /// メッセージ表示
        /// </summary>
        protected void ShowMessage(string format, params object[] args)
        {
            Console.WriteMessage(format, args);
        }

        /// <summary>
        /// エラー表示
        /// </summary>
        protected void ShowError(string format, params object[] args)
        {
            Console.WriteError(format, args);
        }

        ///
        private delegate void SetStateDelegate(ConnectionState state);

        ///
        private void OnSetState(ConnectionState state)
        {
            if (state == _connectionState) { return; }

            //Debug.Assert(!InvokeControl.InvokeRequired);

            lock (_lock)
            {
                if (state == _connectionState) { return; }

                _connectionState = state;

                // 接続状態変更イベント発行
                // 通信スレッドを止めないように非同期でイベント発行する
                OnConnectionChanged
                    (new ConnectionChangedEventArgs(_connectionState, LastError));

                LastError = String.Empty;
            }
        }

        ///
        private void OnDisconnect()
        {
            Debug.Assert(Console != null, "Console is null");

            _toolConnection.Disconnect();
            //_viewerConnection.Disconnect();

            SetState(ConnectionState.Disconnected);
        }

        ///
        private void OnTerminate()
        {
            Debug.Assert(Console != null, "Console is null");

            _toolConnection.Disconnect(0);
            //_viewerConnection.Disconnect( 0 );

            SetState(ConnectionState.Disconnected);
        }
    }

    ///
    public class HIOConsole : ICommConsole
    {
        public void WriteMessage(string format, params object[] args)
        {
            Debug.WriteLine(string.Format(format, args));
        }

        public void WriteError(string format, params object[] args)
        {
            Debug.WriteLine(string.Format(format, args));
        }
    }
}
