﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Net;
using System.Windows.Forms;
//using MCS.Tool;
//using MCS.Viewer;
//using MCS.Ping;

namespace Communications
{
    //-------------------------------------------------------------------------
    // 接続状態変更イベント
    #region ConnectionChangedEvent

    public enum ConnectionState
    {
        Disconnected = 0,   /// 非接続
        Connecting,         /// 接続処理中
        Connected           /// 接続中
    }

    /// <summary>
    /// 接続状態変更イベントハンドラ
    /// <para>
    /// 接続状態が変化したときに発生するイベントのハンドラです。
    /// </para>
    /// </summary>
    public delegate void ConnectionChangedEventHandler(ConnectionChangedEventArgs args);

    /// <summary>
    /// 接続状態変更イベント引数
    /// <para>
    /// このイベント引数により接続したのか切断したのかを判別できます。
    /// </para>
    /// </summary>
    public sealed class ConnectionChangedEventArgs : EventArgs
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ConnectionChangedEventArgs(ConnectionState state)
        {
            _state = state;
        }
        public ConnectionChangedEventArgs(ConnectionState state, string errorText)
        {
            _state     = state;
            _errorText = errorText;
        }

        /// <summary>
        /// 接続中か
        /// </summary>
        public bool IsConnected
        {
            get { return ( ConnectionState.Connected == _state ); }
        }

        /// <summary>
        /// 接続状態
        /// </summary>
        public ConnectionState State
        {
            get { return _state; }
        }

        /// <summary>
        /// エラーが発生したかどうか
        /// </summary>
        public bool IsError
        {
            get { return ( _errorText.Length > 0 ); }
        }

        /// <summary>
        /// エラーテキスト
        /// </summary>
        public string ErrorText
        {
            get { return _errorText; }
        }

        // 接続しているか
        private readonly ConnectionState _state;
        private readonly string          _errorText = String.Empty;
    }
    #endregion

    //-------------------------------------------------------------------------
    // コンソールインターフェイス
    #region ICommConsole
    /// <summary>
    /// コンソールインターフェイス
    /// <para>
    /// メッセージ出力先を指定するためのインターフェイス。
    /// このインターフェイスは他のスレッドから呼ばれることに注意する必要があります。
    /// </para>
    /// </summary>
    public interface ICommConsole
    {
        /// <summary>
        /// メッセージ書き出し
        /// </summary>
        void WriteMessage(string format, params object[] args);

        /// <summary>
        /// エラー書き出し
        /// </summary>
        void WriteError(string format, params object[] args);

    }
    #endregion

    /// <summary>
    /// マネージャ
    /// <para>
    /// 管理を行う静的なクラスです。ツールからビューアへパケットを送る接続
    /// （ツール接続）、ビューアからツールへパケットを送る接続（ビューア接続）、
    /// ビューアへのPingを行う接続（Ping接続）の三種類の接続を管理します。
    /// </para>
    /// </summary>
    public class CommManager
    {
        public static CommManager Instance { get; set; }

        /// <summary>
        /// 接続状態変更イベントハンドラ
        /// </summary>
        public event ConnectionChangedEventHandler ConnectionChanged;

        /// <summary>
        /// 初期化
        /// </summary>
        /// <param name="console">メッセージ出力先となるコンソール</param>
        public virtual void Initialize(Control invokeControl, ICommConsole console)
        {
            if ( null == invokeControl ) {
                this.invokeControl = new Control();
                this.invokeControl.CreateControl();
            }
            else
            {
                this.invokeControl = invokeControl;
            }

            this.console = console;
        }

        /// <summary>
        /// 接続
        /// </summary>
        public virtual bool Connect()
        {
            return false;
        }

        /// <summary>
        /// 接続中か
        /// </summary>
        public virtual bool IsConnected
        {
            get
            {
                return false;
            }
        }

        /// <summary>
        /// 接続を閉じる
        /// </summary>
        public virtual void Disconnect()
        {
        }

        /// <summary>
        /// 強制切断する
        /// </summary>
        public virtual void Terminate()
        {
        }

        /// <summary>
        ///
        /// </summary>
        public virtual ConnectionState GetState()
        {
            return _connectionState;
        }

        public virtual void SetState(ConnectionState state)
        {
        }

        public string LastError
        {
            get { return _errorText; }
            set { _errorText = value; }
        }

        /// <summary>
        /// コンソール
        /// </summary>
        public ICommConsole Console
        {
            get
            {
                Debug.Assert( this.console != null, "Console is null");
                return this.console;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public Control InvokeControl
        {
            get
            {
                //Debug.Assert( this.invokeControl != null);
                return this.invokeControl;
            }
        }

        /// <summary>
        /// パケット送信
        /// </summary>
        public virtual void SendPacket(CommPacket packet)
        {
        }

        /// <summary>
        /// 終了待ち
        /// </summary>
        public virtual bool IsWantToExit {
            get
            {
                return true;
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected void OnConnectionChanged( ConnectionChangedEventArgs e)
        {
            if( ConnectionChanged != null )
            {
#if false
                InvokeControl.BeginInvoke
                    ( ConnectionChanged,
                      new object[] { e });
#else
                ConnectionChanged( e);
#endif
            }
        }

        // Invoke用コントロール
        private Control invokeControl;

        // メッセージ出力用コンソール
        private ICommConsole console;

        // 接続状態
        private ConnectionState _connectionState = ConnectionState.Disconnected;

        // Last Error
        private string _errorText = String.Empty;
    }
}
