﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.IO;
    using System.Linq;
    using System.Xml;
    using System.Xml.Serialization;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate;
    using NintendoWare.ToolDevelopmentKit;
    using System.Collections.Generic;
    using Nintendo.AudioToolkit.DomainModels;

    /// <summary>
    /// サウンドセットドキュメントの読み込みをサポートします。
    /// </summary>
    public class SoundSetDocumentReaderCommon : SoundSetDocumentReader
    {
        private SoundSetXml2ModelTranslatorCommon translator;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SoundSetDocumentReaderCommon(ObjectAggregateFactory<Type, Component> componentFactory)
            : base(componentFactory)
        {
            this.translator = new SoundSetXml2ModelTranslatorCommon(componentFactory);
        }

        public override bool IsSupported(IStreamResource resource)
        {
            if (!(resource is FileResource))
            {
                return false;
            }

            using (FileStream stream =
                    File.Open(resource.Key, FileMode.Open, FileAccess.Read, FileShare.Read))
            {
                using (XmlReader reader = XmlReader.Create(stream))
                {
                    try
                    {
                        XmlSoundSetFileCommon.Validate(reader, this.Platform, this.DocumentVersion);
                    }
                    catch (NotSupportedException)
                    {
                        return false;
                    }
                    catch
                    {
                        throw;
                    }
                }
            }

            return true;
        }

        public override void Read(IStreamResource resource, Document document)
        {
            Ensure.Argument.NotNull(resource);
            Ensure.Argument.True(resource is FileResource);
            Ensure.Argument.True(document is SoundSetDocument);

            SoundSetDocument soundSetDocument = document as SoundSetDocument;

            XmlSoundSetFileCommon xmlSoundSetFileCommon = CreateXmlSoundSetFile(resource);

            translator.PathResolver.BasePath =
                Path.GetDirectoryName(resource.Key.GetFullPath());

            var soundSet = translator.Run(xmlSoundSetFileCommon.Body.SoundSet);

            soundSetDocument.SoundSet = soundSet;
        }

        private XmlSoundSetFileCommon CreateXmlSoundSetFile(IStreamResource resource)
        {
            Assertion.Argument.NotNull(resource);
            Assertion.Argument.True(resource is FileResource);

            using (FileStream fileStream = File.Open(resource.Key, FileMode.Open, FileAccess.Read, FileShare.Read))
            {
                return new XmlSerializer(typeof(XmlSoundSetFileCommon)).Deserialize(fileStream) as XmlSoundSetFileCommon;
            }
        }
    }
}
