﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.BankElements
{
    using System;
    using System.Collections.Generic;
    using System.Collections.Specialized;
    using System.Linq;
    using System.Runtime.InteropServices;
    using Binarization;
    using ToolDevelopmentKit;
    using ToolDevelopmentKit.Collections;

    [StructLayout(LayoutKind.Sequential)]
    internal abstract class InstrumentItemSet<TItem> : IInstrumentItem
        where TItem : class, IInstrumentItem
    {
        private const int RangeTableEntriesLimit = 11;

        private ObservableList<TItem> items = new ObservableList<TItem>();
        private object table = null;
        private IList<IInstrumentItem> data = null;

        public InstrumentItemSet()
        {
            this.items.CollectionChanged += OnVelocityRegionsChanged;
        }

        //-----------------------------------------------------------------
        // データ構造
        //-----------------------------------------------------------------

        [DomObjectReference(
            OriginTag = "Element", HasType = true, HasOffset = true)]
        public object TableReference
        {
            get { return this.Table; }
        }

        public object Table
        {
            get
            {
                if (this.table == null)
                {
                    this.table = this.CreateTable(this.DataInternal);
                }

                return this.table;
            }
        }

        public IEnumerable<IInstrumentItem> Data
        {
            get
            {
                foreach (IInstrumentItem item in this.items)
                {
                    if (item is NullInfo) { continue; }
                    yield return item;
                }
            }
        }

        //-----------------------------------------------------------------
        // パラメータの操作
        //-----------------------------------------------------------------

        [DomIgnoreField]
        public Byte RangeMin { get; set; }

        [DomIgnoreField]
        public Byte RangeMax { get; set; }

        [DomIgnoreField]
        public IList<TItem> Items
        {
            get { return this.items; }
        }

        [DomIgnoreField]
        protected IList<IInstrumentItem> DataInternal
        {
            get
            {
                if (this.data == null)
                {
                    this.data = this.CreateItems();
                }

                return this.data;
            }
        }

        protected virtual object CreateTable(IList<IInstrumentItem> items)
        {
            Assertion.Argument.NotNull(items);

            // アイテム数によって使用するテーブルを決定します。
            //  １つ                ：直接参照テーブルを使います。
            //  １つ以上、規定数未満：範囲参照テーブルを使います。
            //  規定数以上          ：インデックス参照テーブルを使います。

            if (items.Count == 1 && items[0].RangeMin == 0 && items[0].RangeMax == 127)
            {
                return new DirectReferenceTable()
                {
                    ItemReference = items[0],
                };
            }

            int itemCount = (items[0].RangeMin > 0) ? items.Count + 1 : items.Count;

            if (itemCount <= InstrumentItemSet<TItem>.RangeTableEntriesLimit)
            {
                // 範囲参照テーブルの場合のみ、
                // 先頭に空白があれば、NULLオブジェクトを挿入します。
                // 最後尾に空白があれば NULLオブジェクトを挿入します。
                if (items[0].RangeMin > 0)
                {
                    items.Insert(0, new NullInfo(0, (Byte)(items[0].RangeMin - 1)));
                }

                int itemsRangeMax = items[items.Count - 1].RangeMax;
                if (itemsRangeMax < 127)
                {
                    items.Add(new NullInfo((Byte)(itemsRangeMax + 1), 127));
                }

                return new RangeReferenceTable()
                {
                    Entries = items,
                };
            }

            return CreateIndexReferenceTable(items);
        }

        protected virtual IList<IInstrumentItem> CreateItems()
        {
            List<IInstrumentItem> items = new List<IInstrumentItem>();
            int key = 0;

            foreach (IInstrumentItem item in
                this.items.OrderBy(item => item.RangeMin))
            {
                Ensure.Operation.True(key <= item.RangeMax);

                // 前に空白があれば、NULLオブジェクトを挿入します。
                if (key < item.RangeMin)
                {
                    items.Add(new NullInfo((Byte)key, (Byte)(item.RangeMin - 1)));
                }

                items.Add(item);

                key = item.RangeMax + 1;
            }

            return items;
        }

        private IndexReferenceTable CreateIndexReferenceTable(IList<IInstrumentItem> items)
        {
            Assertion.Argument.NotNull(items);
            Ensure.Argument.True(items.Count > 0);

            int itemsRangeMin = items[0].RangeMin;
            int itemsRangeMax = items[items.Count - 1].RangeMax;

            if ((itemsRangeMax - itemsRangeMin + 1) < InstrumentItemSet<TItem>.RangeTableEntriesLimit)
            {
                throw new Exception("internal error : IndexReferenceTable cannot apply.");
            }

            IList<IInstrumentItem> tableItems = new List<IInstrumentItem>();
            int itemIndex = 0;

            for (int i = itemsRangeMin; i <= itemsRangeMax; i++)
            {
                while (items[itemIndex].RangeMax < i)
                {
                    itemIndex++;

                    if (items.Count <= itemIndex)
                    {
                        throw new Exception("internal error : failed to create index reference table.");
                    }
                }

                if (i < items[itemIndex].RangeMin)
                {
                    throw new Exception("internal error : null object not found.");
                }

                tableItems.Add(items[itemIndex]);
            }

            return new IndexReferenceTable()
            {
                Entries = tableItems,
            };
        }

        private void ResetData()
        {
            data = null;
        }

        private void OnVelocityRegionsChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            ResetData();
        }
    }
}
