﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using NintendoWare.SoundFoundation.FileFormats.Binary;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.WaveSoundElements;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// ウェーブサウンドバイナリ DOM の構築をサポートします。
    /// </summary>
    internal class WaveSoundFileBuilder
    {
        private BiquadType2ByteConverter biquadType2ByteConverter;
        private Dictionary<IOutput, uint> waveIDIndexDictionary = new Dictionary<IOutput, uint>();
        private List<WaveID> waveIDs = new List<WaveID>();

        public WaveSoundFileBuilder(string signature, BinaryVersion version)
        {
            Ensure.Argument.NotNull(signature);
            this.Signature = signature;
            this.Version = version;
        }

        public BiquadType2ByteConverter BiquadType2ByteConverter
        {
            get
            {
                if (this.biquadType2ByteConverter == null)
                {
                    this.biquadType2ByteConverter = new BiquadType2ByteConverter();
                }

                return this.biquadType2ByteConverter;
            }
        }

        public string Signature { get; set; }

        public BinaryVersion Version { get; set; }

        public WaveSoundBinary Build(WaveSoundSetBase waveSoundSet, WaveArchiveBase waveArchive)
        {
            Ensure.Argument.NotNull(waveSoundSet);
            Assertion.Argument.NotNull(waveArchive);

            this.Prepare(waveSoundSet, waveArchive);

            WaveSoundBinary binary = new WaveSoundBinary(
                this.Signature,
                this.Version.Major,
                this.Version.Minor,
                this.Version.Micro,
                this.Version.BinaryBugFix);

            this.BuildInfoBlock(binary.InfoBlock.Body, waveSoundSet);

            return binary;
        }

        public void BuildInfoBlock(InfoBlockBody body, WaveSoundSetBase waveSoundSet)
        {
            Assertion.Argument.NotNull(body);
            Assertion.Argument.NotNull(waveSoundSet);

            foreach (WaveID waveID in this.waveIDs)
            {
                body.WaveIDTable.Items.Add(waveID);
            }

            foreach (WaveSoundBase waveSound in waveSoundSet.Children)
            {
                if (!waveSound.IsConvertTarget())
                {
                    continue;
                }

                body.Items.Add(
                    this.CreateMetaData(waveSound)
                    );
            }
        }

        private WaveSoundMetaData CreateMetaData(WaveSoundBase waveSound)
        {
            Assertion.Argument.NotNull(waveSound);

            return new WaveSoundMetaData()
            {
                WaveSoundInfo = CreateWaveSoundInfo(waveSound),
                TrackInfos = CreateTrackInfos(waveSound),
                NoteInfos = CreateNoteInfos(waveSound),
            };
        }

        private WaveSoundInfo CreateWaveSoundInfo(WaveSoundBase waveSound)
        {
            Assertion.Argument.NotNull(waveSound);

            WaveSoundInfo info = new WaveSoundInfo();

            info.Parameters.Pan = new ParameterPan()
            {
                Pan = (Byte)waveSound.Pan,
                SurroundPan = (Byte)waveSound.SurroundPan,
            };

            info.Parameters.Pitch = new ParameterSingle()
            {
                Value = Convert.ToSingle(waveSound.Pitch)
            };

            info.Parameters.Filters = new ParameterFilters()
            {
                LpfFrequency = (byte)waveSound.LPF,
                BiquadType = this.BiquadType2ByteConverter.ConvertTo(waveSound.BiquadType),
                BiquadValue = (byte)waveSound.Biquad,
            };

            info.Parameters.Sends = new ParameterSends()
            {
                MainSend = (Byte)waveSound.Sends.MainSend,
                FxSends = new Byte[]
                {
                    (Byte)waveSound.Sends.AuxASend,
                    (Byte)waveSound.Sends.AuxBSend,
                    (Byte)((SendsCommon)(waveSound.Sends)).AuxCSend,
                },
            };

            info.Parameters.Envelope = new ParameterEnvelope()
            {
                Data = new ParameterAdshrEnvelope()
                {
                    // 現在、ADSH はツールで設定できないが、デフォルト値 (127) を入れておきます。F
                    Attack = (Byte)127,
                    Decay = (Byte)127,
                    Sustain = (Byte)127,
                    Hold = (Byte)127,
                    Release = (Byte)waveSound.ReleaseEnvelope,
                },
            };

            return info;
        }

        private TrackInfoList CreateTrackInfos(WaveSoundBase waveSound)
        {
            Assertion.Argument.NotNull(waveSound);

            TrackInfoList trackInfos = new TrackInfoList();
            trackInfos.Items.Add(CreateTrackInfo(waveSound));

            return trackInfos;
        }

        private TrackInfo CreateTrackInfo(WaveSoundBase waveSound)
        {
            Assertion.Argument.NotNull(waveSound);

            TrackInfo trackInfo = new TrackInfo();
            trackInfo.NoteEvents.Items.Add(new NoteEvent());

            return trackInfo;
        }

        private NoteInfoList CreateNoteInfos(WaveSoundBase waveSound)
        {
            Assertion.Argument.NotNull(waveSound);

            NoteInfoList noteInfos = new NoteInfoList();
            noteInfos.Items.Add(
                this.CreateNoteInfo(waveSound)
                );

            return noteInfos;
        }

        private NoteInfo CreateNoteInfo(WaveSoundBase waveSound)
        {
            Assertion.Argument.NotNull(waveSound);

            return new NoteInfo()
            {
                WaveIDIndex = this.waveIDIndexDictionary[waveSound.GetOutputTarget()],
            };
        }

        private void Prepare(WaveSoundSetBase waveSoundSet, WaveArchiveBase waveArchive)
        {
            Assertion.Argument.NotNull(waveSoundSet);
            Assertion.Argument.NotNull(waveArchive);

            this.waveIDIndexDictionary.Clear();
            this.waveIDs.Clear();

            foreach (WaveSoundBase waveSound in waveSoundSet.Children)
            {
                if (!waveSound.IsConvertTarget())
                {
                    continue;
                }

                IOutput output = waveSound.GetOutputTarget();
                Ensure.Operation.ObjectNotNull(output);

                if (waveIDIndexDictionary.ContainsKey(output)) { continue; }

                waveIDIndexDictionary.Add(output, (uint)waveIDs.Count);

                waveIDs.Add(new WaveID()
                {
                    WaveArchiveID = waveArchive.ID,
                    WaveIndex = this.GetWaveIndex(waveArchive, output),
                });
            }
        }

        private uint GetWaveIndex(WaveArchiveBase waveArchive, IOutput output)
        {
            Assertion.Argument.NotNull(waveArchive);
            Assertion.Argument.NotNull(output);

            uint waveIndex = (uint)waveArchive.GetItemOutputTargets().IndexOf(output);

            if (waveIndex == uint.MaxValue)
            {
                throw new Exception("failed to get wave index.");
            }

            return waveIndex;
        }
    }
}
