﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using FileFormats.NintendoWareBinary;
    using FileFormats.NintendoWareBinary.BankElements;
    using NintendoWare.SoundFoundation.Logs;
    using Projects;
    using ToolDevelopmentKit;

    /// <summary>
    /// バンクバイナリ DOM の構築をサポートします。
    /// </summary>
    internal class BankFileBuilder
    {
        private Dictionary<IOutput, uint> waveIDIndexDictionary = new Dictionary<IOutput, uint>();
        private List<WaveID> waveIDs = new List<WaveID>();

        public BankFileBuilder(string signature, BinaryVersion version)
        {
            Ensure.Argument.NotNull(signature);
            this.Signature = signature;
            this.Version = version;
        }

        public string Signature { get; set; }

        public BinaryVersion Version { get; set; }

        public BankBinary Build(Bank bank, WaveArchiveBase waveArchive)
        {
            Ensure.Argument.NotNull(bank);
            Ensure.Argument.NotNull(waveArchive);

            this.Prepare(bank, waveArchive);

            BankBinary binary = new BankBinary(
                this.Signature,
                this.Version.Major,
                this.Version.Minor,
                this.Version.Micro,
                this.Version.BinaryBugFix);

            this.BuildInfoBlock(binary.InfoBlock.Body, bank);

            return binary;
        }

        public void BuildInfoBlock(InfoBlockBody body, Bank bank)
        {
            foreach (WaveID waveID in this.waveIDs)
            {
                body.WaveIDTable.Items.Add(waveID);
            }

            foreach (Instrument instrument in bank.Children)
            {
                //  Instrument が無効なら処理しません。
                if (!instrument.IsConvertTarget())
                {
                    continue;
                }

                body.InstrumentSet.Items.Add(CreateInstrumentInfo(instrument));
            }
        }

        public InstrumentInfo CreateInstrumentInfo(Instrument instrument)
        {
            Assertion.Argument.NotNull(instrument);
            Assertion.Argument.True(instrument.IsConvertTarget());

            InstrumentInfo instrumentInfo = new InstrumentInfo()
            {
                ProgramNo = instrument.ProgramNo,
                Name = instrument.Name,
                IsEnabled = instrument.IsConvertTarget(),
            };

            foreach (KeyRegion keyRegion in instrument.Children)
            {
                //  KeyRegion が無効なら処理しません。
                if (!keyRegion.IsConvertTarget())
                {
                    continue;
                }

                instrumentInfo.Items.Add(
                    CreateKeyRegionInfo(keyRegion));
            }

            return instrumentInfo;
        }

        public KeyRegionInfo CreateKeyRegionInfo(KeyRegion keyRegion)
        {
            Assertion.Argument.NotNull(keyRegion);
            Assertion.Argument.True(keyRegion.IsConvertTarget());

            if (keyRegion.KeyMin > keyRegion.KeyMax)
            {
                string message = string.Format(
                    Resources.MessageResource.Message_InvalidKeyRange,
                    keyRegion.KeyMin,
                    keyRegion.KeyMax);

                throw new ConversionException(
                    new ErrorLine(message, keyRegion.Instrument));
            }

            KeyRegionInfo keyRegionInfo = new KeyRegionInfo()
            {
                RangeMin = (Byte)keyRegion.KeyMin,
                RangeMax = (Byte)keyRegion.KeyMax,
            };

            foreach (VelocityRegion velocityRegion in keyRegion.Children)
            {
                //  VelocityRegion が無効なら処理しません。
                if (!velocityRegion.IsConvertTarget())
                {
                    continue;
                }

                keyRegionInfo.Items.Add(
                    CreateVelocityRegionInfo(velocityRegion));
            }

            return keyRegionInfo;
        }

        public VelocityRegionInfo CreateVelocityRegionInfo(VelocityRegion velocityRegion)
        {
            Assertion.Argument.NotNull(velocityRegion);
            Assertion.Argument.NotNull(waveIDIndexDictionary);

            if (velocityRegion.VelocityMin > velocityRegion.VelocityMax)
            {
                string message = string.Format(
                    Resources.MessageResource.Message_InvalidVelocityRange,
                    velocityRegion.VelocityMin,
                    velocityRegion.VelocityMax);

                throw new ConversionException(
                    new ErrorLine(message, velocityRegion.Instrument));
            }

            VelocityRegionInfo velocityRegionInfo = new VelocityRegionInfo()
            {
                RangeMin = (Byte)velocityRegion.VelocityMin,
                RangeMax = (Byte)velocityRegion.VelocityMax,
                WaveIDIndex = this.waveIDIndexDictionary[velocityRegion.GetOutputTarget()],
            };

            velocityRegionInfo.Parameters.Key = new ParameterKey()
            {
                OriginalKey = (Byte)velocityRegion.OriginalKey,
            };

            velocityRegionInfo.Parameters.Volume = new ParameterVolume()
            {
                Volume = (Byte)CalcVolume(velocityRegion),
            };

            velocityRegionInfo.Parameters.Pan = new ParameterPan()
            {
                Pan = CalcPan(velocityRegion),
            };

            velocityRegionInfo.Parameters.Pitch = new ParameterSingle()
            {
                Value = CalcPitch(velocityRegion),
            };

            Envelope envelope = GetEnvelope(velocityRegion);

            velocityRegionInfo.Parameters.Envelope = new ParameterEnvelope()
            {
                Data = new ParameterAdshrEnvelope()
                {
                    Attack = (Byte)envelope.Attack,
                    Decay = (Byte)envelope.Decay,
                    Sustain = (Byte)envelope.Sustain,
                    Hold = (Byte)envelope.Hold,
                    Release = (Byte)envelope.Release,
                },
            };

            velocityRegionInfo.Parameters.InstrumentNoteParam = new ParameterInstrumentNoteParam()
            {
                IsIgnoreNoteOff = velocityRegion.PercussionMode,
                KeyGroup = (Byte)velocityRegion.KeyGroup,
                InterpolationType = (Byte)velocityRegion.InterpolationType,
            };

            return velocityRegionInfo;
        }

        private Byte CalcVolume(VelocityRegion velocityRegion)
        {
            Assertion.Argument.NotNull(velocityRegion);

            Instrument instrument = velocityRegion.Parent.Parent as Instrument;
            if (instrument == null)
            {
                throw new Exception("internal error");
            }

            double rate = ((double)instrument.Volume / 127.0f) * ((double)velocityRegion.Volume / 127.0f);
            int volume = (int)(127.0f * rate);

            if (volume < 0) { return 0; }
            if (volume > 255) { return 255; }

            return (Byte)volume;
        }

        private Byte CalcPan(VelocityRegion velocityRegion)
        {
            Assertion.Argument.NotNull(velocityRegion);

            int pan = velocityRegion.Pan;

            if (pan < 0) { return 0; }
            if (pan > 127) { return 127; }

            return (Byte)pan;
        }

        private float CalcPitch(VelocityRegion velocityRegion)
        {
            Assertion.Argument.NotNull(velocityRegion);

            Instrument instrument = velocityRegion.Parent.Parent as Instrument;
            if (instrument == null)
            {
                throw new Exception("internal error");
            }

            double pitch = 1.0f;
            pitch *= Math.Pow(2.0f, velocityRegion.PitchCents / 1200.0f);
            pitch *= Math.Pow(2.0f, instrument.PitchCents / 1200.0f);
            pitch *= Math.Pow(2.0f, velocityRegion.PitchSemitones / 12.0f);
            pitch *= Math.Pow(2.0f, instrument.PitchSemitones / 12.0f);

            return Convert.ToSingle(pitch);
        }

        private Envelope GetEnvelope(VelocityRegion velocityRegion)
        {
            Assertion.Argument.NotNull(velocityRegion);

            Instrument instrument = velocityRegion.Parent.Parent as Instrument;
            if (instrument == null)
            {
                throw new Exception("internal error");
            }

            switch (instrument.EnvelopeMode)
            {
                case InstrumentEnvelopeMode.Instrument:
                    return instrument.Envelope;

                case InstrumentEnvelopeMode.VelocityRegion:
                    return velocityRegion.Envelope;

                case InstrumentEnvelopeMode.Lower:        // 未使用
                case InstrumentEnvelopeMode.KeyRegion:    // 未使用
                default:
                    break;
            }

            throw new Exception("internal error : invalid envelope mode.");
        }

        private void Prepare(Bank bank, WaveArchiveBase waveArchive)
        {
            Assertion.Argument.NotNull(bank);
            Assertion.Argument.NotNull(waveArchive);

            this.waveIDIndexDictionary.Clear();
            this.waveIDs.Clear();

            foreach (VelocityRegion velocityRegion in bank.GetVelocityRegions())
            {
                //  Instrument が無効なら処理しません。
                if (!velocityRegion.Instrument.IsConvertTarget())
                {
                    continue;
                }

                IOutput output = velocityRegion.GetOutputTarget();
                Ensure.Operation.ObjectNotNull(output);

                if (waveIDIndexDictionary.ContainsKey(output)) { continue; }

                waveIDIndexDictionary.Add(output, (uint)waveIDs.Count);

                waveIDs.Add(new WaveID()
                {
                    WaveArchiveID = new BinaryItemID(waveArchive.ID),
                    WaveIndex = this.GetWaveIndex(waveArchive, output),
                });
            }
        }

        private uint GetWaveIndex(WaveArchiveBase waveArchive, IOutput output)
        {
            Assertion.Argument.NotNull(waveArchive);
            Assertion.Argument.NotNull(output);

            return (uint)waveArchive.GetItemOutputTargets().IndexOf(output);
        }
    }
}
