﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.IO;
    using Projects;
    using ToolDevelopmentKit;

    internal class WaveOutputFactory<TContext, TComponent> : ComponentSetup<TContext, TComponent>
        where TContext : ComponentContext
        where TComponent : Component
    {
        private readonly FileManager fileManager;
        private readonly Func<TComponent, int?> getSampleRateFunc;

        public WaveOutputFactory(FileManager fileManager, Func<TComponent, int?> getSampleRateFunc)
        {
            Ensure.Argument.NotNull(fileManager);
            Ensure.Argument.NotNull(getSampleRateFunc);
            this.fileManager = fileManager;
            this.getSampleRateFunc = getSampleRateFunc;
        }

        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(TContext context, TComponent component)
        {
            Ensure.Argument.True(component is WaveSoundBase || component is VelocityRegion || component is WaveSoundClipConvertModel);

            IOutput outputTarget = this.fileManager.GetOutput(component);
            component.SetOutputTarget(outputTarget);

            if (!outputTarget.ItemDictionary.ContainsKey(string.Empty))
            {
                string fileName = CreateFileName(component, this.fileManager.BinaryOutputTraits.WaveBinaryFileExtension);
                this.fileManager.RegisterCacheItem(outputTarget, string.Empty, fileName);
            }

            // WaveSoundBase の波形ファイル登録
            // VelocityRegion は BankContext に登録されるので、別途 SoundArchiveContext への登録が必要
            context.AddWaveFile(component, outputTarget);
        }

        /// <summary>
        /// コンポーネントのファイル名を作成します。
        /// </summary>
        /// <param name="component">コンポーネントを指定します。</param>
        /// <param name="extension">拡張子を指定します。</param>
        /// <returns>ファイル名を返します。</returns>
        private string CreateFileName(TComponent component, string extension)
        {
            Assertion.Argument.NotNull(component);
            Assertion.Argument.True(component is WaveSoundBase || component is VelocityRegion);
            Assertion.Argument.NotNull(extension);
            Assertion.Argument.StringNotEmpty(extension);

            string baseFileName = Path.GetFileNameWithoutExtension(component.GetFilePathForConvert());

            if (baseFileName.Length == 0)
            {
                throw new Exception("internal error : invalid file path.");
            }

            var sampleRate = this.getSampleRateFunc(component);

            if (sampleRate.HasValue)
            {
                return string.Format(
                    "{0}.{1}.{2}.{3}",
                    baseFileName,
                    this.GetEncoding(component),
                    sampleRate,
                    extension);
            }
            else
            {
                return string.Format(
                    "{0}.{1}.{2}",
                    baseFileName,
                    this.GetEncoding(component),
                    extension);
            }
        }

        private string GetEncoding(TComponent component)
        {
            WaveEncoding encoding =
                (WaveEncoding)component.Parameters[ProjectParameterNames.WaveEncoding].Value;

            switch (encoding)
            {
                case WaveEncoding.Adpcm:
                    return "dspadpcm";

                case WaveEncoding.Pcm16:
                    return "pcm16";

                case WaveEncoding.Pcm8:
                    return "pcm8";
            }

            throw new Exception("internal error : invalid wave encoding.");
        }
    }
}
