﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using Logs;
    using Projects;
    using ToolDevelopmentKit;
    using ToolDevelopmentKit.Collections;

    internal class WaveSoundSetNameResolver : NameResolver<WaveSoundSetBase>
    {
        public WaveSoundSetNameResolver(ComponentDictionary componentDictionary)
            : base(componentDictionary)
        {
        }

        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="componentDictionary">コンポーネントディクショナリを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected override void RunInternal(
            SoundArchiveContext context, ComponentDictionary componentDictionary, WaveSoundSetBase component)
        {
            // 自動設定の場合は ItemGenerator にて処理します。
            if (WaveArchiveConsts.IsConstValue(component.WaveArchiveReference)) { return; }

            WaveArchiveBase waveArchive = GetWaveArchive(componentDictionary, component);

            if (waveArchive == null || !waveArchive.IsConvertTarget())
            {
                context.Logger.AddLine(
                    new ErrorLine(
                        string.Format(Resources.MessageResource.Message_WaveArchiveNotFound, component.WaveArchiveReference),
                        component)
                    );
                return;
            }

            // メインサウンドアーカイブから追加サウンドアーカイブのバンクを参照している場合はエラーにする
            if (context.AddonSoundSet == null)
            {
                if (context.Project.IsItemInAddonSoundArchive(waveArchive))
                {
                    context.Logger.AddLine(
                        new ErrorLine(
                            string.Format(Resources.MessageResource.Message_WaveArchiveNotFoundInSoundArchive, waveArchive.Name),
                            component)
                        );
                    return;
                }
            }

            // 逆参照の一覧に追加します。
            WaveArchiveEx.ItemList waveArchiveItems = waveArchive.GetItems();

            foreach (var waveSound in component.Children
                .Cast<WaveSound>()
                .Where(waveSound => waveSound.IsConvertTarget()))
            {
                // WaveSound のクリップ利用設定が食い違っていたらエラー終了
                if (context.Traits.IsWaveSound2BinaryEnabled != waveSound.IsWaveSound2BinaryRequired())
                {
                    if (!waveSound.IsWaveSound2BinaryRequired())
                    {
                        context.Logger.AddLine(
                            new ErrorLine(
                                string.Format(Resources.MessageResourceCommon.Message_WaveSoundMustHasNoClips, waveSound.Name),
                                waveSound)
                        );
                    }
                    else
                    {
                        context.Logger.AddLine(
                            new ErrorLine(
                                string.Format(Resources.MessageResourceCommon.Message_WaveSoundMustNotHasNoClips, waveSound.Name),
                                waveSound)
                        );
                    }
                    return;
                }

                foreach (var waveComponent in waveSound.EnumerateWaveComponents(context.Traits.IsWaveSound2BinaryEnabled)
                    .Where(waveComponent => waveComponent.IsConvertTarget())
                    .Where(waveComponent => !waveArchiveItems.Contains(waveComponent)))
                {
                    waveArchiveItems.Add(waveComponent);
                }
            }
        }

        private WaveArchiveBase GetWaveArchive(ComponentDictionary componentDictionary, WaveSoundSetBase component)
        {
            Assertion.Argument.NotNull(componentDictionary);
            Assertion.Argument.NotNull(component);

            if (!componentDictionary.Contains(component.WaveArchiveReference)) { return null; }
            return componentDictionary[component.WaveArchiveReference][0] as WaveArchiveBase;
        }
    }
}
