﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

//#define AAC_SUPPORTED

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using Nintendo.Foundation.IO;
using NintendoWare.SoundFoundation.Binarization;
using NintendoWare.SoundFoundation.Codecs;
using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
using NintendoWare.SoundFoundation.Logs;
using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundFoundation.Resources;
using NintendoWare.SoundFoundation.Utilities;
using NintendoWare.ToolDevelopmentKit;
using WaveFormats = NintendoWare.SoundFoundation.FileFormats.Wave;

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    internal class StreamSoundProcessor : ComponentProcessor<SoundArchiveContext, StreamSoundBase>
    {
        private const string OpusExtension = ".opus";
        private const string CockpitExtension = "cockpit";
        private const string TargetPrefix = "[NW]";
        private const char RegionIndexSplitterChar = '#';
        private const int MaxRegionCount = 128;
        /// <summary> リージョン名の最大長(終端文字を含まない）</summary>
        private const int MaxRegionNameLength = FileFormats.NintendoWareBinary.StreamSoundElements.RegionInfo.MaxName - 1;
        private const int PrefetchStreamBlockCount = 5;
        private const int MaxChannel = 2;
        private const int MaxAacChannel = 6;

        private List<string> warningMessages;
        private IOutputItem prefetchOutputItem;
        private bool isBinaryForPC = false;

        private CodecOutput codecOutput;
        private uint hashValue;

        private bool? hasOriginalLoop;
        private int? originalLoopStart;
        private int? originalLoopEnd;
        private int? originalSampleRate;

        public StreamSoundProcessor(StreamSoundBase component, IOutputItem outputItem, bool isBinaryForPC)
            : base(component, outputItem)
        {
            this.warningMessages = new List<string>();
            this.isBinaryForPC = isBinaryForPC;
        }

        public IOutputItem PrefetchOutputItem
        {
            get { return this.prefetchOutputItem; }
            set { this.prefetchOutputItem = value; }
        }

        public string PreprocessExePath { get; set; }

        protected override bool PreProcessInternal(SoundArchiveContext context)
        {
            try
            {
                foreach (var component in this.Components.Cast<StreamSoundBase>())
                {
                    this.ExtractTrackInfos(context, component);
                }

                if (base.PreProcessInternal(context))
                {
                    return true;
                }

                return this.prefetchOutputItem == null ? false : this.prefetchOutputItem.IsDirty;
            }
            catch (IOException exception)
            {
                context.Logger.AddLine(
                    new ErrorLine(exception.Message, this.TargetComponent)
                    );
                return false;
            }
        }

        protected override void ProcessInternal(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);

            try
            {
                if (File.Exists(this.OutputTargetItem.Path))
                {
                    File.Delete(this.OutputTargetItem.Path);
                }

                if (ShouldCopyFile(this.TargetComponent as StreamSoundBase))
                {
                    CopyFile(this.OutputTargetItem.Path);
                }
                else
                {
                    Write(context, this.OutputTargetItem);

                    if (this.prefetchOutputItem != null)
                    {
                        WritePrefetch(context, this.prefetchOutputItem);
                    }
                }
            }
            catch
            {
                try
                {
                    if (File.Exists(this.OutputTargetItem.Path))
                    {
                        File.Delete(this.OutputTargetItem.Path);
                    }
                }
                catch
                {
                }

                if (this.isBinaryForPC)
                {
                    if (context.Project.DoWarnPCBinariesForAACNotFound)
                    {
                        var track = this.TargetComponent.Children.FirstOrDefault<Component>(item => item.IsEnabled) as StreamSoundTrackBase;
                        Ensure.Operation.ObjectNotNull(track);

                        context.Logger.AddLine(new WarningLine(
                            string.Format(
                                Resources.MessageResource.Message_PCBinariesForAACNotFound,
                                this.TargetComponent.Name,
                                Path.GetFileNameWithoutExtension(track.FilePath)),
                            this.TargetComponent)
                            );
                    }
                    return;
                }

                throw;
            }
            finally
            {
                this.codecOutput = null;
            }
        }

        protected override void PostProcessInternal(SoundArchiveContext context)
        {
            this.SetBinaryPaths(context);
        }

        protected override void OutputLog(SoundArchiveContext context, Component[] components)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(components);

            StreamSoundBase streamSound = components[0] as StreamSoundBase;

            if (this.isBinaryForPC)
            {
                var track = this.TargetComponent.Children.FirstOrDefault<Component>(item => item.IsEnabled) as StreamSoundTrackBase;
                Ensure.Operation.ObjectNotNull(track);

                context.Logger.AddLine(new InformationLine(
                    string.Format("[STRM] {0} (for PC) > {1}",
                    Path.GetFileName(track.FilePath),
                    Path.GetFileName(this.OutputTargetItem.Path)
                    )));
                return;
            }

            string preprocessedTag = this.GetPreprocessedTag(context.ProjectFilePath);

            StringBuilder filePaths = new StringBuilder();
            int trackIndex = 0;

            foreach (StreamSoundTrackBase track in
                streamSound.Children.Where<Component>(item => item.IsEnabled))
            {
                if (filePaths.Length > 0)
                {
                    filePaths.Append(", ");
                }

                filePaths.Append(Path.GetFileName(track.FilePath));

                // Cockpit ログはここで出力するが、実際にフラグを立てるのは、bxsar 処理のタイミング。
                if (track.IsPreprocessed(preprocessedTag))
                {
                    context.Logger.AddLine(new InformationLine(
                        string.Format(
                        Resources.MessageResource.Message_StreamSoundTrackIsCockpit,
                        streamSound.Name,
                        trackIndex,
                        Path.GetFileName(track.FilePath)
                        )));
                }

                ++trackIndex;
            }

            context.Logger.AddLine(new InformationLine(
                string.Format("[STRM] {0} ({1}) > {2}",
                filePaths.ToString(),
                streamSound.Encoding.ToText(),
                Path.GetFileName(this.OutputTargetItem.Path)
                )));
        }

        private StreamSoundBase GetFirstStreamSound()
        {
            return (StreamSoundBase)this.Components.First();
        }

        private void ValidateWaveFormat(StreamSoundBase streamSound)
        {
            if (this.IsAdtsFile(streamSound))
            {
#if !AAC_SUPPORTED
                var errorLine = new ErrorLine(
                    string.Format(
                        Resources.MessageResourceCommon.Message_UnsupportedWaveFile,
                        "AAC"),
                    streamSound);
                throw new ConversionException(errorLine);
#endif
            }
        }

        private bool ShouldCopyFile(StreamSoundBase streamSound)
        {
            if (this.isBinaryForPC)
            {
                return false;
            }

            if (streamSound.Children.Count == 0)
            {
                return false;
            }

            if (this.IsOpusFile(streamSound))
            {
                return true;
            }

#if AAC_SUPPORTED
            if (this.IsAdtsFile(streamSound))
            {
                return true;
            }
#endif
            return false;
        }

        private bool IsOpusFile(StreamSoundBase streamSound)
        {
            var track = streamSound.Children.FirstOrDefault<Component>(item => item.IsEnabled) as StreamSoundTrackBase;

            return track == null
                ? false
                : string.Compare(Path.GetExtension(track.FilePath), OpusExtension, true) == 0;
        }

        private bool IsAdtsFile(StreamSoundBase streamSound)
        {
            return streamSound.GetAdtsHeader() != null;
        }

        private void CopyFile(string destPath)
        {
            var streamSound = this.TargetComponent as StreamSoundBase;
            Assertion.Operation.True(ShouldCopyFile(streamSound));

            Directory.CreateDirectory(Path.GetDirectoryName(destPath));

            var track = streamSound.Children.FirstOrDefault<Component>(item => item.IsEnabled) as StreamSoundTrackBase;
            File.Copy(track.FilePath, destPath, true);
        }

        private void Write(SoundArchiveContext context, IOutputItem outputItem)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(outputItem);

            var streamSound = this.GetFirstStreamSound();

            WaveStream[] waveStreams = this.ReadTrackFiles(context, streamSound);

            try
            {
                this.ValidateTrackFiles(waveStreams);

                NintendoWareStreamEncoder encoder = this.CreateEncoder(context.Traits.IsLittleEndian);
                CodecOutput[] codecOutputs = encoder.Run(waveStreams);

                // ループなしの場合は、全範囲をループ範囲とします。
                if (!codecOutputs[0].Format.HasLoop)
                {
                    codecOutputs[0].Format = new WaveFormat(codecOutputs[0].Format)
                    {
                        OriginalLoopStartFrame = 0,
                        OriginalLoopEndFrame = 0,
                        LoopStartFrame = 0,
                        LoopEndFrame =
                            ((int)waveStreams[0].Payload.Length / waveStreams[0].Format.FrameLength) - 1,
                    };
                }

                var builder = new StreamSoundFileBuilder(
                    context.Traits.BinaryFileInfo.StreamSoundSignature,
                    context.Traits.BinaryFileInfo.StreamSoundVersion);

                var file = builder.Build(codecOutputs[0], 0, streamSound.GetWaveMarkers(), streamSound.GetWaveRegions());
                ulong bfstmHashValueOffset;

                using (var stream = outputItem.OpenWrite())
                {
                    using (var writer = context.CreateBinaryWriter(stream))
                    {
                        DomElement fileElement = new DomBuilder().Build(file);
                        var domWriter = new DomWriter(writer);
                        domWriter.Run(new DomObjectWriter(), fileElement);

                        //  bfstm 中のハッシュ値へのオフセットを取得する
                        bfstmHashValueOffset = domWriter.Context.ReferenceResolver.GetObjectReference(file.InfoBlock.Body.StreamSoundInformation.HashInfo).Address;
                    }
                }

                // bfstm と bfstp リビジョン不一致検出を可能にするために、bfstm, bfstp に bfstm の CRC32 コードを埋め込みます。
                // ゲーム開発フロー中の事故などにより、bfstm と bfstp のリビジョンが食い違ってしまった場合に、それを検知できるようにします。
                this.hashValue = this.ComputeHash(File.ReadAllBytes(outputItem.Path));

                using (var stream = File.OpenWrite(outputItem.Path))
                {
                    using (var writer = LittleEndianBinaryWriter.Create(stream))
                    {
                        writer.Seek((int)bfstmHashValueOffset, SeekOrigin.Begin);
                        writer.Write(this.hashValue);
                    }
                }

                this.codecOutput = codecOutputs[0];
            }
            finally
            {
                foreach (var waveStream in waveStreams)
                {
                    waveStream.Dispose();
                }
            }

            foreach (string message in this.warningMessages)
            {
                context.Logger.AddLine(new WarningLine(message));
            }
            this.warningMessages.Clear();
        }

        private void WritePrefetch(SoundArchiveContext context, IOutputItem outputItem)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(outputItem);
            Assertion.Argument.NotNull(this.codecOutput);

            var streamSound = this.GetFirstStreamSound();

            var builder = new StreamSoundPrefetchFileBuilder(
                context.Traits.BinaryFileInfo.StreamSoundPrefetchSignature,
                context.Traits.BinaryFileInfo.StreamSoundPrefetchVersion);

            var prefetchDataLength = this.GetPrefetchDataLength(context.Project.SoundArchivePlayerStreamBufferTimes, this.codecOutput);

            // bfstp にも bfstm と同じ CRC32 コードを埋め込みます。
            var file = builder.Build(
                this.codecOutput,
                this.hashValue,
                streamSound.GetWaveRegions(),
                (uint)prefetchDataLength);

            using (var stream = outputItem.OpenWrite())
            {
                using (var writer = context.CreateBinaryWriter(stream))
                {
                    DomElement fileElement = new DomBuilder().Build(file);
                    new DomWriter(writer).Run(new DomObjectWriter(), fileElement);

                }
            }

            foreach (string message in this.warningMessages)
            {
                context.Logger.AddLine(new WarningLine(message));
            }
            this.warningMessages.Clear();
        }

        private uint ComputeHash(byte[] data)
        {
            var crc32 = new CRC32();
            crc32.Initialize();
            return crc32.ComputeHashUInt32(data);
        }

        private int GetPrefetchDataLength(int streamBufferTimes, CodecOutput codecOutput)
        {
            return Math.Min(
                PrefetchStreamBlockCount * streamBufferTimes * codecOutput.StreamData.BlockByte * codecOutput.Format.ChannelCount,
                codecOutput.Data.Length);
        }

        private string GetPreprocessedTag(string projectFilePath)
        {
            string cockpitFilePath = Path.ChangeExtension(projectFilePath, CockpitExtension);

            if (!File.Exists(cockpitFilePath))
            {
                return string.Empty;
            }

            using (var fileStream = File.Open(cockpitFilePath, FileMode.Open, FileAccess.Read, FileShare.Read))
            {
                using (var reader = new StreamReader(fileStream))
                {
                    return reader.ReadLine();
                }
            }
        }

        private WaveStream[] ReadTrackFiles(ComponentContext context, StreamSoundBase streamSound)
        {
            Assertion.Argument.NotNull(streamSound);

            List<WaveStream> data = new List<WaveStream>();

            var streamSoundRegions = streamSound.GetWaveRegions();

            foreach (StreamSoundTrackBase track in
                streamSound.Children.Where<Component>(item => item.IsEnabled))
            {
                var trackFilePath = track.FilePath;
                int maxChannel = MaxChannel;

                if (this.isBinaryForPC)
                {
                    // PC 向け AAC 代替バイナリ向け処理
                    // ⇒ 6ch まで許可する
                    // ⇒ マルチトラックであっても１ファイル前提なので、１トラック目のみ処理する
                    if (streamSound.GetAdtsHeader() != null)
                    {
                        maxChannel = MaxAacChannel;

                        if (data.Count > 0)
                        {
                            break;
                        }
                    }

                    trackFilePath = StreamSoundExtension.GetLinearPcmWaveFilePath(trackFilePath);
                }

                trackFilePath = this.Preprocess(context, trackFilePath);

                using (WaveFormats.WaveFileReader reader =
                    WaveFormats.WaveFileReader.CreateInstance(trackFilePath))
                {
                    reader.LoopEndFrameModified += OnLoopEndFrameModified;

                    WaveFormats.WaveFile waveFile;

                    try
                    {
                        waveFile = reader.Open(trackFilePath);
                    }
                    catch (Exception e)
                    {
                        ErrorLine errorLine = new ErrorLine(e.Message, streamSound);
                        throw new ConversionException(errorLine);
                    }

                    if (waveFile.ChannelCount > maxChannel)
                    {
                        ErrorLine errorLine = new ErrorLine(
                            string.Format(
                               Resources.MessageResource.Message_InvalidWaveChannels,
                               waveFile.ChannelCount,
                               maxChannel,
                               trackFilePath),
                           streamSound);
                        throw new ConversionException(errorLine);
                    }

                    var waveFormat = new WaveFormat()
                    {
                        BitsPerSample = waveFile.SampleBit,
                        ChannelCount = waveFile.ChannelCount,
                        Encoding = Codecs.Encoding.Pcm,
                        IsLittleEndian = waveFile is WaveFormats.WaveFileWav,
                        IsSigned = !(waveFile is WaveFormats.WaveFileWav && waveFile.SampleBit == 8),
                        SamplingRate = waveFile.SampleRate,
                        OriginalLoopStartFrame = this.originalLoopStart ?? (int)waveFile.LoopStartFrame,
                        OriginalLoopEndFrame = this.originalLoopEnd ?? (int)waveFile.LoopEndFrame,
                        LoopStartFrame = this.originalLoopStart ?? (int)waveFile.LoopStartFrame,
                        LoopEndFrame = this.originalLoopEnd ?? (int)waveFile.LoopEndFrame,
                    };

                    if (this.hasOriginalLoop.HasValue)
                    {
                        waveFormat.HasLoop = this.hasOriginalLoop.Value;
                    }
                    else
                    {
                        waveFormat.HasLoop = waveFile.IsLoop;
                    }

                    if (this.originalSampleRate == null)
                    {
                        waveFormat.Regions = streamSoundRegions;
                    }
                    else
                    {
                        waveFormat.Regions = streamSoundRegions
                            .Select(regionInfo => this.AdjustFrames(regionInfo, this.originalSampleRate.Value));
                    }

                    data.Add(new WaveStream(waveFormat, reader.OpenDataStream()));
                }
            }

            return data.ToArray();
        }

        private string Preprocess(ComponentContext context, string filePath)
        {
            var result = filePath;

            using (WaveFormats.WaveFileReader reader = WaveFormats.WaveFileReader.CreateInstance(filePath))
            {
                WaveFormats.WaveFile srcFile;

                try
                {
                    srcFile = reader.Open(filePath);
                }
                catch (Exception e)
                {
                    ErrorLine errorLine = new ErrorLine(e.Message, this.TargetComponent);
                    throw new ConversionException(errorLine);
                }

                var firstStreamSound = this.GetFirstStreamSound();
                var targetSampleRate = firstStreamSound.GetTargetSampleRate();
                var targetChannelCount = firstStreamSound.GetTargetChannelCount();

                // サンプルレートが指定されている場合
                if (targetSampleRate.HasValue == true)
                {
                    // ダウンサンプルになっていない（アップサンプルは禁止）
                    if (srcFile.SampleRate < targetSampleRate)
                    {
                        var message = string.Format(MessageResourceCommon.Message_UpsampleIsNotSupported, targetSampleRate, srcFile.SampleRate);
                        throw new ConversionException(new ErrorLine(message, this.TargetComponent));
                    }

                    // サンプルレートが範囲外
                    if (targetSampleRate < ConversionTraits.MinSampleRate || ConversionTraits.MaxSampleRate < targetSampleRate)
                    {
                        var message = string.Format(
                            MessageResourceCommon.Message_SampleRateMustBeValidRange,
                            ConversionTraits.MinSampleRate,
                            ConversionTraits.MaxSampleRate);
                        throw new ConversionException(new ErrorLine(message, this.TargetComponent));
                    }
                }

                // チャンネル数が指定されている場合（＝現時点ではモノラル化が指定されている場合）
                if (targetChannelCount.HasValue == true)
                {
                    // 現時点では Assert のチェックをするだけ。モノラル化のみ。
                    Debug.Assert(targetChannelCount == 1);
                }

                if (targetSampleRate.HasValue == true || targetChannelCount.HasValue == true)
                {
                    if (string.IsNullOrEmpty(this.PreprocessExePath))
                    {
                        throw new ConversionException(
                            new ErrorLine(MessageResourceCommon.Message_SoxExePathIsEmpty, this.TargetComponent));
                    }

                    if (!File.Exists(this.PreprocessExePath))
                    {
                        var message = string.Format(MessageResourceCommon.Message_SoxExeNotFound, this.PreprocessExePath);
                        throw new ConversionException(new ErrorLine(message, this.TargetComponent));
                    }

                    var sox = new SoxExecutor(this.PreprocessExePath)
                    {
                        InputPath = filePath,
                        OutputPath = this.GetTempFileName("wav"),
                        TargetComponents = new Component[] { this.TargetComponent },
                    };

                    if (targetSampleRate.HasValue == true)
                    {
                        sox.SampleRate = targetSampleRate.Value;
                    }

                    if (targetChannelCount.HasValue == true)
                    {
                        sox.ChannelCount = targetChannelCount.Value;
                    }

                    if (!sox.Run(context))
                    {
                        throw new ConversionException(
                            new ErrorLine(MessageResourceCommon.Message_FailedToPreprocessWaveFile, this.TargetComponent));
                    }

                    int sampleRate = srcFile.SampleRate;
                    if (targetSampleRate.HasValue == true)
                    {
                        sampleRate = targetSampleRate.Value;
                    }

                    this.originalLoopStart = (int)(srcFile.LoopStartFrame * sampleRate / srcFile.SampleRate);
                    this.originalLoopEnd = (int)(srcFile.LoopEndFrame * sampleRate / srcFile.SampleRate);
                    this.hasOriginalLoop = srcFile.IsLoop && this.originalLoopStart < this.originalLoopEnd;
                    this.originalSampleRate = srcFile.SampleRate;

                    result = sox.OutputPath;
                }
            }

            return result;
        }

        private string GetTempFileName(string extension)
        {
            while (true)
            {
                var filePath = Path.ChangeExtension(Path.GetTempFileName(), extension);

                if (!File.Exists(filePath))
                {
                    return filePath;
                }
            }
        }

        private WaveRegionInfo AdjustFrames(WaveRegionInfo regionInfo, int sourceSampleRate)
        {
            var firstStreamSound = this.GetFirstStreamSound();
            var targetSampleRate = firstStreamSound.GetTargetSampleRate();

            if (targetSampleRate.HasValue)
            {
                return new WaveRegionInfo(
                    regionInfo.Name,
                    (uint)((long)regionInfo.StartFrame * targetSampleRate / sourceSampleRate),
                    (uint)((long)regionInfo.EndFrame * targetSampleRate / sourceSampleRate)
                    );
            }
            else
            {
                return regionInfo;
            }
        }

        private void ExtractTrackInfos(SoundArchiveContext context, StreamSoundBase streamSound)
        {
            Assertion.Argument.NotNull(streamSound);

            // PC 用バイナリの場合、トラック情報は不要。
            if (this.isBinaryForPC)
            {
                return;
            }

            // AAC に対応しない (AAC_SUPPORTED が定義されていない) 場合は、ここで例外を発生させています
            this.ValidateWaveFormat(streamSound);

            var tracks = streamSound.Children.
                Where<Component>(item => item.IsEnabled).
                Cast<StreamSoundTrackBase>().
                ToArray();

            streamSound.SetTrackAllocationFlags((ushort)~(ushort.MaxValue << tracks.Length));
            Ensure.Operation.True(tracks.Length > 0);

            // ループ情報参照元の波形ファイルが見つからない場合のために、ループ情報をリセットしておく
            streamSound.SetHasLoop(false);
            streamSound.SetLoopStartFrame(0);
            streamSound.SetLoopEndFrame(0);

            // AAC, Opus ファイルの場合
            if (ShouldCopyFile(streamSound))
            {
                int? sampleRate = null;

                var adtsHeader = streamSound.GetAdtsHeader();

                if (adtsHeader != null && adtsHeader.HasSyncword)
                {
                    if (adtsHeader.Channel > 6)
                    {
                        ErrorLine errorLine = new ErrorLine(
                            string.Format(
                                Resources.MessageResource.Message_InvalidAACChannels,
                                adtsHeader.Channel),
                            streamSound);
                        throw new ConversionException(errorLine);
                    }

                    string aacFilePath = tracks[0].FilePath;
                    int validChannelCount = 0;

                    foreach (var track in tracks)
                    {
                        // ２つ以上の aac ファイルを含めることはできない
                        if (aacFilePath != track.FilePath)
                        {
                            ErrorLine errorLine = new ErrorLine(
                                string.Format(
                                    Resources.MessageResource.Message_AacStreamSoundMustHasAAacFile,
                                    streamSound.Name),
                                streamSound);
                            throw new ConversionException(errorLine);
                        }

                        validChannelCount += track.ChannelCount;
                        track.SetChannelCount((uint)track.ChannelCount);
                    }

                    // aac チャンネル数とトラックへの割り当て数をチェック
                    if (validChannelCount != adtsHeader.Channel)
                    {
                        ErrorLine errorLine = new ErrorLine(
                            string.Format(
                                Resources.MessageResource.Message_InvalidTrackAssignment,
                                streamSound.Name,
                                adtsHeader.Channel),
                            streamSound);
                        throw new ConversionException(errorLine);
                    }

                    streamSound.SetTotalChannelCount((uint)adtsHeader.Channel);
                    streamSound.SetContainerType(ContainerType.Adts);

                    sampleRate = adtsHeader.SamplesPerSec;
                }
                else if (this.IsOpusFile(streamSound))
                {
                    var opusBasicInfo = streamSound.GetAudioOpusBasicInfo();
                    Ensure.Operation.ObjectNotNull(opusBasicInfo);

                    if (tracks.Length > 1)
                    {
                        ErrorLine errorLine = new ErrorLine(
                            Resources.MessageResourceCommon.Message_InvalidOpusTrackCount,
                            streamSound);
                        throw new ConversionException(errorLine);
                    }

                    if (opusBasicInfo.ChannelCount < 1 && 2 < opusBasicInfo.ChannelCount)
                    {
                        ErrorLine errorLine = new ErrorLine(
                            Resources.MessageResourceCommon.Message_InvalidOpusChannelCount,
                            streamSound);
                        throw new ConversionException(errorLine);
                    }

                    tracks[0].SetChannelCount((uint)opusBasicInfo.ChannelCount);

                    streamSound.SetTotalChannelCount((uint)opusBasicInfo.ChannelCount);
                    streamSound.SetContainerType(ContainerType.Opus);

                    sampleRate = opusBasicInfo.SampleRate;
                }

                string waveFilePath = streamSound.GetLinearPcmWaveFilePath();

                if (string.IsNullOrEmpty(waveFilePath))
                {
                    return;
                }

                using (var reader = WaveFormats.WaveFileReader.CreateInstance(waveFilePath))
                {
                    var waveFile = reader.Open(waveFilePath);
                    var frameCorrection = 1.0f;

                    if (sampleRate.HasValue)
                    {
                        frameCorrection = (float)sampleRate.Value / waveFile.SampleRate;
                    }

                    streamSound.SetHasLoop(waveFile.IsLoop);

                    if (waveFile.IsLoop)
                    {
                        streamSound.SetLoopStartFrame(Convert.ToUInt32(waveFile.LoopStartFrame * frameCorrection));
                        streamSound.SetLoopEndFrame(Convert.ToUInt32(waveFile.LoopEndFrame * frameCorrection));
                    }
                }

                return;
            }

            int totalChannelCount = 0;

            foreach (StreamSoundTrackBase track in tracks)
            {
                using (WaveFormats.WaveFileReader reader =
                    WaveFormats.WaveFileReader.CreateInstance(track.FilePath))
                {
                    WaveFormats.WaveFile waveFile;

                    try
                    {
                        waveFile = reader.Open(track.FilePath);
                    }
                    catch (Exception e)
                    {
                        ErrorLine errorLine = new ErrorLine(e.Message, streamSound);
                        throw new ConversionException(errorLine);
                    }

                    int channelCount = streamSound.GetTargetChannelCount() ?? waveFile.ChannelCount;

                    track.SetChannelCount((uint)channelCount);

                    if (totalChannelCount == 0)
                    {
                        this.ExtractTrackWaveMarkers(context, streamSound, waveFile);
                        this.ExtractTrackWaveRegions(context, streamSound, track.FilePath, waveFile);
                    }

                    totalChannelCount += channelCount;
                }
            }

            streamSound.SetTotalChannelCount((uint)totalChannelCount);
            streamSound.SetContainerType(ContainerType.NwStreamBinary);
        }

        private void ExtractTrackWaveMarkers(
            SoundArchiveContext context,
            StreamSoundBase streamSound,
            WaveFormats.WaveFile waveFile)
        {
            var waveMarkers = waveFile.Markers.
                Where(marker => this.IsTargetMarker(marker)).ToArray();

            var streamSoundMarkers = streamSound.GetWaveMarkers();
            streamSoundMarkers.Clear();

            foreach (var marker in waveMarkers)
            {
                var markerName = this.GetMarkerName(marker);
                var newMarker = new WaveMarkerInfo(markerName, marker.Position);

                streamSoundMarkers.Add(newMarker);
            }
        }

        private bool IsTargetMarker(WaveFormats.WaveFile.IMarkerInfo marker)
        {
            return marker.Name.StartsWith(TargetPrefix);
        }

        private string GetMarkerName(WaveFormats.WaveFile.IMarkerInfo marker)
        {
            Assertion.Argument.True(this.IsTargetMarker(marker));

            var markerName = marker.Name;
            return markerName.Substring(TargetPrefix.Length);
        }

        private void ExtractTrackWaveRegions(
            SoundArchiveContext context,
            StreamSoundBase streamSound,
            string filePath,
            WaveFormats.WaveFile waveFile)
        {
            Assertion.Operation.True(!this.isBinaryForPC);

            var waveRegions = waveFile.Regions.
                Where(region => this.IsTargetRegion(region)).ToArray();

            if (waveRegions.Length == 0)
            {
                return;
            }

            if (waveRegions.Length >= MaxRegionCount)
            {
                ErrorLine errorLine = new ErrorLine(
                    string.Format(
                    Resources.MessageResource.Message_WaveRegionsTooMany,
                    filePath
                    ));
                throw new ConversionException(errorLine);
            }

            var streamSoundRegions = streamSound.GetWaveRegions();
            streamSoundRegions.Clear();

            var autoIndexedRegions = new List<WaveRegionInfo>();
            uint minRegionFrames = this.GetMinimumWaveRegionFrames(streamSound);

            foreach (var region in waveRegions)
            {
                this.ValidateRegion(filePath, region, streamSound.Encoding, minRegionFrames);

                int regionIndex = -1;

                try
                {
                    regionIndex = this.GetRegionIndex(region);
                }
                catch
                {
                    ErrorLine errorLine = new ErrorLine(
                        string.Format(
                        Resources.MessageResource.Message_InvalidWaveRegionIndex,
                        region.Name,
                        filePath
                        ));
                    throw new ConversionException(errorLine);
                }

                var regionName = this.ValidateRegionName(context, filePath, this.GetRegionName(region));
                var newRegion = new WaveRegionInfo(regionName, region.StartFrame, region.EndFrame);

                // 自動インデックス指定の場合
                if (regionIndex < 0)
                {
                    // 後で空インデックスをみつけて追加します。
                    autoIndexedRegions.Add(newRegion);
                    continue;
                }

                // 以下、手動インデックス指定の場合
                // インデックスの上限チェック
                if (MaxRegionCount <= regionIndex)
                {
                    ErrorLine errorLine = new ErrorLine(
                        string.Format(
                        Resources.MessageResource.Message_WaveRegionIndexTooLarge,
                        region.Name,
                        regionIndex,
                        MaxRegionCount,
                        filePath
                        ));
                    throw new ConversionException(errorLine);
                }

                // インデックスにあわせて配列（リスト）を拡張
                while (regionIndex >= streamSoundRegions.Count)
                {
                    streamSoundRegions.Add(null);
                }

                // インデックスの重複チェック
                if (streamSoundRegions[regionIndex] != null)
                {
                    ErrorLine errorLine = new ErrorLine(
                        string.Format(
                        Resources.MessageResource.Message_WaveRegionIndexAlreadyExisted,
                        region.Name,
                        regionIndex,
                        filePath
                        ));
                    throw new ConversionException(errorLine);
                }

                streamSoundRegions[regionIndex] = newRegion;
            }

            // 自動インデックス指定リージョンを追加します。
            // 自動インデックス指定リージョンがない場合は、これで終了
            if (autoIndexedRegions.Count == 0)
            {
                return;
            }

            int currentIndex = 0;

            foreach (var region in autoIndexedRegions)
            {
                for (; currentIndex < streamSoundRegions.Count; ++currentIndex)
                {
                    if (streamSoundRegions[currentIndex] == null)
                    {
                        break;
                    }
                }

                if (currentIndex + 1 >= streamSoundRegions.Count)
                {
                    streamSoundRegions.Add(region);
                }
                else
                {
                    Assertion.Operation.ObjectNull(streamSoundRegions[currentIndex]);
                    streamSoundRegions[currentIndex] = region;
                }
            }
        }

        private void ValidateRegion(
            string filePath,
            WaveFormats.WaveFile.IRegionInfo region,
            WaveEncoding waveEncoding,
            uint minRegionFrames)
        {
            Assertion.Operation.True(!this.isBinaryForPC);

            if (region.EndFrame <= region.StartFrame)
            {
                ErrorLine errorLine = new ErrorLine(
                    string.Format(
                    Resources.MessageResource.Message_InvalidWaveRegionFrames,
                    region.Name,
                    filePath
                    ));
                throw new ConversionException(errorLine);
            }

            uint regionFrames = region.EndFrame - region.StartFrame;

            // エンコード後のリージョンサイズが 16KB 以上になるように制限します。
            if (regionFrames < minRegionFrames)
            {
                ErrorLine errorLine = new ErrorLine(
                    string.Format(
                    Resources.MessageResource.Message_WaveRegionFramesTooShort,
                    region.Name,
                    waveEncoding.ToText(),
                    minRegionFrames,
                    filePath
                    ));
                throw new ConversionException(errorLine);
            }
        }

        private bool IsTargetRegion(WaveFormats.WaveFile.IRegionInfo region)
        {
            return region.Name.StartsWith(TargetPrefix);
        }

        private int GetRegionIndex(WaveFormats.WaveFile.IRegionInfo region)
        {
            Assertion.Argument.True(this.IsTargetRegion(region));

            var regionIndexSplitter = region.Name.IndexOf(RegionIndexSplitterChar);
            if (regionIndexSplitter < 0)
            {
                return -1;
            }

            return int.Parse(region.Name.Substring(regionIndexSplitter + 1));
        }

        private string GetRegionName(WaveFormats.WaveFile.IRegionInfo region)
        {
            Assertion.Argument.True(this.IsTargetRegion(region));

            var regionName = region.Name;
            var regionIndexSplitter = regionName.IndexOf(RegionIndexSplitterChar);

            if (regionIndexSplitter >= 0)
            {
                regionName = regionName.Substring(0, regionIndexSplitter);
            }

            return regionName.Substring(TargetPrefix.Length);
        }

        private string ValidateRegionName(SoundArchiveContext context, string filePath, string name)
        {
            if (string.IsNullOrEmpty(name))
            {
                return name;
            }

            string result = name;

            // 文字列長チェック (終端文字を含まない）
            // 長い場合は後ろを切り取る
            if (name.Length > MaxRegionNameLength)
            {
                result = name.Substring(0, MaxRegionNameLength);

                var line = new WarningLine(
                    string.Format(
                        MessageResource.Message_WaveRegionNameTooLong,
                        MaxRegionNameLength,
                        name,
                        result,
                        filePath),
                    this.TargetComponent);

                context.Logger.AddLine(line);
            }

            // 先頭文字が英字か '_' でなければ、 '_' に置換
            // それ以降の文字が英数字か '_' でなければ、 '_' に置換
            var source = result;
            result = Regex.Replace(result, @"^[^a-zA-Z_]", "_");
            result = Regex.Replace(result, @"[^a-zA-Z0-9]", "_");

            if (result != source)
            {
                var line = new WarningLine(
                    string.Format(
                        MessageResource.Message_WaveRegionNameReplaceInvalidChars,
                        name,
                        result,
                        filePath),
                    this.TargetComponent);

                context.Logger.AddLine(line);
            }

            return result;
        }

        private uint GetMinimumWaveRegionFrames(StreamSoundBase streamSound)
        {
            // エンコード後のリージョンサイズが 16KB 以上になるフレーム数を返します。
            switch (streamSound.Encoding)
            {
                case WaveEncoding.Pcm16:
                    return 8192;

                case WaveEncoding.Pcm8:
                    return 8192 * 2;

                case WaveEncoding.Adpcm:
                    return 14336 * 2;
            }

            throw new NotImplementedException();
        }

        private void ValidateTrackFiles(WaveStream[] waveStreams)
        {
            Assertion.Argument.NotNull(waveStreams);

            var streamSound = this.GetFirstStreamSound();

            if (waveStreams.Length == 0)
            {
                throw new ConversionException(
                    new ErrorLine(
                        Resources.MessageResource.Message_StreamSoundTrackNotFound,
                        streamSound));
            }

            int trackMax = ComponentConfiguration.Instance.StreamSoundTrackNumberMaximum;
            if (trackMax < waveStreams.Length)
            {
                string errorMessage0 =
                    string.Format(
                        Resources.MessageResource.Message_StreamSoundTrackExceededMaximumNumber,
                        trackMax + 1);
                string errorMessage1 =
                    string.Format(
                        Resources.MessageResource.Message_StreamSoundTrackMaximumNumber,
                        trackMax);

                ErrorLine[] errorLines =
                    new ErrorLine[2]
                    {
                        new ErrorLine(errorMessage0, streamSound),
                        new ErrorLine(errorMessage1, streamSound)
                    };

                throw new ConversionException(errorLines);
            }

            int samplingRate = waveStreams[0].Format.SamplingRate;

            foreach (WaveStream waveStream in waveStreams)
            {
                if (samplingRate != waveStream.Format.SamplingRate)
                {
                    throw new ConversionException(
                        new ErrorLine(
                            Resources.MessageResource.Message_StreamSoundTracksMustHaveTheSameSamplingRate,
                            streamSound));
                }
            }
        }

        private NintendoWareStreamEncoder CreateEncoder(bool isLittleEndian)
        {
            var encoding = this.isBinaryForPC ?
                WaveEncoding.Pcm16 : this.GetFirstStreamSound().Encoding;

            switch (encoding)
            {
                case WaveEncoding.Adpcm:
                    return new NintendoWareDspAdpcmStreamEncoder();

                case WaveEncoding.Pcm16:
                    return new NintendoWareLinearPcmStreamEncoder()
                    {
                        BitsPerSample = 16,
                        IsLittleEndian = isLittleEndian,
                    };
            }

            throw new ConversionException(
                new ErrorLine(
                    string.Format(Resources.MessageResourceCommon.Message_InvalidWaveEncoding, encoding.ToString()),
                    this.TargetComponent));
        }

        private void OnLoopEndFrameModified(object send, WaveFormats.LoopEndFrameModifiedEventArgs e)
        {
            string format = string.Format(
                MessageResource.Message_LoopEndFrameModified,
                e.LoopEndFrame,
                e.FrameCount,
                e.FilePath);

            this.warningMessages.Add(format);
        }

        private void SetBinaryPaths(SoundArchiveContext context)
        {
            if (!context.Settings.IsConvertParts)
            {
                return;
            }

            if (this.isBinaryForPC)
            {
                // bxstm ファイルパスを設定します。
                foreach (var component in this.Components.Cast<StreamSoundBase>())
                {
                    component.SetPCBinaryFilePathForPartsConvert(this.OutputTargetItem.Path);
                }
            }
            else
            {
                // bxstm / aac ファイルパスを設定します。
                foreach (var component in this.Components.Cast<StreamSoundBase>())
                {
                    component.SetBinaryFilePathForPartsConvert(this.OutputTargetItem.Path);

                    if (this.PrefetchOutputItem != null)
                    {
                        component.SetPrefetchBinaryFilePathForPartsConvert(this.PrefetchOutputItem.Path);
                    }
                }
            }
        }
    }
}
