﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Resources;
    using NintendoWare.ToolDevelopmentKit;
    using FileFormats.NintendoSdk;
    using System.Text.RegularExpressions;

    /// <summary>
    /// サウンドアーカイブバイナリの作成に必要なコンテキストを構築します。
    /// </summary>
    internal class SoundArchiveBuilder
    {
        private const int MAX_PATH = 256;

        //-----------------------------------------------------------------

        private FileManager fileManager;

        private SoundProjectService projectService;
        private SoundSet addonSoundSet;
        private Dictionary<string, BankService> bankServiceDictionary = new Dictionary<string, BankService>();

        //-----------------------------------------------------------------

        public SoundArchiveBuilder(FileManager fileManager)
        {
            Ensure.Argument.NotNull(fileManager);
            this.fileManager = fileManager;
        }

        //-----------------------------------------------------------------

        public bool IsInitialized
        {
            get { return this.projectService != null; }
        }

        public FileManager FileManager
        {
            get { return this.fileManager; }
        }

        public SoundProjectService SoundProjectService
        {
            get { return this.projectService; }
        }

        public IDictionary<string, BankService> BankServiceDictionary
        {
            get { return this.bankServiceDictionary; }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// サウンドアーカイブの作成に必要なコンテキストを構築するために初期化します。
        /// </summary>
        /// <param name="projectService">対象のプロジェクトサービス。</param>
        /// <param name="addonSoundSet">出力対象の追加サウンドアーカイブ。</param>
        /// <param name="bankServices">対象のバンクサービスの列挙子。</param>
        public void Initialize(
            SoundProjectService projectService,
            SoundSet addonSoundSet,
            IEnumerable<BankService> bankServices)
        {
            Ensure.Argument.NotNull(projectService);
            Ensure.Argument.NotNull(projectService.ProjectDocument);
            Ensure.Argument.NotNull(bankServices);

            this.projectService = projectService;
            this.addonSoundSet = addonSoundSet;

            foreach (BankService bankService in bankServices)
            {
                bankServiceDictionary.Add(
                    bankService.BankDocument.Resource.Key.GetFullPath(),
                    bankService);
            }
        }

        /// <summary>
        /// サウンドアーカイブの作成に必要なコンテキストを構築します。
        /// </summary>
        /// <param name="context">対象となるコンテキスト。</param>
        /// <returns>成否。</returns>
        public bool Build(SoundArchiveContext context)
        {
            Ensure.Argument.NotNull(context);

            try
            {
                if (!this.IsInitialized)
                {
                    throw new Exception("internal error : not initialized.");
                }

                if (this.fileManager == null)
                {
                    this.fileManager = new FileManager(
                        context.Traits.IntermediateOutputTraits,
                        context.Traits.BinaryOutputTraits);

                    string dependFilePath = this.projectService.ProjectDocument.GetDependFilePath();

                    this.fileManager.Initialize(
                        new FileManager.InitializeArgs()
                        {
                            DependFilePath = dependFilePath,
                            OutputDirectoryPath = PathEx.MakeRelative(
                                this.projectService.ProjectOutputPath,
                                Path.GetDirectoryName(dependFilePath)
                                ),
                            CacheDiretoryPath = "./",
                            IsAllCachesClean = true,
                            IsKeepAllCaches = false,
                            IsWaveSound2Enabled = context.Traits.IsWaveSound2BinaryEnabled,
                        });
                }

                context.Settings.IsConvertParts = false;

                this.ValidateNames(context);

                // 名前に問題があった場合は、以降の AssignItems() 等で
                // 失敗する可能性があるため、ここで中断します。
                if (context.IsFailed)
                {
                    this.Cleanup(context);
                    return false;
                }

                this.AssignItems(context);
                this.ResolveGroupReferences(context);
                this.ValidateFilePaths(context);

                this.GenerateItems(context);
                this.ValidateItems(context);

                this.ResolveNames(context);

                if (context.IsFailed)
                {
                    this.Cleanup(context);
                    return false;
                }

                this.CreateOutputItems(context);
                this.CreateComponentProcessors(context);

                if (context.IsFailed)
                {
                    this.Cleanup(context);
                    return false;
                }

                this.LinkComponentProcessors(context);

                this.OptimizeFiles(context);
                this.LinkDependedFiles(context);

                this.CollectGarbageOutputItems(context);

                this.RenumberItemIndexes(context);
                this.CreateSoundArchiveProcessor(context);
            }
            catch (Exception exception)
            {
                try
                {
                    if (exception.Message != null && exception.Message.Length > 0)
                    {
                        context.Logger.AddLine(new InternalErrorLine(exception.ToString()));
                    }

                    this.Cleanup(context);
                }
                catch
                {
                }

                throw;
            }

            return !context.IsFailed;
        }

        /// <summary>
        /// サウンドアーカイブの作成に必要なコンテキストを構築します。
        /// サウンドアーカイブに含めるアイテムを指定し、それらに限定したコンテキストが作成されます。
        /// </summary>
        /// <param name="context">対象となるコンテキスト。</param>
        /// <param name="soundSetItems">対象となるアイテムの列挙子。</param>
        /// <returns>成否。</returns>
        public bool BuildParts(
            SoundArchiveContext context,
            IEnumerable<SoundSetItem> soundSetItems)
        {
            Ensure.Argument.NotNull(context);
            Ensure.Argument.NotNull(soundSetItems);
            Ensure.Operation.ObjectNotNull(this.fileManager);

            try
            {
                if (!this.IsInitialized)
                {
                    throw new Exception("internal error : not initialized.");
                }

                context.Settings.IsConvertParts = true;

                this.ValidateNames(context);

                // 名前に問題があった場合は、以降の AssignItems() 等で
                // 失敗する可能性があるため、ここで中断します。
                if (context.IsFailed)
                {
                    this.Cleanup(context);
                    return false;
                }

                this.AssignPartsItems(context, soundSetItems);
                this.ValidateFilePaths(context);

                this.GeneratePartsItems(context);
                this.ValidateItems(context);

                this.ResolveNames(context);

                if (context.IsFailed)
                {
                    this.Cleanup(context);
                    return false;
                }

                this.CreateOutputItems(context);
                this.CreateComponentProcessors(context);

                if (context.IsFailed)
                {
                    this.Cleanup(context);
                    return false;
                }

                this.LinkComponentProcessors(context);

                this.OptimizeFiles(context);
                this.LinkDependedFiles(context);

                this.CollectGarbageOutputItems(context);

                this.RenumberItemIndexes(context);
            }
            catch (Exception exception)
            {
                try
                {
                    if (exception.Message != null && exception.Message.Length > 0)
                    {
                        context.Logger.AddLine(new InternalErrorLine(exception.ToString()));
                    }

                    this.Cleanup(context);
                }
                catch
                {
                }

                throw;
            }

            return !context.IsFailed;
        }

        /// <summary>
        /// サウンドアーカイブ作成のための追加した情報をクリーンアップします。
        /// </summary>
        /// <param name="context">コンテキストを指定します。</param>
        /// <remarks>
        /// 各アイテムに追加されたコンバート処理専用のパラメータが削除されます。
        /// </remarks>
        public void Cleanup(SoundArchiveContext context)
        {
            Ensure.Argument.NotNull(context);

            foreach (SoundSetItem item in context.AllItems)
            {
                this.Cleanup(item);
            }
        }

        /// <summary>
        /// アイテム名を検証します。
        /// </summary>
        /// <param name="context">対象のコンテキスト。</param>
        private void ValidateNames(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);

            NameValidatorSelector runner = new NameValidatorSelector();

            foreach (SoundSetItem soundSetItem in this.EnumerateTargetSoundSetItems(context.Settings.IsConvertParts))
            {
                if (!soundSetItem.IsConvertTarget())
                {
                    continue;
                }
                runner.Run(context, soundSetItem);
            }

            ValidateNameDuplication(context);
        }

        /// <summary>
        /// アイテム名の重複を検証します。
        /// </summary>
        /// <param name="context">対象のコンテキスト。</param>
        private void ValidateNameDuplication(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Operation.True(this.projectService != null);
            Assertion.Operation.True(this.projectService.ComponentDictionary != null);

            foreach (string name in this.projectService.ComponentDictionary.DuplicatedNames)
            {
                if (name.Length == 0) { continue; }

                Component[] components = this.projectService.ComponentDictionary[name];
                Component[] targets = this.GetDuplicatedComponents(components);

                if (targets.Length > 0)
                {
                    context.Logger.AddLine(
                        new ErrorLine(
                            string.Format(Resources.MessageResource.Message_NameAlreadyExists, name,
                            targets[0].Name), targets
                            ));
                }
            }
        }

        private Component[] GetDuplicatedComponents(Component[] components)
        {
            Assertion.Argument.NotNull(components);

            if (components.Length < 2) { return new Component[0]; }

            List<Component> targetComponents = new List<Component>();

            foreach (Component component in components)
            {
                if (!component.IsConvertTarget())
                {
                    continue;
                }

                if (component is SoundSetItemPack)
                {
                    continue;
                }

                if (component is SoundSet)
                {
                    continue;
                }

                if (component is FolderComponent)
                {
                    continue;
                }

                if (component is SoundProject)
                {
                    continue;
                }

                targetComponents.Add(component);
            }

            if (targetComponents.Count < 2)
            {
                return new Component[0];
            }

            return targetComponents.ToArray();
        }

        /// <summary>
        /// サウンドアーカイブに含めるアイテムをコンテキストに追加します。
        /// アイテムID、文字列IDもこのタイミングで割り当てます。
        /// </summary>
        /// <param name="context">対象のコンテキスト。</param>
        private void AssignItems(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);

            // サウンドプロジェクト
            context.Project = this.projectService.Project;
            context.AddonSoundSet = this.addonSoundSet;

            // サウンドアーカイブに指定するフィルタ情報を探す
            var excludeFilter = context.Settings?.ExcludeItemFilterSettings
               ?.Filters?.FirstOrDefault(filter => string.IsNullOrEmpty(filter.Target));
            var includeFilter = context.Settings?.IncludeItemFilterSettings
               ?.Filters?.FirstOrDefault(filter => string.IsNullOrEmpty(filter.Target));

            uint stringID = 0;
            uint soundID = 0;

            // ストリームサウンド
            foreach (var sound in this.EnumerateTargetStreamSounds(context.Settings.IsConvertParts)
                                        .Do(sound => this.ExcludeItem(ref sound, excludeFilter))
                                        .Do(sound => this.IncludeItem(ref sound, includeFilter))
                                        .Where(sound => !sound.GetIsExcluded()))
            {
                if (context.StreamSounds.ContainsKey(sound.Name))
                {
                    continue;
                }

                context.StreamSounds.Add(sound);
                sound.ID = new BinaryItemID(ItemType.Sound, soundID);
                soundID++;

                context.StringTableItems.Add(sound);
                sound.SetStringID(stringID);
                stringID++;
            }

            // ウェーブサウンド
            foreach (var waveSoundSet in this.EnumerateTargetWaveSoundSets(context.Settings.IsConvertParts)
                                                .Do(soundSet => this.ExcludeItem(ref soundSet, excludeFilter))
                                                .Do(soundSet => this.IncludeItem(ref soundSet, includeFilter))
                                                .Where(soundSet => !soundSet.GetIsExcluded()))
            {
                uint waveSoundIndex = 0;

                foreach (WaveSoundBase sound in waveSoundSet.Children)
                {
                    if (!sound.IsEnabled)
                    {
                        continue;
                    }

                    if (sound.Parent is WaveSoundSetBase && !sound.Parent.IsEnabled)
                    {
                        continue;
                    }

                    if (context.WaveSounds.ContainsKey(sound.Name))
                    {
                        continue;
                    }

                    // フィルタ処理
                    if (excludeFilter != null && IsItemMatched(sound, excludeFilter))
                    {
                        sound.SetIsExcluded(true);
                        continue;
                    }

                    if (includeFilter != null && !IsItemMatched(sound, includeFilter))
                    {
                        sound.SetIsExcluded(true);
                        continue;
                    }

                    context.WaveSounds.Add(sound as WaveSoundBase);
                    sound.ID = new BinaryItemID(ItemType.Sound, soundID);
                    soundID++;

                    sound.SetIndex(waveSoundIndex);
                    waveSoundIndex++;

                    context.StringTableItems.Add(sound);
                    sound.SetStringID(stringID);
                    stringID++;
                }
            }

            // シーケンスサウンド
            foreach (var sound in this.EnumerateTargetSequenceSounds(context.Settings.IsConvertParts)
                                        .Do(sound => this.ExcludeItem(ref sound, excludeFilter))
                                        .Do(sound => this.IncludeItem(ref sound, includeFilter))
                                        .Where(sound => !sound.GetIsExcluded()))
            {
                if (sound.Parent is SequenceSoundSetBase && !sound.Parent.IsConvertTarget())
                {
                    continue;
                }

                if (context.SequenceSounds.ContainsKey(sound.Name))
                {
                    continue;
                }

                context.SequenceSounds.Add(sound);
                sound.ID = new BinaryItemID(ItemType.Sound, soundID);
                soundID++;

                context.StringTableItems.Add(sound);
                sound.SetStringID(stringID);
                stringID++;
            }

            uint soundGroupID = 0;

            // ウェーブサウンドセット
            foreach (var waveSoundSet in this.EnumerateTargetWaveSoundSets(context.Settings.IsConvertParts)
                                                .Do(soundSet => this.ExcludeItem(ref soundSet, excludeFilter))
                                                .Do(soundSet => this.IncludeItem(ref soundSet, includeFilter))
                                                .Where(soundSet => !soundSet.GetIsExcluded()))
            {
                if (waveSoundSet.TargetWaveArchive != null)
                {
                    waveSoundSet.GetWaveArchiveDictionary().Add(
                        string.Empty, waveSoundSet.TargetWaveArchive);
                }

                context.WaveSoundSets.Add(waveSoundSet);
                waveSoundSet.ID = new BinaryItemID(ItemType.SoundGroup, soundGroupID);
                soundGroupID++;

                context.StringTableItems.Add(waveSoundSet);
                waveSoundSet.SetStringID(stringID);
                stringID++;
            }

            // シーケンスサウンドセット
            foreach (var sequenceSoundSet in this.EnumerateTargetSequenceSoundSets(context.Settings.IsConvertParts)
                                        .Do(sound => this.ExcludeItem(ref sound, excludeFilter))
                                        .Do(sound => this.IncludeItem(ref sound, includeFilter))
                                        .Where(sound => !sound.GetIsExcluded()))
            {
                if (context.SequenceSoundSets.ContainsKey(sequenceSoundSet.Name))
                {
                    continue;
                }

                context.SequenceSoundSets.Add(sequenceSoundSet);
                sequenceSoundSet.ID = new BinaryItemID(ItemType.SoundGroup, soundGroupID);
                soundGroupID++;

                context.StringTableItems.Add(sequenceSoundSet);
                sequenceSoundSet.SetStringID(stringID);
                stringID++;
            }

            // バンク
            uint bankID = 0;
            foreach (var soundSetBank in this.EnumerateTargetSoundSetBanks(context)
                                        .Do(bank => this.ExcludeItem(ref bank, excludeFilter))
                                        .Do(bank => this.IncludeItem(ref bank, includeFilter))
                                        .Where(bank => !bank.GetIsExcluded()))
            {
                if (context.SoundSetBanks.ContainsKey(soundSetBank.Name))
                {
                    continue;
                }

                context.SoundSetBanks.Add(soundSetBank);
                soundSetBank.ID = new BinaryItemID(ItemType.Bank, bankID);
                bankID++;

                context.StringTableItems.Add(soundSetBank);
                soundSetBank.SetStringID(stringID);
                stringID++;

                // メインサウンドアーカイブの場合                    ：バンク情報＋バンク関連ファイルをバイナリに含める
                // 追加サウンドアーカイブで自身のサウンドセットの場合：バンク情報＋バンク関連ファイルをバイナリに含める
                // それ以外の場合                                    ：バンク情報のみバイナリに含める
                if (context.AddonSoundSet == null || soundSetBank.SoundSet == context.AddonSoundSet)
                {
                    var bank = this.bankServiceDictionary[soundSetBank.FilePath].Bank;
                    context.BankDictionary.Add(soundSetBank, bank);

                    // 逆参照チェック時や、 BankBuilder による生成時にバンク情報を利用するため、
                    // このタイミングでインストに適用するフィルタを検索し、適用します。
                    var excludeInstrumentFilter = context.Settings?.ExcludeItemFilterSettings?.Filters
                             ?.FirstOrDefault(filter => filter.Target == soundSetBank.FilePath);
                    var includeInstrumentFilter = context.Settings?.IncludeItemFilterSettings?.Filters
                             ?.FirstOrDefault(filter => filter.Target == soundSetBank.FilePath);

                    foreach (VelocityRegion velocityRegion in bank.GetVelocityRegions())
                    {
                        var instrument = velocityRegion.Instrument;
                        // インストのフィルタ処理
                        if (excludeInstrumentFilter != null && IsItemMatched(instrument, excludeInstrumentFilter))
                        {
                            instrument.SetIsExcluded(true);
                            continue;
                        }
                        if (includeInstrumentFilter != null && !IsItemMatched(velocityRegion.Instrument, includeInstrumentFilter))
                        {
                            instrument.SetIsExcluded(true);
                            continue;
                        }
                    }

                    if (!context.BankFilePathDictionary.ContainsKey(bank))
                    {
                        context.BankFilePathDictionary.Add(bank, soundSetBank.FilePath);
                    }

                    if (soundSetBank.TargetWaveArchive != null)
                    {
                        soundSetBank.GetWaveArchiveDictionary().Add(string.Empty, soundSetBank.TargetWaveArchive);
                    }
                }
            }

            // 波形アーカイブ
            uint waveArchiveID = 0;
            foreach (var waveArchive in this.EnumerateTargetWaveArchives(context.Settings.IsConvertParts)
                                            .Do(archive => this.ExcludeItem(ref archive, excludeFilter))
                                            .Do(archive => this.IncludeItem(ref archive, includeFilter))
                                            .Where(archive => !archive.GetIsExcluded()))
            {
                if (context.WaveArchives.ContainsKey(waveArchive.Name))
                {
                    continue;
                }

                context.WaveArchives.Add(waveArchive);

                waveArchive.ID = new BinaryItemID(ItemType.WaveArchive, waveArchiveID);
                waveArchiveID++;

                if (!waveArchive.Name.EndsWith(SoundArchiveContext.AutoGeneratedNamePostfix))
                {
                    context.StringTableItems.Add(waveArchive);

                    waveArchive.SetStringID(stringID);
                    stringID++;
                }
            }

            // グループ
            uint groupID = 0;
            foreach (var group in this.EnumerateTargetGroups()
                                        .Do(group => this.ExcludeItem(ref group, excludeFilter))
                                        .Do(group => this.IncludeItem(ref group, includeFilter))
                                        .Where(group => !group.GetIsExcluded()))
            {
                if (context.Groups.ContainsKey(group.Name))
                {
                    continue;
                }

                if (group.OutputType == GroupOutputType.None) { continue; }

                context.Groups.Add(group);
                context.StringTableItems.Add(group);

                group.ID = new BinaryItemID(ItemType.Group, groupID);
                groupID++;

                group.SetStringID(stringID);
                stringID++;
            }

            // プレイヤー
            uint playerID = 0;
            foreach (var player in this.EnumerateTargetPlayers(context)
                                            .Do(player => this.ExcludeItem(ref player, excludeFilter))
                                            .Do(player => this.IncludeItem(ref player, includeFilter))
                                            .Where(player => !player.GetIsExcluded()))
            {
                if (context.Players.ContainsKey(player.Name))
                {
                    continue;
                }

                context.Players.Add(player);
                context.StringTableItems.Add(player);

                player.ID = new BinaryItemID(ItemType.Player, playerID);
                playerID++;

                player.SetStringID(stringID);
                stringID++;
            }
        }

        /// <summary>
        /// サウンドアーカイブに含めるアイテムを指定してコンテキストに追加します。
        /// 列挙子に指定したアイテムのみが処理対象になります。
        /// アイテムID、文字列IDもこのタイミングで割り当てます。
        /// </summary>
        /// <param name="context">対象のコンテキスト。</param>
        /// <param name="soundSetItems">対象となるアイテムの列挙子。</param>
        private void AssignPartsItems(
            SoundArchiveContext context,
            IEnumerable<SoundSetItem> soundSetItems)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(soundSetItems);

            // サウンドプロジェクト
            context.Project = this.projectService.Project;

            // ストリームサウンド
            foreach (StreamSoundBase sound in soundSetItems.OfType<StreamSoundBase>())
            {
                sound.SetBinaryFilePathForPartsConvert(string.Empty);
                sound.SetPCBinaryFilePathForPartsConvert(string.Empty);

                if (!sound.IsEnabled)
                {
                    continue;
                }

                if (context.StreamSounds.ContainsKey(sound.Name))
                {
                    continue;
                }

                context.StreamSounds.Add(sound);
            }

            // ウェーブサウンド
            foreach (WaveSoundBase sound in soundSetItems.OfType<WaveSoundBase>())
            {
                sound.SetBinaryFilePathForPartsConvert(string.Empty);
                sound.SetWaveArchiveBinaryFilePathForPartsConvert(string.Empty);

                if (!sound.IsEnabled)
                {
                    continue;
                }

                if (sound.Parent is WaveSoundSetBase && !sound.Parent.IsEnabled)
                {
                    continue;
                }

                if (context.WaveSounds.ContainsKey(sound.Name))
                {
                    continue;
                }

                context.WaveSounds.Add(sound);

                // WSDSET は WSD 毎に自動生成するので、インデックスはすべて 0 にします。
                sound.SetIndex(0);
            }

            // シーケンスサウンド
            foreach (SequenceSoundBase sound in soundSetItems.OfType<SequenceSoundBase>())
            {
                sound.SetBinaryFilePathForPartsConvert(string.Empty);

                if (!sound.IsEnabled)
                {
                    continue;
                }

                if (sound.Parent is SequenceSoundSetBase && !sound.Parent.IsEnabled)
                {
                    continue;
                }

                if (context.SequenceSounds.ContainsKey(sound.Name))
                {
                    continue;
                }

                context.SequenceSounds.Add(sound);
            }

            // バンク
            foreach (SoundSetBankBase soundSetBank in soundSetItems.OfType<SoundSetBankBase>())
            {
                soundSetBank.SetBinaryFilePathForPartsConvert(string.Empty);
                soundSetBank.SetWaveArchiveBinaryFilePathForPartsConvert(string.Empty);

                if (!soundSetBank.IsEnabled)
                {
                    continue;
                }

                if (context.SoundSetBanks.ContainsKey(soundSetBank.Name))
                {
                    continue;
                }

                context.SoundSetBanks.Add(soundSetBank);

                var bank = this.bankServiceDictionary[soundSetBank.FilePath].Bank;
                context.BankDictionary.Add(soundSetBank, bank);

                if (!context.BankFilePathDictionary.ContainsKey(bank))
                {
                    context.BankFilePathDictionary.Add(bank, soundSetBank.FilePath);
                }
            }
        }

        /// <summary>
        /// グループの名前参照を解決します。
        /// </summary>
        /// <param name="context">対象のコンテキスト。</param>
        private void ResolveGroupReferences(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);

            NameResolverSelector runner = new NameResolverSelector(this.projectService.ComponentDictionary);

            foreach (SoundSetItem soundSetItem in context.Groups)
            {
                runner.Run(context, soundSetItem);
            }
        }

        /// <summary>
        /// アイテムの名前参照を解決します。
        /// </summary>
        /// <param name="context">対象のコンテキスト。</param>
        private void ResolveNames(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);

            NameResolverSelector runner = new NameResolverSelector(this.projectService.ComponentDictionary);

            foreach (SoundSetItem soundSetItem in context.AllItems.Where(item => !(item is GroupBase)))
            {
                runner.Run(context, soundSetItem);
            }
        }

        /// <summary>
        /// ファイルパスの有効性を検証します。
        /// </summary>
        /// <param name="context">対象のコンテキスト。</param>
        private void ValidateFilePaths(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);

            // .depend ファイルパスを確認します。
            string dependFilePath =
                this.projectService.ProjectDocument.GetDependFilePath().GetFullPath();

            if (dependFilePath.Length >= MAX_PATH - 1)
            {
                context.Logger.AddLine(
                    new Logs.ErrorLine(
                        string.Format(MessageResource.Message_FilePathTooLong, dependFilePath)));
                return;
            }

            // cache フォルダパスを確認します。
            string cacheDirectoryPath = Path.GetDirectoryName(dependFilePath);

            if (cacheDirectoryPath.Length >= MAX_PATH - 1)
            {
                context.Logger.AddLine(
                    new Logs.ErrorLine(
                        string.Format(MessageResource.Message_FilePathTooLong, cacheDirectoryPath)));
                return;
            }

            ParameterValidatorSelector selector = new ParameterValidatorSelector();

            foreach (SoundSetItem soundSetItem in context.AllItems)
            {
                if (!soundSetItem.IsConvertTarget())
                {
                    continue;
                }

                if (this.IsWaveSound2BinaryRequired(soundSetItem))
                {
                    foreach (var waveSoundClip in ((WaveSound)soundSetItem).GetClipConvertModels())
                    {
                        selector.Run(context, waveSoundClip);
                    }
                }
                else
                {
                    selector.Run(context, soundSetItem);
                }
            }
        }

        /// <summary>
        /// アイテムを自動生成し、コンテキストに追加します。
        /// </summary>
        /// <param name="context">対象のコンテキスト。</param>
        private void GenerateItems(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);

            ItemGeneratorSelector generator = new ItemGeneratorSelector();

            // グループ
            // 以降のアイテム自動生成処理でグループの共有波形アーカイブを参照するため、グループを先に処理します。
            foreach (GroupBase group in context.Groups)
            {
                generator.Run(context, group);
            }

            // ウェーブサウンド
            // ウェーブサウンドクリップ（コンバート用）を生成します。
            // ConvertParts() では実行されない = WaveSoundBase が生成されないので、ToArray() は不要です。
            foreach (WaveSoundBase waveSound in context.WaveSounds)
            {
                generator.Run(context, waveSound);
            }

            // ウェーブサウンドセット
            foreach (WaveSoundSetBase waveSoundSet in context.WaveSoundSets)
            {
                generator.Run(context, waveSoundSet);
            }

            // バンク
            foreach (SoundSetBankBase soundSetBank in context.SoundSetBanks)
            {
                generator.Run(context, soundSetBank);
            }
        }

        /// <summary>
        /// 部分コンバート用にアイテムを自動生成し、コンテキストに追加します。
        /// </summary>
        /// <param name="context">対象のコンテキスト。</param>
        private void GeneratePartsItems(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);

            ItemGeneratorSelector generator = new ItemGeneratorSelector();

            // ウェーブサウンド
            // WaveSoundSet から 各 WaveSound を分離して、個別の bfwsd, bfwar が出力できるようにします。
            // このタイミングで、新しい WaveSoundSet, WaveSound インスタンスが生成され、コンテキストの内容が差し替えられるため、
            // ToArray() が必要です。
            foreach (WaveSoundBase waveSound in context.WaveSounds.ToArray())
            {
                generator.Run(context, waveSound);
            }

            // バンク
            foreach (SoundSetBankBase soundSetBank in context.SoundSetBanks)
            {
                generator.Run(context, soundSetBank);
            }
        }

        /// <summary>
        /// 出力アイテムを生成し、コンテキスト, FileManager に追加します。
        /// </summary>
        /// <param name="context">対象のコンテキスト。</param>
        private void CreateOutputItems(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);

            OutputItemFactorySelector runner = new OutputItemFactorySelector(this.fileManager);

            foreach (SoundSetItem soundSetItem in context.AllItems)
            {
                if (this.IsWaveSound2BinaryRequired(soundSetItem))
                {
                    foreach (var waveSoundClip in ((WaveSound)soundSetItem).GetClipConvertModels())
                    {
                        runner.Run(context, waveSoundClip);
                    }
                }
                else
                {
                    runner.Run(context, soundSetItem);
                }
            }

            // なるべくアイテム名の重複が起きないように未使用アイテムを回収してから、出力アイテムを登録します。
            // bfstm, bfgrp（サウンドアーカイブ外に出力されるファイル）が対象です。
            //
            // コンテキストの RegisteringOutputs は、登録を遅延するための暫定措置として定義されています。
            // 本来、***OutputFactory で出力を登録したいのですが、その場合、不要なキャッシュがクリアされる前に処理されるために、
            // 出力名が重複して不要なポストフィックス（番号）が付加されてしまう問題が発生します。
            // 出力の登録タイミングをキャッシュクリア後に遅延させることで、この問題に対処しています。
            this.fileManager.CollectGarbage();

            foreach (var registeringOutput in context.RegisteringOutputs)
            {
                var outputItem = this.fileManager.RegisterOutputItem(
                    registeringOutput.TargetOutput,
                    string.Empty,
                    registeringOutput.OutputFilePath);

                if (registeringOutput.ExternalFileDirectoryPath != null)
                {
                    context.AddExternalFile(
                        registeringOutput.TargetComponent,
                        registeringOutput.TargetOutput,
                        Path.Combine(registeringOutput.ExternalFileDirectoryPath, Path.GetFileName(outputItem.Path)).Replace('\\', '/')
                        );
                }
            }
        }

        /// <summary>
        /// 不要な出力アイテムへの依存が残っていたら、FileManager から削除します。
        /// </summary>
        /// <param name="context">対象のコンテキスト。</param>
        private void CollectGarbageOutputItems(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);

            var runner = new GarbageOutputCollectorSelector(this.fileManager);

            foreach (SoundSetItem soundSetItem in context.AllItems)
            {
                runner.Run(context, soundSetItem);
            }
        }

        /// <summary>
        /// アイテムの設定を検証し、警告、エラー出力します。
        /// </summary>
        /// <param name="context">対象のコンテキスト。</param>
        private void ValidateItems(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);

            this.ValidateGroupItemRegisterTypes(context);
            this.WarnUnreferencedItems(context);
        }

        /// <summary>
        /// 未参照アイテムを警告します。
        /// </summary>
        /// <param name="context">対象のコンテキスト。</param>
        private void WarnUnreferencedItems(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);

            if (!context.Settings.DoWarnUnReferencedItems)
            {
                return;
            }

            foreach (SoundSetBankBase soundSetBank in context.SoundSetBanks)
            {
                this.WarnUnreferencedItem(context, soundSetBank);
            }

            foreach (PlayerBase player in context.Players)
            {
                this.WarnUnreferencedItem(context, player);
            }
        }

        private void WarnUnreferencedItem(SoundArchiveContext context, SoundSetItem item)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(item);

            if (item.GetIsUsed())
            {
                return;
            }

            context.Logger.AddLine(
                new WarningLine(
                    string.Format(Resources.MessageResource.Message_UnreferencedItemFound, item.Name),
                    item)
                    );
        }

        /// <summary>
        ///グループの関連アイテム登録を検証します。
        /// </summary>
        /// <param name="context">対象のコンテキスト。</param>
        private void ValidateGroupItemRegisterTypes(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);

            foreach (GroupBase group in context.Groups)
            {
                foreach (GroupItemBase groupItem in group.GetAllItems())
                {
                    if (groupItem.Target == null || !groupItem.Target.IsHierarchicalConvertTarget())
                    {
                        continue;
                    }

                    this.ValidateGroupItemRegisterTypes(context, groupItem);
                }
            }
        }

        private void ValidateGroupItemRegisterTypes(SoundArchiveContext context, GroupItemBase groupItem)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(groupItem);

            ValidationResult result = GroupItemRegisterTypeValidator.Validate(groupItem);

            if (result.IsValid)
            {
                return;
            }

            context.Logger.AddLine(
                new ErrorLine(result.ToString(), groupItem.Parent));
        }

        /// <summary>
        /// 各アイテムのバイナライズなどを行う ComponentProcessor を生成します。
        /// </summary>
        /// <param name="context">対象のコンテキスト。</param>
        private void CreateComponentProcessors(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);

            SoundArchiveProcessorFactorySelector runner = new SoundArchiveProcessorFactorySelector(this);

            foreach (SoundSetItem soundSetItem in context.AllItems)
            {
                if (this.IsWaveSound2BinaryRequired(soundSetItem))
                {
                    foreach (var waveSoundClip in ((WaveSound)soundSetItem).GetClipConvertModels())
                    {
                        runner.Run(context, waveSoundClip);
                    }
                }
                else
                {
                    runner.Run(context, soundSetItem);
                }
            }
        }

        /// <summary>
        /// 並列処理が可能になるように ComponentProcessor の依存関係を設定します。
        /// </summary>
        /// <param name="context">対象のコンテキスト。</param>
        private void LinkComponentProcessors(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);

            ComponentProcessorLinkerSelector runner = new ComponentProcessorLinkerSelector();

            foreach (SoundSetItem soundSetItem in context.AllItems)
            {
                runner.Run(context, soundSetItem);
            }
        }

        /// <summary>
        /// 重複ファイルを取り除くなど、ファイルの最適化を行います。
        /// </summary>
        /// <param name="context">対象のコンテキスト。</param>
        private void OptimizeFiles(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);

            FileOptimizerSelector runner = new FileOptimizerSelector();

            foreach (WaveArchiveBase waveArchive in context.WaveArchives)
            {
                runner.Run(context, waveArchive);
            }

            foreach (GroupBase group in context.Groups)
            {
                runner.Run(context, group);
            }
        }

        /// <summary>
        /// コンバート後のキャッシュを有効化するために、依存ファイルを設定します。
        /// </summary>
        /// <param name="context">対象のコンテキスト。</param>
        private void LinkDependedFiles(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);

            DependedFileLinkerSelector runner = new DependedFileLinkerSelector(this.projectService, this.fileManager);

            foreach (SoundSetItem soundSetItem in context.AllItems)
            {
                runner.Run(context, soundSetItem);
            }
        }

        /// <summary>
        /// サウンドアーカイブのアイテムインデックスを振りなおします。
        /// </summary>
        /// <param name="context">サウンドアーカイブコンテキストを指定します。</param>
        /// <remarks>
        /// 並列コンバートに対応するために、アイテムインデックスの遅延割り振りができません。
        /// スケジューラによる実処理開始前に全て割り振っておく必要があります。
        /// </remarks>
        private void RenumberItemIndexes(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);

            foreach (WaveArchiveBase waveArchive in context.WaveArchives)
            {
                waveArchive.GetItemOutputTargets().RenumberItems();
                waveArchive.GetItems().RenumberItems();
            }
        }

        /// <summary>
        /// サウンドアーカイブを生成する ComponentProcessor を作成します。
        /// </summary>
        /// <param name="context">対象のコンテキスト。</param>
        private void CreateSoundArchiveProcessor(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);

            FileOutputItem outputItem = new FileOutputItem(
                GetSoundArchiveFilePath(context),
                filePath => this.IsSoundArchiveDirty(context, filePath));

            FileOutputItem mapOutputItem = new FileOutputItem(
                GetSoundArchiveMapFilePath(context),
                filePath => this.IsSoundArchiveDirty(context, filePath));

            IConversionProcessor soundArchiveProcessor =
                new SoundArchiveProcessor(this.projectService.Project, outputItem, mapOutputItem);

            context.ConversionProcessors.Add(soundArchiveProcessor);

            foreach (StreamSoundBase soundSetItem in context.StreamSounds)
            {
                this.LinkSoundArchiveProcessorToComponentProcessors(
                    context, soundArchiveProcessor, soundSetItem);
            }

            if (context.Traits.IsWaveSound2BinaryEnabled)
            {
                foreach (var waveSound in context.WaveSoundSets
                    .SelectMany<WaveSoundSetBase, Component>(waveSoundSet => waveSoundSet.Children)
                    .Where(waveSound => waveSound.IsConvertTarget())
                    .Cast<WaveSound>())
                {
                    this.LinkSoundArchiveProcessorToComponentProcessors(
                        context, soundArchiveProcessor, waveSound);
                }
            }
            else
            {
                foreach (WaveSoundSetBase soundSetItem in context.WaveSoundSets)
                {
                    this.LinkSoundArchiveProcessorToComponentProcessors(
                        context, soundArchiveProcessor, soundSetItem);
                }
            }

            foreach (SequenceSoundBase soundSetItem in context.SequenceSounds)
            {
                this.LinkSoundArchiveProcessorToComponentProcessors(
                    context, soundArchiveProcessor, soundSetItem);
            }

            foreach (SoundSetBankBase soundSetItem in context.SoundSetBanks)
            {
                // メインサウンドアーカイブのバンクを追加サウンドアーカイブで参照する場合はバンク情報のみを出力するので、スキップする
                if (context.IsExternalSoundArchiveBank(soundSetItem))
                {
                    continue;
                }

                this.LinkSoundArchiveProcessorToComponentProcessors(
                    context, soundArchiveProcessor, soundSetItem);
            }

            foreach (WaveArchiveBase soundSetItem in context.WaveArchives)
            {
                this.LinkSoundArchiveProcessorToComponentProcessors(
                    context, soundArchiveProcessor, soundSetItem);
            }

            context.SoundArchiveOutput = outputItem;
            context.SoundArchiveMapOutput = mapOutputItem;
        }

        private void LinkSoundArchiveProcessorToComponentProcessors(
            SoundArchiveContext context, IConversionProcessor soundArchiveProcessor, SoundSetItem soundSetItem)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(soundArchiveProcessor);
            Assertion.Argument.NotNull(soundSetItem);

            foreach (IConversionProcessor processor in
                context.GetProcessors(soundSetItem.GetOutputTarget()))
            {
                soundArchiveProcessor.Dependencies.Add(processor);
            }
        }

        private bool IsSoundArchiveDirty(SoundArchiveContext context, string filePath)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(filePath);
            Assertion.Argument.StringNotEmpty(filePath);

            // バイナリサウンドアーカイブファイルの存在を確認します。
            if (!File.Exists(filePath))
            {
                return true;
            }

            DateTime soundArchiveFileDate = File.GetLastWriteTime(filePath);

            // プロジェクトファイルより古い場合は、Dirtyとします。
            if (File.GetLastWriteTime(this.projectService.ProjectDocument.Resource.Key) > soundArchiveFileDate)
            {
                return true;
            }

            // サウンドセットファイルより古い場合は、Dirtyとします。
            foreach (string soundSetFilePath in (this.projectService.SoundSetDocuments
                                                 .Where(d => context.IsTargetSoundSet(d.SoundSet))
                                                 .Select(d => d.Resource.Key)))
            {
                if (File.GetLastWriteTime(soundSetFilePath) > soundArchiveFileDate)
                {
                    return true;
                }
            }

            // 各出力ファイルより古い場合は、Dirtyとします。
            foreach (SoundSetItem soundSetItem in context.AllItems)
            {
                IOutput outputTarget = soundSetItem.GetOutputTarget();
                if (outputTarget == null)
                {
                    continue;
                }

                foreach (IOutputItem outputTargetItem in outputTarget.ItemDictionary.Values)
                {
                    if (outputTargetItem.Path.Length == 0)
                    {
                        continue;
                    }

                    if (File.GetLastWriteTime(outputTargetItem.Path) > soundArchiveFileDate)
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        private string GetSoundArchiveFilePath(SoundArchiveContext context)
        {
            return Path.Combine(
                this.projectService.ProjectOutputPath,
                context.GetSoundArchiveName() + "." +
                this.fileManager.BinaryOutputTraits.SoundArchiveBinaryFileExtension);
        }

        private string GetSoundArchiveMapFilePath(SoundArchiveContext context)
        {
            return Path.Combine(
                this.projectService.ProjectOutputPath,
                context.GetSoundArchiveName() + ".xml");
        }

        /// <summary>
        /// コンバートのために作成した情報をクリーンアップします。
        /// </summary>
        /// <param name="context">コンテキストを指定します。</param>
        private void Cleanup(Component component)
        {
            Assertion.Argument.NotNull(component);

            foreach (Component child in component.Children)
            {
                this.Cleanup(child);
            }

            this.CleanupParameters(component);
        }

        /// <summary>
        /// パラメータをクリーンアップします。
        /// </summary>
        /// <param name="parameterProvider">対象のパラメータプロバイダを指定します。</param>
        private void CleanupParameters(IParameterProvider parameterProvider)
        {
            Assertion.Argument.NotNull(parameterProvider);

            // 全てのテンポラリパラメータを削除します。
            foreach (string key in parameterProvider.Parameters.Keys)
            {
                if (!key.EndsWith(ParameterNames.TemporaryPostfix))
                {
                    continue;
                }

                parameterProvider.Parameters.RemoveValue(key);
            }
        }

        private bool IsWaveSound2BinaryRequired(Component component)
        {
            return component is WaveSound && ((WaveSound)component).IsWaveSound2BinaryRequired();
        }

        private bool IsTargetSoundSetItem<TSoundSetItem>(TSoundSetItem item)
            where TSoundSetItem : SoundSetItem
        {
            Assertion.Operation.ObjectNotNull(this.projectService);

            if (this.addonSoundSet == null)
            {
                SoundArchiveOutputTypes outputType;

                if (!this.projectService.Project.SoundArchiveOutputTypes.TryGetValue(item.SoundSet, out outputType))
                {
                    return true;
                }

                return outputType == SoundArchiveOutputTypes.SoundArchive;
            }
            else
            {
                return object.ReferenceEquals(this.addonSoundSet, item.SoundSet);
            }
        }

        private bool IsItemMatched<Type>(Type item, ItemFilter filter) where Type : Component
        {
            Assertion.Operation.ObjectNotNull(filter);

            if (filter.Patterns == null)
            {
                return false;
            }

            foreach (var pattern in filter.Patterns)
            {
                Regex regexPattern = new Regex(pattern);

                if (regexPattern.Match(item.Name).Success)
                {
                    return true;
                }
            }
            return false;
        }

        private void ExcludeItem<Type>(ref Type item, ItemFilter filter) where Type : Component
        {
            if (filter == null)
            {
                return;
            }

            if (IsItemMatched(item, filter))
            {
                item.SetIsExcluded(true);
            }
        }

        private void IncludeItem<Type>(ref Type item, ItemFilter filter) where Type : Component
        {
            if (filter == null)
            {
                return;
            }

            if (!IsItemMatched(item, filter))
            {
                item.SetIsExcluded(true);
            }
        }

        private IEnumerable<SoundSetItem> EnumerateTargetSoundSetItems(bool isConvertParts)
        {
            Assertion.Operation.ObjectNotNull(this.projectService);

            var items = this.projectService.SoundSetItems.Where(item => item.IsEnabled);
            return isConvertParts ? items : items.Where(item => this.IsTargetSoundSetItem(item));
        }

        private IEnumerable<StreamSoundBase> EnumerateTargetStreamSounds(bool isConvertParts)
        {
            Assertion.Operation.ObjectNotNull(this.projectService);

            var items = this.projectService.StreamSounds.Where(item => item.IsEnabled);
            return isConvertParts ? items : items.Where(item => this.IsTargetSoundSetItem(item));
        }

        private IEnumerable<WaveSoundSetBase> EnumerateTargetWaveSoundSets(bool isConvertParts)
        {
            Assertion.Operation.ObjectNotNull(this.projectService);

            var items = this.projectService.WaveSoundSets.Where(item => item.IsEnabled);
            return isConvertParts ? items : items.Where(item => this.IsTargetSoundSetItem(item));
        }

        private IEnumerable<SequenceSoundSetBase> EnumerateTargetSequenceSoundSets(bool isConvertParts)
        {
            Assertion.Operation.ObjectNotNull(this.projectService);

            var items = this.projectService.SequenceSoundSets.Where(item => item.IsEnabled);
            return isConvertParts ? items : items.Where(item => this.IsTargetSoundSetItem(item));
        }

        private IEnumerable<SequenceSoundBase> EnumerateTargetSequenceSounds(bool isConvertParts)
        {
            Assertion.Operation.ObjectNotNull(this.projectService);

            var items = this.projectService.SequenceSounds.Where(item => item.IsEnabled);
            return isConvertParts ? items : items.Where(item => this.IsTargetSoundSetItem(item));
        }

        private IEnumerable<SoundSetBankBase> EnumerateTargetSoundSetBanks(SoundArchiveContext context)
        {
            Assertion.Operation.ObjectNotNull(this.projectService);

            var items = this.projectService.SoundSetBanks.Where(item => item.IsEnabled);

            if (context.Settings.IsConvertParts)
            {
                foreach (var soundSetBank in items)
                {
                    yield return soundSetBank;
                }
            }
            else
            {
                foreach (var soundSetBank in items.Where(item => this.IsTargetSoundSetItem(item)))
                {
                    yield return soundSetBank;
                }
            }

            // 追加サウンドアーカイブの場合は、対象サウンドセットに含まれるバンクに加えて、サウンドが参照するバンクを含める
            // ただし、BankID は、追加サウンドアーカイブのローカルID として利用され、
            // メインサウンドアーカイブの BankID としては利用されない前提です
            if (this.addonSoundSet != null)
            {
                if (context.Settings.ExcludeStringTable)
                {
                    context.Logger.AddLine(
                        new ErrorLine(MessageResource.Message_MustContainsStringTableForAddonSoundArchive));
                    yield break;
                }
                else
                {
                    foreach (var soundSetBank in context.SequenceSounds
                        .SelectMany(sound => sound.SoundSetBanks)
                        .Where(bank => bank != null)
                        .Where(bank => bank.IsConvertTarget())
                        .Distinct())
                    {
                        yield return soundSetBank;
                    }
                }
            }
        }

        private IEnumerable<WaveArchiveBase> EnumerateTargetWaveArchives(bool isConvertParts)
        {
            Ensure.Operation.ObjectNotNull(this.projectService);

            var items = this.projectService.WaveArchives.Where(item => item.IsEnabled);
            return isConvertParts ? items : items.Where(item => this.IsTargetSoundSetItem(item));
        }

        private IEnumerable<GroupBase> EnumerateTargetGroups()
        {
            Ensure.Operation.ObjectNotNull(this.projectService);

            if (this.addonSoundSet == null)
            {
                return this.projectService.Groups
                    .Where(item => item.IsEnabled)
                    .Where(item => this.IsTargetSoundSetItem(item));
            }
            else
            {
                // 追加サウンドアーカイブにグループは含めない
                return new GroupBase[0];
            }
        }

        private IEnumerable<PlayerBase> EnumerateTargetPlayers(SoundArchiveContext context)
        {
            Ensure.Operation.ObjectNotNull(this.projectService);

            if (this.addonSoundSet == null)
            {
                return this.projectService.Players
                    .Where(item => item.IsEnabled)
                    .Where(item => this.IsTargetSoundSetItem(item));
            }
            else
            {
                // 追加サウンドアーカイブの場合は、サウンドが参照するプレイヤーを含める
                // ただし、PlayerID は、追加サウンドアーカイブのローカルID として利用され、
                // メインサウンドアーカイブの PlayerID としては利用されない前提です
                if (context.Settings.ExcludeStringTable)
                {
                    context.Logger.AddLine(
                        new ErrorLine(MessageResource.Message_MustContainsStringTableForAddonSoundArchive));
                    return new PlayerBase[0];
                }
                else
                {
                    return context.AllSounds
                        .Select(sound => sound.TargetPlayer)
                        .Where(player => player != null)
                        .Where(player => player.IsEnabled)
                        .Distinct();
                }
            }
        }
    }
}
