﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Codecs
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Runtime.InteropServices;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// ADPCMにエンコードするフィルタです。
    /// </summary>
    internal class AdpcmEncoder : Filter
    {
        private object adpcmInformation;
        private object[] regionAdpcmContexts = new object[0];

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public AdpcmEncoder()
            : base(new InputConnectionPoint())
        {
            this.BlockLength = 0;
            this.ExtraSamplesForSmoothLoop = 0;
        }

        /// <summary>
        /// 出力エンコーディングを指定します。
        /// </summary>
        public string OutputEncoding { get; set; }

        /// <summary>
        /// 出力データのブロックの長さを取得または設定します。
        /// </summary>
        public int BlockLength { get; set; }

        /// <summary>
        /// ループのつなぎを滑らかにするための余分なサンプル数を取得または設定します。
        /// </summary>
        public int ExtraSamplesForSmoothLoop { get; set; }

        /// <summary>
        /// ADPCM情報を取得します。
        /// </summary>
        public object AdpcmInformation
        {
            get { return this.adpcmInformation; }
        }

        /// <summary>
        /// リージョン毎の ADPCM コンテキスト情報を取得します。
        /// </summary>
        public object[] RegionAdpcmContexts
        {
            get { return this.regionAdpcmContexts; }
        }

        /// <summary>
        /// 入力ストリーム情報を取得します。
        /// </summary>
        private WaveStreamInformation InputStreamInformation
        {
            get { return this.Inputs[0].StreamInformation; }
        }

        /// <summary>
        /// 入力フォーマットを取得します。
        /// </summary>
        private WaveFormat InputFormat
        {
            get { return this.InputStreamInformation.Format; }
        }

        /// <summary>
        /// 出力ストリーム情報を取得します。
        /// </summary>
        private WaveStreamInformation OutputStreamInformation
        {
            get { return this.Outputs[0].StreamInformation; }
        }

        /// <summary>
        /// 出力フォーマットを取得します。
        /// </summary>
        private WaveFormat OutputFormat
        {
            get { return this.OutputStreamInformation.Format; }
        }

        protected override void ValidateInputs(InputConnectionPoint[] inputs)
        {
            Ensure.Operation.True(
                inputs[0].StreamInformation.Format.ChannelCount == 1,
                "InputChannel for AdpcmEncoder must be 1."
                );
        }

        /// <summary>
        /// 出力接続ポイントを生成します。
        /// </summary>
        /// <returns>出力接続ポイントの配列を返します。</returns>
        protected override OutputConnectionPoint[] CreateOutputs()
        {
            int extraLengthForSmoothLoop = 0;
            int extraSamplesForSmoothLoop = 0;

            if (this.InputFormat.HasLoop)
            {
                extraLengthForSmoothLoop = this.ExtraSamplesForSmoothLoop * this.InputFormat.FrameLength;
                extraSamplesForSmoothLoop = this.ExtraSamplesForSmoothLoop;
            }

            Ensure.Operation.True(
                this.InputStreamInformation.Length == 0
                || this.InputStreamInformation.Length > extraLengthForSmoothLoop);
            Ensure.Operation.True(
                this.InputStreamInformation.FrameCount == 0
                || this.InputStreamInformation.FrameCount > extraSamplesForSmoothLoop);

            return new OutputConnectionPoint[]
            {
                new OutputConnectionPoint(
                    new WaveStreamInformation(
                        this.GetOutputLength(this.InputStreamInformation.Length - extraLengthForSmoothLoop),
                        this.InputStreamInformation.FrameCount - extraSamplesForSmoothLoop,
                        this.GetOutputFormat() )
                    )
            };
        }

        /// <summary>
        /// パケットを処理します。
        /// </summary>
        /// <param name="input">入力接続ポイントを指定します。</param>
        /// <param name="packet">パケットを指定します。</param>
        protected override void Process(InputConnectionPoint input, Packet packet)
        {
            Assertion.Argument.NotNull(packet);
            Ensure.Argument.True(packet.Samples.Length == this.InputStreamInformation.Length);

            int outputLength = this.GetOutputLength(packet.Samples.Length);
            byte[] outputSamples = new byte[outputLength];

            AdpcmPacket outputPacket = AdpcmPacket.Create(
                outputSamples, this.ExtractAdpcmSeekInformation(packet));

            GCHandle inputHandle = GCHandle.Alloc(packet.Samples, GCHandleType.Pinned);
            GCHandle outputHandle = GCHandle.Alloc(outputPacket.Samples, GCHandleType.Pinned);

            try
            {
                int frameCount = packet.Samples.Length / this.InputFormat.FrameLength;

                switch (this.OutputFormat.Encoding)
                {
                    case Encoding.DspAdpcm:
                        {
                            IntPtr outputPtr = outputHandle.AddrOfPinnedObject();

                            WaveCodecDspAdpcmData waveCodecAdpcmInfo;
                            EncodeDspAdpcm(
                                           outputPtr,
                                           out waveCodecAdpcmInfo,
                                           inputHandle.AddrOfPinnedObject(),
                                           this.InputFormat.HasLoop ?
                                           this.InputFormat.LoopStartFrame : 0,
                                           packet.Samples.Length / this.InputFormat.FrameLength
                                           );
                            this.adpcmInformation = waveCodecAdpcmInfo.ToAdpcmInformation();
                            this.regionAdpcmContexts = this.GetRegionAdpcmContexts(outputPtr);
                            break;
                        }

                    default:
                        throw new Exception("invalid format.");
                }

                if (this.InputFormat.HasLoop)
                {
                    int extraSamplesLength = this.GetOutputLength(this.ExtraSamplesForSmoothLoop * this.InputFormat.FrameLength);

                    if (outputPacket.ValidSamplesLength >= extraSamplesLength)
                    {
                        outputPacket.ValidSamplesLength -= extraSamplesLength;
                    }
                }

                this.SendToAll(outputPacket);
            }
            finally
            {
                inputHandle.Free();
                outputHandle.Free();
            }
        }

        [DllImport("WaveCodecGeneric.dll", CallingConvention = CallingConvention.Cdecl)]
        private static extern int GetBufferSizeForEncodeDspAdpcm(int frameCount);

        [DllImport("WaveCodecGeneric.dll", CallingConvention = CallingConvention.Cdecl)]
        private static extern void EncodeDspAdpcm(
            IntPtr output,
            [Out] out WaveCodecDspAdpcmData info,
            IntPtr input,
            int loopStartSample,
            int frameCount);

        [DllImport("WaveCodecGeneric.dll", EntryPoint = "getLoopContext", CallingConvention = CallingConvention.Cdecl)]
        private static extern void GetLoopContext
        (
            IntPtr adpcmData,
            [Out] out WaveCodecDspAdpcmData info,
            UInt32 samples
        );

        /// <summary>
        /// 出力の長さを取得します。
        /// </summary>
        /// <param name="inputLength">入力の長さを指定します。</param>
        /// <returns>出力の長さを返します。</returns>
        private int GetOutputLength(int inputLength)
        {
            switch (this.OutputEncoding)
            {
                case Encoding.DspAdpcm:
                    return GetBufferSizeForEncodeDspAdpcm(inputLength / this.InputFormat.FrameLength);
            }

            throw new Exception("invalid encording.");
        }

        /// <summary>
        /// 出力フォーマットを取得します。
        /// </summary>
        /// <returns>出力フォーマットを返します。</returns>
        private WaveFormat GetOutputFormat()
        {
            return new WaveFormat(this.InputFormat)
            {
                Encoding = this.OutputEncoding,
            };
        }

        /// <summary>
        /// ADPCMシーク情報のバイト列を抽出します。
        /// </summary>
        /// <param name="waveStreamData">ストリームコーデック情報を指定します。</param>
        /// <returns>ADPCMシーク情報のバイト列を返します。</returns>
        private byte[] ExtractAdpcmSeekInformation(Packet packet)
        {
            Assertion.Argument.NotNull(packet);

            if (this.BlockLength <= 0)
            {
                return new byte[0];
            }

            int sampleCountPerBlock = this.BlockLength * 14 / 8;
            int itemLength = 2 * this.InputFormat.FrameLength;

            // 2サンプル(Yn1とYn2分) × ストリームブロック数
            byte[] results = new byte[
               itemLength * ((this.InputStreamInformation.FrameCount + sampleCountPerBlock - 1) /
               sampleCountPerBlock)];
            int writtenLength = 0;

            for (int currentSample = 0;
                currentSample < this.InputStreamInformation.FrameCount;
                currentSample += sampleCountPerBlock)
            {
                if (currentSample == 0)
                {
                    // 先頭フレームは 0 のままでOK。
                }
                else
                {
                    Ensure.Operation.True(currentSample >= 2);

                    // Yn1
                    Array.Copy(
                        packet.Samples,
                        (currentSample - 1) * this.InputFormat.FrameLength,
                        results,
                        writtenLength,
                        this.InputFormat.FrameLength);

                    // Yn2
                    Array.Copy(
                        packet.Samples,
                        (currentSample - 2) * this.InputFormat.FrameLength,
                        results,
                        writtenLength + this.InputFormat.FrameLength,
                        this.InputFormat.FrameLength);
                }

                writtenLength += itemLength;
            }

            return results;
        }

        private object[] GetRegionAdpcmContexts(IntPtr outputPtr)
        {
            if (this.InputFormat.Regions == null)
            {
                return new object[0];
            }

            List<object> result = new List<object>();

            // 波形長を超えるリージョンは破棄する
            foreach (var region in this.InputFormat.Regions
                .Where(region => region.EndFrame <= this.OutputStreamInformation.FrameCount))
            {
                DspAdpcmContext context = new DspAdpcmContext();

                if (region != null)
                {
                    WaveCodecDspAdpcmData adpcmData = new WaveCodecDspAdpcmData();
                    GetLoopContext(outputPtr, out adpcmData, region.StartFrame);

                    context = new DspAdpcmContext()
                    {
                        Pred_scale = adpcmData.Loop_pred_scale,
                        Yn1 = adpcmData.Loop_yn1,
                        Yn2 = adpcmData.Loop_yn2,
                    };
                }

                result.Add(context);
            }

            return result.ToArray();
        }
    }
}
