﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Codecs
{
    using System;
    using System.Collections.Generic;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// Waveストリームにエンコードします。
    /// </summary>
    internal abstract class NintendoWareStreamEncoder : NintendoWareWaveEncoder
    {
        /// <summary>
        /// 波形データをエンコードします。
        /// </summary>
        /// <param name="waveStreams">波形ストリームの配列を指定します。</param>
        /// <returns>出力を返します。</returns>
        public override CodecOutput[] Run(WaveStream[] waveStreams)
        {
            Ensure.Argument.NotNull(waveStreams);
            Ensure.Argument.True(waveStreams.Length > 0);

            WaveStream[] modifiedStreams = this.FitLoopRange(waveStreams);

            try
            {
                SourceStreamReader[] readers = this.InitializeFilters(
                    modifiedStreams,
                    this.GetNormalizeFrameCount(waveStreams)
                    );

                foreach (SourceStreamReader reader in readers)
                {
                    reader.Run();
                }
            }
            finally
            {
                foreach (var modifiedStream in modifiedStreams)
                {
                    modifiedStream.Dispose();
                }
            }

            return new CodecOutput[] { this.CreateOutput() };
        }

        // HACK : ★IRunableFilter インターフェイスの切り出しを検討。
        /// <summary>
        /// フィルタを初期化します。
        /// </summary>
        /// <param name="waveStream">波形ストリームの配列を指定します。</param>
        /// <param name="normalizeOutputLength">正規化する入力波形ストリームの長さを指定します。</param>
        /// <returns>開始フィルタの配列を返します。</returns>
        protected abstract SourceStreamReader[] InitializeFilters(WaveStream[] waveStreams, int normalizeOutputLength);

        /// <summary>
        /// エンコード処理の出力を作成します。
        /// </summary>
        /// <returns>エンコード処理の出力を返します。</returns>
        protected abstract CodecOutput CreateOutput();

        /// <summary>
        /// 正規化の出力フレーム数を取得します。
        /// </summary>
        /// <param name="inputs">波形ストリームの配列を指定します。</param>
        /// <returns>
        /// ワンショット波形の場合は、一番長い波形のフレーム数を返します。
        /// ループ波形の場合は、先頭トラックのループ終了フレーム数を返します。
        /// </returns>
        private int GetNormalizeFrameCount(WaveStream[] inputs)
        {
            Assertion.Argument.NotNull(inputs);

            if (!inputs[0].Format.HasLoop)
            {
                int maxFrameCount = 0;

                foreach (WaveStream input in inputs)
                {
                    int frameCount = (int)input.Payload.Length / input.Format.FrameLength;

                    if (maxFrameCount < frameCount)
                    {
                        maxFrameCount = frameCount;
                    }
                }

                return maxFrameCount;
            }

            return Math.Min(
                (int)inputs[0].Payload.Length / inputs[0].Format.FrameLength,
                inputs[0].Format.LoopEndFrame
                );
        }

        /// <summary>
        /// 最初の波形がループ波形なら、全てのトラックをそのループ範囲に合わせます。
        /// </summary>
        /// <param name="inputs">波形ストリームの配列を指定します。</param>
        /// <returns>調整後の波形ストリーム配列を返します。</returns>
        private WaveStream[] FitLoopRange(WaveStream[] inputs)
        {
            Assertion.Argument.NotNull(inputs);

            WaveStream[] outputs = new WaveStream[inputs.Length];

            outputs[0] = inputs[0];

            for (int i = 1; i < inputs.Length; i++)
            {
                WaveFormat format = new WaveFormat(inputs[i].Format)
                {
                    HasLoop = inputs[0].Format.HasLoop,
                    OriginalLoopStartFrame = inputs[0].Format.OriginalLoopStartFrame,
                    OriginalLoopEndFrame = inputs[0].Format.OriginalLoopEndFrame,
                    LoopStartFrame = inputs[0].Format.LoopStartFrame,
                    LoopEndFrame = inputs[0].Format.LoopEndFrame,
                };

                outputs[i] = new WaveStream(format, inputs[i].Payload);
            }

            return outputs;
        }
    }
}
