﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Codecs
{
    using System.Collections.Generic;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// IMAADPCM にエンコードします。
    /// </summary>
    internal class NintendoWareImaAdpcmEncoder : NintendoWareWaveEncoder
    {
        private SourceStreamReader reader;
        private List<SampleGrabber> sampleGrabbers = new List<SampleGrabber>();
        private List<AdpcmEncoder> adpcmEncoders = new List<AdpcmEncoder>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public NintendoWareImaAdpcmEncoder()
        {
            this.OutputEncoding = Encoding.ImaAdpcm;
        }

        /// <summary>
        /// 出力エンコーディングを指定します。
        /// </summary>
        /// <remarks>
        /// "DSPADPCM", "IMAADPCM" を指定できます。
        /// </remarks>
        public string OutputEncoding { get; set; }

        /// <summary>
        /// 波形データをエンコードします。
        /// </summary>
        /// <param name="waveStreams">波形ストリームの配列を指定します。</param>
        /// <returns>出力の配列を返します。</returns>
        public override CodecOutput[] Run(WaveStream[] waveStreams)
        {
            Ensure.Argument.NotNull(waveStreams);
            Ensure.Argument.True(waveStreams.Length == 1);

            this.Initialize(waveStreams[0]);

            this.reader.Run();

            List<CodecOutput> codecOutputs = new List<CodecOutput>();

            for (int i = 0; i < this.sampleGrabbers.Count; i++)
            {
                codecOutputs.Add(
                    new CodecOutput()
                    {
                        Data = this.sampleGrabbers[i].Samples ?? new byte[0],
                        Format = this.sampleGrabbers[i].Inputs[0].StreamInformation.Format,
                        FrameCount = this.sampleGrabbers[i].Inputs[0].StreamInformation.FrameCount,
                        CodecData = this.adpcmEncoders[i].AdpcmInformation,
                    });
            }

            return codecOutputs.ToArray();
        }

        /// <summary>
        /// 初期化します。
        /// </summary>
        /// <param name="waveStream">波形ストリームを指定します。</param>
        private void Initialize(WaveStream waveStream)
        {
            Assertion.Argument.NotNull(waveStream);

            this.sampleGrabbers.Clear();

            // 1. SourceStreamReader
            this.reader = new SourceStreamReader()
            {
                Source = waveStream.Payload,
                SourceFormat = waveStream.Format,
            };

            // 2. LinearPcmNormalizer
            LinearPcmNormalizer normalizer = new LinearPcmNormalizer();
            this.reader.Outputs[0].ConnectTo(normalizer.Inputs[0]);

            // 3. ChannelDivider
            ChannelDivider channelDivider = new ChannelDivider();
            normalizer.Outputs[0].ConnectTo(channelDivider.Inputs[0]);

            for (int i = 0; i < channelDivider.Outputs.Length; i++)
            {
                // 4. SampleSpooler
                // SamplingrateConverter は部分コンバートに対応していないので、
                // パケットを１つにまとめる必要があります。
                SampleSpooler sampleSpooler = new SampleSpooler();
                channelDivider.Outputs[i].ConnectTo(sampleSpooler.Inputs[0]);

                // 5. SamplingrateConverter
                SamplingrateConverter samplingrateConverter = new SamplingrateConverter();
                sampleSpooler.Outputs[0].ConnectTo(samplingrateConverter.Inputs[0]);

                // 6. LoopedSampleAligner
                LoopedSampleAligner loopedSampleAligner = new LoopedSampleAligner()
                {
                    LoopStartAlignmentFrames = this.GetLoopStartAlignmentFrames(),
                };
                samplingrateConverter.Outputs[0].ConnectTo(loopedSampleAligner.Inputs[0]);

                // 7. SampleSpooler
                // AdpcmEncoder は部分エンコードに対応していないので、
                // パケットを１つにまとめる必要があります。
                sampleSpooler = new SampleSpooler();
                loopedSampleAligner.Outputs[0].ConnectTo(sampleSpooler.Inputs[0]);

                // 8. AdpcmEncoder
                AdpcmEncoder adpcmEncoder = new AdpcmEncoder()
                {
                    OutputEncoding = this.OutputEncoding,
                };
                sampleSpooler.Outputs[0].ConnectTo(adpcmEncoder.Inputs[0]);

                this.adpcmEncoders.Add(adpcmEncoder);

                // 9. SampleGrabber
                SampleGrabber sampleGrabber = new SampleGrabber();
                adpcmEncoder.Outputs[0].ConnectTo(sampleGrabber.Inputs[0]);

                this.sampleGrabbers.Add(sampleGrabber);
            }
        }

        /// <summary>
        /// ループ開始位置のフレームアライメントを取得します。
        /// </summary>
        /// <returns>ループ開始位置のフレームアライメントを返します。</returns>
        private int GetLoopStartAlignmentFrames()
        {
            // IMA ADPCM : 2 Sample / 1 Byte
            return 2 / 1;
        }
    }
}
