﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;

using NintendoWare.SoundFoundation.Core.Parameters;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public enum ListPartActions
    {
        None = 0,
        ItemSelectable = 1,
        ItemDragable = 2,
        DoubleClickable = 4,

        EditByClick = 8,
        EditByClickOnCaret = 16,
        EditByClickDelayOnCaret = 32,        //未実装
        EditByDoubleClick = 64,
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// パーツ
    /// </summary>
    public class ListPart : IListPart
    {
        private static ListPartActions _DefaultAction =
          (ListPartActions.EditByClickOnCaret |
            ListPartActions.EditByDoubleClick);

        private string _DrawerName = null;
        private string _EditorName = null;
        private Rectangle _Bounds;
        private ListPartActions _Action = ListPartActions.None;

        private string _EditorNameByDoubleClick = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListPart(string drawerName, string editorName) :
            this(drawerName, editorName, Rectangle.Empty)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListPart(string drawerName, string editorName, Rectangle bounds) :
            this(drawerName, editorName, null, bounds, _DefaultAction)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListPart(string drawerName, string editorName, ListPartActions action) :
            this(drawerName, editorName, null, Rectangle.Empty, action)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListPart(string drawerName, string editorName, string editorNameByDoubleClick, ListPartActions action) :
            this(drawerName, editorName, editorNameByDoubleClick, Rectangle.Empty, action)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListPart(string drawerName, string editorName, string editorNameByDoubleClick, Rectangle bounds, ListPartActions action)
        {
            _DrawerName = drawerName;
            _EditorName = editorName;
            _EditorNameByDoubleClick = editorNameByDoubleClick;
            _Bounds = bounds;
            _Action = action;
        }

        ///--------------------------------
        /// <summary>
        /// パーツ名
        /// Drawerの検索に使用
        /// </summary>
        public string DrawerName
        {
            get { return _DrawerName; }
        }

        ///--------------------------------
        /// <summary>
        /// エディタ名
        /// Editorの検索に使用
        /// </summary>
        public string EditorName
        {
            get { return _EditorName; }
        }

        ///--------------------------------
        /// <summary>
        /// ダブルクリック時のエディタ名
        /// Editorの検索に使用
        /// </summary>
        public string EditorNameByDoubleClick
        {
            get
            {
                if (_EditorNameByDoubleClick != null) { return _EditorNameByDoubleClick; }
                return _EditorName;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Rectangle Bounds
        {
            get { return _Bounds; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListPartActions Action
        {
            get { return _Action; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool ItemSelectable
        {
            get { return CanAction(ListPartActions.ItemSelectable); }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool ItemDragable
        {
            get { return CanAction(ListPartActions.ItemDragable); }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool DoubleClickable
        {
            get { return CanAction(ListPartActions.DoubleClickable); }
        }

        ///--------------------------------
        /// <summary>
        /// 1クリックで編集できるのか
        /// </summary>
        public bool EditByClick
        {
            get { return CanAction(ListPartActions.EditByClick); }
        }

        ///--------------------------------
        /// <summary>
        /// カレット上の 1クリックで編集できるのか
        /// </summary>
        public bool EditByClickOnCaret
        {
            get { return CanAction(ListPartActions.EditByClickOnCaret); }
        }

        ///--------------------------------
        /// <summary>
        /// ダブルクリックで編集できるのか
        /// </summary>
        public bool EditByDoubleClick
        {
            get { return CanAction(ListPartActions.EditByDoubleClick); }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool CanAction(ListPartActions action)
        {
            return (Action & action) != 0 ? true : false;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// サブアイテムの引数
    /// </summary>
    public class ListSubItemArgument : IListSubItemArgument
    {
        private IListDrawerArgument _DrawerArgument = null;
        private IInplaceEditorArgument _EditorArgument = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListSubItemArgument(IListDrawerArgument drawerArgument, IInplaceEditorArgument editorArgument)
        {
            _DrawerArgument = drawerArgument;
            _EditorArgument = editorArgument;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IListDrawerArgument DrawerArgument
        {
            get { return _DrawerArgument; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IInplaceEditorArgument EditorArgument
        {
            get { return _EditorArgument; }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ListPartParameter
    {
        private Dictionary<string, object> _Parameters = new Dictionary<string, object>();

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Dictionary<string, object> PartParameters
        {
            get { return _Parameters; }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// サブアイテム
    /// </summary>
    public class ListSubItem : ListPartParameter, IListSubItem
    {
        private static ListPart[] _Parts = new[] {
            new ListPart( String.Empty, String.Empty),
        };

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.Name
        {
            get { return String.Empty; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        IListPart[] IListSubItem.GetParts(ListDrawDescriptor desc)
        {
            return _Parts;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.RepresentativeEditorName(bool immediateEdit)
        {
            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        bool IListSubItem.EditByKey
        {
            get { return true; }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 行のヘッダ
    /// </summary>
    public class ListSubItemRowHeader : ListPartParameter, IListSubItem
    {
        private static ListPart[] _Parts = new[] {
            new ListPart( "Text", String.Empty,
                          ( ListPartActions.ItemSelectable |
                            ListPartActions.ItemDragable   |
                            ListPartActions.DoubleClickable )),
        };

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.Name
        {
            get { return "RowHeader"; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        IListPart[] IListSubItem.GetParts(ListDrawDescriptor desc)
        {
            return _Parts;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.RepresentativeEditorName(bool immediateEdit)
        {
            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        bool IListSubItem.EditByKey
        {
            get { return false; }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// テキスト
    /// </summary>
    public class ListSubItemText : ListPartParameter, IListSubItem
    {
        private static ListPart[] _Parts = new[] {
            new ListPart( "Text", "Text"),
        };

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.Name
        {
            get { return "Text"; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        IListPart[] IListSubItem.GetParts(ListDrawDescriptor desc)
        {
            return _Parts;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.RepresentativeEditorName(bool immediateEdit)
        {
            return "Text";
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        bool IListSubItem.EditByKey
        {
            get { return true; }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 整数
    /// </summary>
    public class ListSubItemInteger : ListPartParameter, IListSubItem
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListSubItemInteger()
        {
            this.Parts = new[] {
                new ListPart( "Integer", "Integer", DefaultPartAction),
            };
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.Name
        {
            get { return this.Name; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        IListPart[] IListSubItem.GetParts(ListDrawDescriptor desc)
        {
            return this.Parts;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.RepresentativeEditorName(bool immediateEdit)
        {
            return "Integer";
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        bool IListSubItem.EditByKey
        {
            get { return true; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual string Name
        {
            get { return "Integer"; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected IListPart[] Parts { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ListPartActions DefaultPartAction
        {
            get
            {
                return (ListPartActions.EditByDoubleClick |
                         ListPartActions.EditByClickOnCaret);
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 整数(ULong)
    /// </summary>
    public class ListSubItemULong : ListPartParameter, IListSubItem
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListSubItemULong()
        {
            this.Parts = new[] {
                new ListPart( "ULong", "ULong", DefaultPartAction),
            };
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.Name
        {
            get { return this.Name; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        IListPart[] IListSubItem.GetParts(ListDrawDescriptor desc)
        {
            return this.Parts;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.RepresentativeEditorName(bool immediateEdit)
        {
            return "ULong";
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        bool IListSubItem.EditByKey
        {
            get { return true; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual string Name
        {
            get { return "ULong"; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected IListPart[] Parts { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ListPartActions DefaultPartAction
        {
            get
            {
                return (ListPartActions.EditByDoubleClick |
                         ListPartActions.EditByClickOnCaret);
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 少数
    /// </summary>
    public class ListSubItemDecimal : ListPartParameter, IListSubItem
    {
        private static ListPart[] _Parts = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        static ListSubItemDecimal()
        {
            _Parts = new[] {
                new ListPart( "Decimal", "Decimal", DefaultPartAction),
            };
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.Name
        {
            get { return "Decimal"; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        IListPart[] IListSubItem.GetParts(ListDrawDescriptor desc)
        {
            return _Parts;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.RepresentativeEditorName(bool immediateEdit)
        {
            return "Decimal";
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        bool IListSubItem.EditByKey
        {
            get { return true; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected static ListPartActions DefaultPartAction
        {
            get
            {
                return (ListPartActions.EditByDoubleClick |
                         ListPartActions.EditByClickOnCaret);
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 少数(double)
    /// </summary>
    public class ListSubItemDouble : ListPartParameter, IListSubItem
    {
        private static ListPart[] _Parts = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        static ListSubItemDouble()
        {
            _Parts = new[] {
                new ListPart( "Double", "Double", DefaultPartAction),
            };
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.Name
        {
            get { return "Double"; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        IListPart[] IListSubItem.GetParts(ListDrawDescriptor desc)
        {
            return _Parts;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.RepresentativeEditorName(bool immediateEdit)
        {
            return "Double";
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        bool IListSubItem.EditByKey
        {
            get { return true; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected static ListPartActions DefaultPartAction
        {
            get
            {
                return (ListPartActions.EditByDoubleClick |
                         ListPartActions.EditByClickOnCaret);
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 少数(float)
    /// </summary>
    public class ListSubItemFloat : ListPartParameter, IListSubItem
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListSubItemFloat()
        {
            this.Parts = new[] {
                new ListPart( "Float", "Float", DefaultPartAction),
            };
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.Name
        {
            get { return this.Name; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        IListPart[] IListSubItem.GetParts(ListDrawDescriptor desc)
        {
            return this.Parts;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.RepresentativeEditorName(bool immediateEdit)
        {
            return "Float";
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        bool IListSubItem.EditByKey
        {
            get { return true; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual string Name
        {
            get { return "Float"; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected IListPart[] Parts { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ListPartActions DefaultPartAction
        {
            get
            {
                return (ListPartActions.EditByDoubleClick |
                         ListPartActions.EditByClickOnCaret);
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// サイズ
    /// </summary>
    public class ListSubItemSize : ListPartParameter, IListSubItem
    {
        private static ListPart[] _Parts = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        static ListSubItemSize()
        {
            _Parts = new[] {
                new ListPart( "Size", "Integer", DefaultPartAction),
            };
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.Name
        {
            get { return "Size"; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        IListPart[] IListSubItem.GetParts(ListDrawDescriptor desc)
        {
            return _Parts;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.RepresentativeEditorName(bool immediateEdit)
        {
            return "Integer";
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        bool IListSubItem.EditByKey
        {
            get { return true; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected static ListPartActions DefaultPartAction
        {
            get
            {
                return (ListPartActions.EditByDoubleClick |
                         ListPartActions.EditByClickOnCaret);
            }
        }
    }

    /// <summary>
    /// ファイルパス
    /// </summary>
    public class ListSubItemFilePath : ListPartParameter, IListSubItem
    {
        protected static IListPart DefaultPart = null;

        /// <summary>
        ///
        /// </summary>
        public ListSubItemFilePath()
        {
            PartParameters["Text"] = "...";
        }

        /// <summary>
        ///
        /// </summary>
        string IListSubItem.Name
        {
            get { return Name; }
        }

        /// <summary>
        ///
        /// </summary>
        bool IListSubItem.EditByKey
        {
            get { return true; }
        }

        /// <summary>
        ///
        /// </summary>
        IListPart[] IListSubItem.GetParts(ListDrawDescriptor desc)
        {
            return GetPartsInternal(desc);
        }

        /// <summary>
        ///
        /// </summary>
        string IListSubItem.RepresentativeEditorName(bool immediateEdit)
        {
            return immediateEdit == true ? "FilePath" : DialogName;
        }

        /// <summary>
        ///
        /// </summary>
        protected static ListPartActions DefaultPartAction
        {
            get
            {
                return (ListPartActions.EditByDoubleClick |
                         ListPartActions.EditByClickOnCaret);
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual string DialogName
        {
            get
            {
                return "FilePathDialog";
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual string Name
        {
            get
            {
                return "FilePath";
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected IListPart CreateDefaultPart()
        {
            return new ListPart("FilePath", "FilePath", DialogName, DefaultPartAction);
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual IListPart[] GetPartsInternal(ListDrawDescriptor desc)
        {
            if (desc.Selected == false ||
                !(desc.Parameter is FilePathParameterValue))
            {
                if (DefaultPart == null)
                {
                    DefaultPart = CreateDefaultPart();
                }
                return new[] { DefaultPart };
            }

            ListPart part0 = null;
            ListPart part1 = null;
            Rectangle bounds = new Rectangle();
            int width = 16;
            int height = 16;
            int y = (desc.Bounds.Height - height) / 2;

            //
            bounds.X = desc.Bounds.Width - width;
            bounds.Y = y;
            bounds.Width = width;
            bounds.Height = height;

            part0 = new ListPart("Button", DialogName, null, bounds,
                                  ListPartActions.EditByClick);

            //
            bounds.X = 0;
            bounds.Y = 0;
            bounds.Width = desc.Bounds.Width - width;
            bounds.Height = desc.Bounds.Height;

            part1 = new ListPart("FilePath", "FilePath", DialogName,
                                  bounds, DefaultPartAction);

            //
            return new[] { part0, part1 };
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// チェックボックス
    /// </summary>
    public class ListSubItemCheckBox : ListPartParameter, IListSubItem
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.Name
        {
            get { return this.Name; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        bool IListSubItem.EditByKey
        {
            get { return true; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        IListPart[] IListSubItem.GetParts(ListDrawDescriptor desc)
        {
            return this.GetPartsInternal(desc);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.RepresentativeEditorName(bool immediateEdit)
        {
            return "CheckBox";
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual IListPart[] GetPartsInternal(ListDrawDescriptor desc)
        {
            ListPart part0 = null;
            ListPart part1 = null;
            Rectangle bounds = new Rectangle();
            int width = desc.Bounds.Width < 16 ? desc.Bounds.Width : 16;
            int height = desc.Bounds.Height < 16 ? desc.Bounds.Height : 16;
            int x = (desc.Bounds.Width - width) / 2;
            int y = (desc.Bounds.Height - height) / 2;

            bounds.X = x;
            bounds.Y = y;
            bounds.Width = width;
            bounds.Height = height;

            part0 = new ListPart("CheckBox", "CheckBox", null, bounds,
                                  (ListPartActions.EditByClick |
                                    ListPartActions.EditByClickOnCaret));

            part1 = new ListPart("Blank", "CheckBox", null, Rectangle.Empty,
                                 ListPartActions.EditByDoubleClick);

            return new[] { part0, part1 };
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual string Name
        {
            get { return "CheckBox"; }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// ドロップダウン
    /// </summary>
    public class ListSubItemDropDown : ListPartParameter, IListSubItem
    {
        private static ListPart[] _Parts = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        static ListSubItemDropDown()
        {
            _Parts = new[] {
                new ListPart( "Text", "DropDown", DefaultPartAction),
            };
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.Name
        {
            get { return "DropDown"; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        IListPart[] IListSubItem.GetParts(ListDrawDescriptor desc)
        {
            return _Parts;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.RepresentativeEditorName(bool immediateEdit)
        {
            return "DropDown";
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        bool IListSubItem.EditByKey
        {
            get { return true; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected static ListPartActions DefaultPartAction
        {
            get
            {
                return (ListPartActions.EditByDoubleClick |
                         ListPartActions.EditByClickOnCaret);
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// ドロップダウン(入力補完機能付き)
    /// </summary>
    public class ListSubItemAssistDropDown : ListPartParameter, IListSubItem
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListSubItemAssistDropDown()
        {
            this.Parts = new[] {
                new ListPart( "Text", "AssistDropDown", DefaultPartAction),
            };
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        IListPart[] IListSubItem.GetParts(ListDrawDescriptor desc)
        {
            return this.Parts;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.RepresentativeEditorName(bool immediateEdit)
        {
            return "AssistDropDown";
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        bool IListSubItem.EditByKey
        {
            get { return true; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.Name
        {
            get { return this.Name; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual string Name
        {
            get { return "AssistDropDown"; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected IListPart[] Parts { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ListPartActions DefaultPartAction
        {
            get
            {
                return (ListPartActions.EditByDoubleClick |
                         ListPartActions.EditByClickOnCaret);
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// ドロップダウンリスト
    /// </summary>
    public class ListSubItemDropDownList : ListPartParameter, IListSubItem
    {
        private ListPart[] _Parts = null;
        private string _Name = null;
        private string _EditorName = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListSubItemDropDownList() : this("DropDownList", "DropDownList")
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListSubItemDropDownList(string name, string editorName)
        {
            _Name = name;
            _EditorName = editorName;
            _Parts = new[] { new ListPart("Text", editorName, DefaultPartAction), };
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.Name
        {
            get { return _Name; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        IListPart[] IListSubItem.GetParts(ListDrawDescriptor desc)
        {
            return _Parts;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.RepresentativeEditorName(bool immediateEdit)
        {
            return _EditorName;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        bool IListSubItem.EditByKey
        {
            get { return true; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected static ListPartActions DefaultPartAction
        {
            get
            {
                return (ListPartActions.EditByDoubleClick |
                         ListPartActions.EditByClickOnCaret);
            }
        }
    }
}
