﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Utilities;

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public interface IListBackgroundDrawer
    {
        void Draw(ListDrawDescriptor desc);
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ListBackgroundDrawer : IListBackgroundDrawer
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        void IListBackgroundDrawer.Draw(ListDrawDescriptor desc)
        {
            DrawInternal(desc);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual void DrawInternal(ListDrawDescriptor desc)
        {
            Brush brush = null;

            if (desc.SubSelected != false)
            {
                brush = desc.SelectedSubItemBackBrush;
            }
            else
            {
                if (desc.Selected != false)
                {
                    brush = desc.SelectedItemBackBrush;
                }
                else
                {
                    brush = desc.ItemBackBrush;
                }
            }

            desc.Gc.FillRectangle(brush, desc.Bounds);
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public interface IListDrawerArgument
    {
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ListTextDrawerArgument : IListDrawerArgument
    {
        public StringAlignment Alignment { get; set; }
        public StringAlignment LineAlignment { get; set; }

        //
        public ListTextDrawerArgument()
        {
            LineAlignment = StringAlignment.Far;
            Alignment = StringAlignment.Near;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 少数用(float、double、etc...)
    /// </summary>
    public class ListDecimalDrawerArgument : IListDrawerArgument
    {
        public int EffectivePlace { get; set; }

        //
        public ListDecimalDrawerArgument()
        {
            EffectivePlace = 0;
        }

        //
        public ListDecimalDrawerArgument(int effectivePlace)
        {
            EffectivePlace = effectivePlace;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public interface IListPartDrawer
    {
        string Name { get; }
        void Draw(ListDrawDescriptor desc);
        int DrawWidth(ListDrawDescriptor desc);
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 無効
    /// </summary>
    public class ListPartDrawer : IListPartDrawer
    {
        /// <summary>
        ///
        /// </summary>
        public string Name
        {
            get { return String.Empty; }
        }

        /// <summary>
        ///
        /// </summary>
        public void Draw(ListDrawDescriptor desc)
        {
            DrawInternal(desc);
        }

        /// <summary>
        ///
        /// </summary>
        public int DrawWidth(ListDrawDescriptor desc)
        {
            //return desc.Bounds.Width + 2;
            return DrawWidthInternal(desc);
        }

        protected virtual int DrawWidthInternal(ListDrawDescriptor desc)
        {
            return desc.Bounds.Width + 2;
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual void DrawInternal(ListDrawDescriptor desc)
        {
            Pen pen = new Pen(desc.Part == null ? Color.Red : Color.Blue, 2);
            int minX = desc.Bounds.Left + 1;
            int minY = desc.Bounds.Top + 1;
            int maxX = desc.Bounds.Right - 2;
            int maxY = desc.Bounds.Bottom - 3;

            desc.Gc.DrawLine(pen, minX, minY, maxX, minY);
            desc.Gc.DrawLine(pen, minX, maxY, maxX, maxY);

            desc.Gc.DrawLine(pen, minX, minY, minX, maxY);
            desc.Gc.DrawLine(pen, maxX, minY, maxX, maxY);

            desc.Gc.DrawLine(pen, minX, minY, maxX, maxY);
            desc.Gc.DrawLine(pen, minX, maxY, maxX, minY);
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ListPartBlankDrawer : IListPartDrawer
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string Name
        {
            get { return "Blank"; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Draw(ListDrawDescriptor desc)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int DrawWidth(ListDrawDescriptor desc)
        {
            return 0;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// テキスト
    /// </summary>
    public class ListPartTextDrawer : IListPartDrawer
    {
        private static StringFormat _Format = new StringFormat();

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        static ListPartTextDrawer()
        {
            _Format.Trimming = StringTrimming.EllipsisCharacter;
            _Format.LineAlignment = StringAlignment.Far;
            _Format.Alignment = StringAlignment.Near;
            _Format.FormatFlags = StringFormatFlags.NoWrap;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string Name
        {
            get { return "Text"; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Draw(ListDrawDescriptor desc)
        {
            ListTextDrawerArgument arg = desc.DrawerArgument as ListTextDrawerArgument;

            if (arg != null)
            {
                _Format.LineAlignment = arg.LineAlignment;
                _Format.Alignment = arg.Alignment;
            }
            else
            {

                _Format.LineAlignment = StringAlignment.Far;
                _Format.Alignment = StringAlignment.Near;
            }

            desc.ContentDrawer.DrawString
                (desc, desc.Parameter.ToString(),
                  desc.Font, desc.TextBrush, desc.Bounds, _Format);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int DrawWidth(ListDrawDescriptor desc)
        {
            string valueString = desc.Parameter.ToString();

            return (int)(desc.Gc.MeasureString(valueString, desc.Font).Width) + 2;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 整数
    /// </summary>
    public class ListPartIntegerDrawer : IListPartDrawer
    {
        private static StringFormat _Format = new StringFormat();

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        static ListPartIntegerDrawer()
        {
            _Format.Trimming = StringTrimming.EllipsisCharacter;
            _Format.LineAlignment = StringAlignment.Far;
            _Format.Alignment = StringAlignment.Far;
            _Format.FormatFlags = StringFormatFlags.NoWrap;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string Name
        {
            get { return "Integer"; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Draw(ListDrawDescriptor desc)
        {
            desc.ContentDrawer.DrawString
                (desc, desc.Parameter.ToString(),
                  desc.Font, desc.TextBrush, desc.Bounds, _Format);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int DrawWidth(ListDrawDescriptor desc)
        {
            string valueString = desc.Parameter.ToString();

            return (int)(desc.Gc.MeasureString(valueString, desc.Font).Width) + 2;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 整数(ULong)
    /// </summary>
    public class ListPartULongDrawer : IListPartDrawer
    {
        private static StringFormat _Format = new StringFormat();

        /// <summary>
        ///
        /// </summary>
        static ListPartULongDrawer()
        {
            _Format.Trimming = StringTrimming.EllipsisCharacter;
            _Format.LineAlignment = StringAlignment.Far;
            _Format.Alignment = StringAlignment.Far;
            _Format.FormatFlags = StringFormatFlags.NoWrap;
        }

        /// <summary>
        ///
        /// </summary>
        public virtual string Name
        {
            get { return "ULong"; }
        }

        /// <summary>
        ///
        /// </summary>
        public virtual void Draw(ListDrawDescriptor desc)
        {
            desc.ContentDrawer.DrawString
                (desc, desc.Parameter.ToString(),
                  desc.Font, desc.TextBrush, desc.Bounds, _Format);
        }

        /// <summary>
        ///
        /// </summary>
        public int DrawWidth(ListDrawDescriptor desc)
        {
            string valueString = desc.Parameter.ToString();
            return (int)(desc.Gc.MeasureString(valueString, desc.Font).Width) + 2;
        }

        /// <summary>
        ///
        /// </summary>
        protected static StringFormat Format
        {
            get
            {
                return _Format;
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 少数
    /// </summary>
    public class ListPartDecimalDrawer : IListPartDrawer
    {
        private static StringFormat _Format = new StringFormat();

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        static ListPartDecimalDrawer()
        {
            _Format.Trimming = StringTrimming.EllipsisCharacter;
            _Format.LineAlignment = StringAlignment.Far;
            _Format.Alignment = StringAlignment.Far;
            _Format.FormatFlags = StringFormatFlags.NoWrap;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string Name
        {
            get { return "Decimal"; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Draw(ListDrawDescriptor desc)
        {
            desc.ContentDrawer.DrawString
                (desc, desc.Parameter.ToString(),
                  desc.Font, desc.TextBrush, desc.Bounds, _Format);
        }


        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int DrawWidth(ListDrawDescriptor desc)
        {
            string valueString = desc.Parameter.ToString();

            return (int)(desc.Gc.MeasureString(valueString, desc.Font).Width) + 2;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 少数(double)
    /// </summary>
    public class ListPartDoubleDrawer : IListPartDrawer
    {
        private static StringFormat _Format = new StringFormat();

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        static ListPartDoubleDrawer()
        {
            _Format.Trimming = StringTrimming.EllipsisCharacter;
            _Format.LineAlignment = StringAlignment.Far;
            _Format.Alignment = StringAlignment.Far;
            _Format.FormatFlags = StringFormatFlags.NoWrap;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string Name
        {
            get { return "Double"; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Draw(ListDrawDescriptor desc)
        {
            ListDecimalDrawerArgument arg = desc.DrawerArgument as ListDecimalDrawerArgument;
            double value = (double)desc.Parameter.Value;
            int effectivePlace = 0;
            string text = null;

            if (arg != null)
            {
                effectivePlace = arg.EffectivePlace;
                text = value.ToString("F" + effectivePlace.ToString());
            }
            else
            {

                text = value.ToString();
            }

            desc.ContentDrawer.DrawString
                //( desc, desc.Parameter.ToString(),
                (desc, text,
                  desc.Font, desc.TextBrush, desc.Bounds, _Format);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int DrawWidth(ListDrawDescriptor desc)
        {
            string valueString = desc.Parameter.ToString();

            return (int)(desc.Gc.MeasureString(valueString, desc.Font).Width) + 2;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 少数(float)
    /// </summary>
    public class ListPartFloatDrawer : IListPartDrawer
    {
        private static StringFormat _Format = new StringFormat();

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        static ListPartFloatDrawer()
        {
            _Format.Trimming = StringTrimming.EllipsisCharacter;
            _Format.LineAlignment = StringAlignment.Far;
            _Format.Alignment = StringAlignment.Far;
            _Format.FormatFlags = StringFormatFlags.NoWrap;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual string Name
        {
            get { return "Float"; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual void Draw(ListDrawDescriptor desc)
        {
            ListDecimalDrawerArgument arg = desc.DrawerArgument as ListDecimalDrawerArgument;
            float value = (float)desc.Parameter.Value;
            int effectivePlace = 0;
            string text = null;

            if (arg != null)
            {
                effectivePlace = arg.EffectivePlace;
                text = value.ToString("F" + effectivePlace.ToString());
            }
            else
            {

                text = value.ToString();
            }

            desc.ContentDrawer.DrawString
                //( desc, desc.Parameter.ToString(),
                (desc, text,
                  desc.Font, desc.TextBrush, desc.Bounds, _Format);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int DrawWidth(ListDrawDescriptor desc)
        {
            string valueString = desc.Parameter.ToString();

            return (int)(desc.Gc.MeasureString(valueString, desc.Font).Width) + 2;
        }

        /// <summary>
        ///
        /// </summary>
        protected static StringFormat Format
        {
            get
            {
                return _Format;
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// サイズ
    /// </summary>
    public class ListPartSizeDrawer : IListPartDrawer
    {
        private static Brush NATextBrush = SystemBrushes.GrayText;
        private static StringFormat _Format = new StringFormat();

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        static ListPartSizeDrawer()
        {
            _Format.Trimming = StringTrimming.EllipsisCharacter;
            _Format.LineAlignment = StringAlignment.Far;
            _Format.Alignment = StringAlignment.Far;
            _Format.FormatFlags = StringFormatFlags.NoWrap;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string Name
        {
            get { return "Size"; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Draw(ListDrawDescriptor desc)
        {
            Brush brush = desc.TextBrush;
            int size = 0;
            string text = null;

            if (desc.Parameter.Value is int)
            {
                size = (int)desc.Parameter.Value;
                if (size >= 0)
                {
                    text = size.ToString();
                }
                else
                {
                    text = NotAvailable.Text;
                    brush = NATextBrush;
                }
            }
            else
            {
                text = desc.Parameter.Value.ToString();
            }

            desc.ContentDrawer.DrawString
                (desc, text, desc.Font, brush, desc.Bounds, _Format);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int DrawWidth(ListDrawDescriptor desc)
        {
            int val;
            string valueString = desc.Parameter.ToString();
            int.TryParse(valueString, out val);

            if (val < 0)
            {
                valueString = NotAvailable.Text;
            }

            return (int)(desc.Gc.MeasureString(valueString, desc.Font).Width) + 2;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// ファイルパス
    /// </summary>
    public class ListPartFilePathDrawer : IListPartDrawer
    {
        private static StringFormat _Format = new StringFormat();

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        static ListPartFilePathDrawer()
        {
            _Format.Trimming = StringTrimming.EllipsisPath;
            _Format.LineAlignment = StringAlignment.Far;
            _Format.Alignment = StringAlignment.Near;
            _Format.FormatFlags = StringFormatFlags.NoWrap;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListPartFilePathDrawer()
        {
            ShowFullPath = true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool ShowFullPath { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string Name
        {
            get { return "FilePath"; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Draw(ListDrawDescriptor desc)
        {
            string filePath = desc.Parameter.ToString();

            if (ShowFullPath == false)
            {
                filePath = Path.GetFileName(filePath);
            }

            desc.ContentDrawer.DrawString
                (desc, filePath, desc.Font, desc.TextBrush, desc.Bounds, _Format);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int DrawWidth(ListDrawDescriptor desc)
        {
            string valueString = desc.Parameter.ToString();

            if (ShowFullPath == false)
            {
                valueString = Path.GetFileName(valueString);
            }

            return (int)(desc.Gc.MeasureString(valueString, desc.Font).Width) + 2;
        }
    }

    /// <summary>
    /// チェックボックス
    /// </summary>
    public class ListPartCheckBoxDrawer : IListPartDrawer
    {
        private ListPartTextDrawer textDrawer = new ListPartTextDrawer();

        /// <summary>
        ///
        /// </summary>
        public virtual string Name
        {
            get { return "CheckBox"; }
        }

        /// <summary>
        ///
        /// </summary>
        public virtual void Draw(ListDrawDescriptor desc)
        {
            BoolParameterValue parameter = desc.Parameter as BoolParameterValue;
            if (parameter == null &&
                desc.Parameter is TextParameterValue)
            {
                this.textDrawer.Draw(desc);
                return;
            }

            ButtonState state = ButtonState.Normal;
            if (parameter != null && (bool)parameter != false)
            {
                state |= ButtonState.Checked;
            }
            state = GetButtonState(desc, state);

            ControlPaint.DrawCheckBox(desc.Gc, desc.Bounds, state);
        }

        /// <summary>
        ///
        /// </summary>
        public int DrawWidth(ListDrawDescriptor desc)
        {
            return desc.Bounds.Width + 2;
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual ButtonState GetButtonState(ListDrawDescriptor desc, ButtonState state)
        {
            return state;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// ボタン
    /// </summary>
    public class ListPartButtonDrawer : IListPartDrawer
    {
        private static StringFormat _Format = new StringFormat();

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        static ListPartButtonDrawer()
        {
            _Format.Trimming = StringTrimming.EllipsisCharacter;
            _Format.LineAlignment = StringAlignment.Far;
            _Format.Alignment = StringAlignment.Far;
            _Format.FormatFlags = StringFormatFlags.NoWrap;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string Name
        {
            get { return "Button"; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Draw(ListDrawDescriptor desc)
        {
            Rectangle bounds = desc.Bounds;
            string text = GetText(desc);

            _Format.Trimming = StringTrimming.EllipsisPath;
            _Format.LineAlignment = StringAlignment.Far;
            _Format.Alignment = StringAlignment.Center;
            _Format.FormatFlags = StringFormatFlags.NoWrap;

            ControlPaint.DrawButton(desc.Gc, bounds, ButtonState.Normal);
            bounds.Inflate(-2, -2);
            desc.Gc.DrawString(text, desc.Font, SystemBrushes.WindowText, bounds, _Format);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int DrawWidth(ListDrawDescriptor desc)
        {
            return desc.Bounds.Width + 2;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected string GetText(ListDrawDescriptor desc)
        {
            object data = null;
            string text = null;

            if (desc.PartParameters.TryGetValue("Text", out data) != false)
            {
                text = data as string;
            }

            return text != null ? text : string.Empty;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ListContentDrawer
    {
        public virtual void DrawString(ListDrawDescriptor desc, string text, Font font, Brush brush, Rectangle bounds, StringFormat format)
        {
            desc.Gc.DrawString(text, font, brush, bounds, format);
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class ListDrawDescriptor
    {
        private Graphics _Gc = null;
        private Font _Font = null;
        private ListImageDictionary _Images = null;

        /// <summary>
        ///
        /// </summary>
        public ListDrawDescriptor(Graphics gc, Font font, ListImageDictionary images)
        {
            _Gc = gc;
            _Font = font;
            _Images = images;
        }

        /// <summary>
        ///
        /// </summary>
        public Graphics Gc
        {
            get { return _Gc; }
        }

        /// <summary>
        ///
        /// </summary>
        public Font Font
        {
            get { return _Font; }
        }

        /// <summary>
        ///
        /// </summary>
        public ListImageDictionary Images
        {
            get { return _Images; }
        }

        /// <summary>
        ///
        /// </summary>
        public Rectangle Bounds { get; set; }

        /// <summary>
        ///
        /// </summary>
        public int Height
        {
            get { return Bounds.Height; }
        }

        /// <summary>
        ///
        /// </summary>
        public IConstParameterValue Parameter { get; set; }

        /// <summary>
        ///
        /// </summary>
        public Dictionary<string, object> PartParameters { get; set; }

        /// <summary>
        ///
        /// </summary>
        public IListDrawerArgument DrawerArgument { get; set; }

        /// <summary>
        ///
        /// </summary>
        public bool Focused { get; set; }
        public bool Enabled { get; set; }
        public bool Selected { get; set; }
        public bool SubSelected { get; set; }
        public bool Caret { get; set; }

        /// <summary>
        ///
        /// </summary>
        public int Counter { get; set; }

        /// <summary>
        ///
        /// </summary>
        public IListPart Part { get; set; }

        /// <summary>
        ///
        /// </summary>
        public int Index { get; set; }
        public string Name { get; set; }
        public IListItem Item { get; set; }

        /// <summary>
        ///
        /// </summary>
        public Brush TextBrush { get; set; }
        public Brush ItemBackBrush { get; set; }
        public Brush SelectedItemBackBrush { get; set; }
        public Brush SelectedSubItemBackBrush { get; set; }
        public Pen ItemBorderPen { get; set; }

        /// <summary>
        ///
        /// </summary>
        public ListContentDrawer ContentDrawer { get; set; }

        /// <summary>
        ///
        /// </summary>
        public ListColumnStyle ColumnStyle { get; set; }
    }
}
