﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

using Win32 = NintendoWare.SoundFoundation.Core.Win32;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    public class DockingForm : FloatingFormT<IDockingPage>
    {
        private IDockingPage _page = null;
        private ContextMenuStrip _ncContextMenu = null;

        // 状態
        private bool _captionRButtonDown = false;

        public DockingForm(Form owner, IDockingPage page)
        {
            Debug.Assert(null != owner);
            Debug.Assert(null != page);

            if (!(page is Control)) { throw new ArgumentException("page"); }

            Owner = owner;
            AdjustToPageSize = false;

            SetPage(page);
        }

        private DockingForm()
        {
        }

        #region ** プロパティ

        public IDockingPage Page
        {
            get { return _page; }
            set { SetPage(value); }
        }

        public ContextMenuStrip NcContextMenuStrip
        {
            get { return _ncContextMenu; }
            set { _ncContextMenu = value; }
        }

        #endregion

        #region ** イベント

        public event EventHandler NcContextMenuShown;
        public event EventHandler CaptionDoubleClick;

        #endregion

        #region ** イベントハンドラ

        protected override void OnVisibleChanged(EventArgs e)
        {
            base.OnVisibleChanged(e);
            if (null == _page) { return; }

            if (Visible)
            {

                if (-1 == _page.FloatingBounds.X && -1 == _page.FloatingBounds.Y)
                {
                    StartPosition = FormStartPosition.WindowsDefaultLocation;
                    Size = _page.FloatingBounds.Size;
                }
                else
                {
                    StartPosition = FormStartPosition.Manual;
                    Bounds = _page.FloatingBounds;
                }

            }
        }

        protected override void OnMove(EventArgs e)
        {
            base.OnMove(e);

            if (!Visible) { return; }

            if (null != _page)
            {
                _page.FloatingBounds = Bounds;
            }
        }

        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);

            if (null != _page)
            {
                _page.FloatingBounds = Bounds;
            }
        }

        protected override void OnFormClosed(FormClosedEventArgs e)
        {
            if (null != _page)
            {
                RemovePage(_page);
            }

            base.OnFormClosed(e);
        }

        protected virtual bool OnNcLButtonDoubleClick(int hitTest)
        {
            if (Win32.HT.HTCAPTION == hitTest)
            {

                if (null != CaptionDoubleClick)
                {
                    CaptionDoubleClick(this, new EventArgs());
                }

            }

            return true;
        }

        protected virtual bool OnContextMenu(Point screenPoint)
        {
            if (!_captionRButtonDown) { return true; }
            ShowNcContextMenu(screenPoint);

            _captionRButtonDown = false;
            return true;
        }

        protected virtual void OnNcRButtonDown(int hitTest, Point screenPoint)
        {
            _captionRButtonDown = (Win32.HT.HTCAPTION == hitTest);
        }

        #endregion

        #region ** メソッド

        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                case Win32.WM.WM_NCLBUTTONDBLCLK:
                    if (OnNcLButtonDoubleClick((int)m.WParam))
                    {
                        return;
                    }
                    break;

                case Win32.WM.WM_CONTEXTMENU:
                    if (OnContextMenu(new Point(Win32.Utility.LOWORD(m.LParam),
                                              Win32.Utility.HIWORD(m.LParam))))
                    {
                        return;
                    }
                    break;

                case Win32.WM.WM_NCRBUTTONDOWN:
                    OnNcRButtonDown((int)m.WParam, new Point(Win32.Utility.LOWORD(m.LParam),
                                                               Win32.Utility.HIWORD(m.LParam)));
                    break;
            }

            base.WndProc(ref m);
        }

        private void SetPage(IDockingPage newPage)
        {
            if (_page == newPage) { return; }

            if (null != _page)
            {
                RemovePage(_page);
            }

            _page = newPage;

            if (null != newPage)
            {
                AddPage(newPage);
                Text = newPage.Text;
            }
            else
            {
                Text = string.Empty;
            }
        }

        private bool ShowNcContextMenu(Point screenPoint)
        {
            if (null == _ncContextMenu) { return false; }

            if (null != NcContextMenuShown)
            {
                NcContextMenuShown(this, new EventArgs());
            }

            _ncContextMenu.Show(screenPoint);
            return true;
        }

        #endregion
    }
}
