﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    #region ** 列挙体

    public enum NumericEditMode
    {
        UpDownOnly = 0,
        TrackBar,
        FloatingBar
    }

    #endregion

    public partial class NumericEdit : NUserControl, IValueSynchronizable
    {
        // 状態
        private bool _initialized = false;
        private bool _showFloatingBar = false;

        // パラメータ
        private NumericEditMode _editMode = NumericEditMode.FloatingBar;
        private HorizontalAlignment _floatingBarAlign = HorizontalAlignment.Center;

        private int _upDownEditWidth = 100;
        private decimal _largeChange = 5;
        private decimal _smallChange = 1;
        private decimal _tickFrequency = 5;
        private TickStyle _tickStyle = TickStyle.BottomRight;

        // コンポーネント		// トラックバー
        private FloatingTrackBar _floatingBar;	// フローティングトラックバー
        private ValueSynchronizable _valueSync;		// 値の同期用コンポーネント


        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="upDown"></param>
        /// <param name="floatingBar"></param>
        /// <param name="trackBar"></param>
        public NumericEdit()
        {
            // コンポーネントの生成
            _valueSync = new ValueSynchronizable(this);
            Debug.Assert(null != _valueSync);

            // コントロールの初期化
            InitializeComponent();

            // 同期設定
            _upDown.ValueSynchronized += _trackBar;
            _trackBar.ValueSynchronized += _upDown;
        }

        #region ** プロパティ

        /// <summary>
        /// NumericEdit の編集モードを取得または設定します。
        /// </summary>
        [Category("Appearance")]
        [DefaultValue(true)]
        public NumericEditMode EditMode
        {
            get { return _editMode; }
            set
            {
                if (value == _editMode) { return; }

                _editMode = value;
                Relayout();
            }
        }

        [Category("Appearance")]
        [DefaultValue(typeof(HorizontalAlignment), "Center")]
        public HorizontalAlignment FloatingBarAlign
        {
            get { return _floatingBarAlign; }
            set { _floatingBarAlign = value; }
        }

        /// <summary>
        /// UpDownコントロールの幅を取得または設定します。
        /// </summary>
        [DefaultValue(100)]
        public int UpDownEditWidth
        {
            get { return _upDownEditWidth; }
            set
            {
                if (value == _upDownEditWidth) { return; }

                _upDownEditWidth = value;
                Relayout();
            }
        }

        /// <summary>
        /// NumericEdit に割り当てる値を取得または設定します。
        /// </summary>
        [Category("Behavior")]
        public virtual decimal Value
        {
            get { return _upDown.Value; }
            set
            {
                _upDown.Value = value;
                _trackBar.Value = value;

                if (null != _floatingBar)
                {
                    _floatingBar.Value = value;
                }
            }
        }

        /// <summary>
        /// この NumericEdit が使用する範囲の上限値を取得または設定します。
        /// </summary>
        [Category("Behavior")]
        [RefreshProperties(RefreshProperties.All)]
        public virtual decimal Maximum
        {
            get { return _upDown.Maximum; }
            set
            {
                _upDown.Maximum = value;
                _trackBar.Maximum = value;

                if (null != _floatingBar)
                {
                    _floatingBar.Maximum = value;
                }
            }
        }

        /// <summary>
        /// この NumericEdit が使用する範囲の下限値を取得または設定します。
        /// </summary>
        [Category("Behavior")]
        [RefreshProperties(RefreshProperties.All)]
        public virtual decimal Minimum
        {
            get { return _upDown.Minimum; }
            set
            {
                _upDown.Minimum = value;
                _trackBar.Minimum = value;

                if (null != _floatingBar)
                {
                    _floatingBar.Minimum = value;
                }
            }
        }

        /// <summary>
        /// 最小の増減値を取得または設定します。
        /// </summary>
        [Category("Behavior")]
        [RefreshProperties(RefreshProperties.All)]
        public virtual decimal Increment
        {
            get { return _upDown.Increment; }
            set
            {
                _upDown.Increment = value;
                _trackBar.Increment = value;

                if (null != _floatingBar)
                {
                    _floatingBar.Increment = value;
                }
            }
        }

        /// <summary>
        /// スクロール ボックスがわずかに移動したときに Value プロパティに対して加算または減算される値を取得または設定します。
        /// </summary>
        [Category("Behavior")]
        [RefreshProperties(RefreshProperties.All)]
        [DefaultValue(1)]
        public virtual decimal SmallChange
        {
            get { return _smallChange; }
            set
            {
                if (value == _smallChange) { return; }

                _smallChange = value;
                _trackBar.SmallChange = value;

                if (null != _floatingBar)
                {
                    _floatingBar.SmallChange = value;
                }
            }
        }

        /// <summary>
        /// スクロール ボックスが大きく移動したときに Value プロパティに対して加算または減算される値を取得または設定します。
        /// </summary>
        [Category("Behavior")]
        [RefreshProperties(RefreshProperties.All)]
        [DefaultValue(5)]
        public virtual decimal LargeChange
        {
            get { return _largeChange; }
            set
            {
                if (value == _largeChange) { return; }

                _largeChange = value;
                _trackBar.LargeChange = value;

                if (null != _floatingBar)
                {
                    _floatingBar.LargeChange = value;
                }
            }
        }

        /// <summary>
        /// 小数部の桁数を取得または設定します。
        /// </summary>
        [Category("Behavior")]
        [RefreshProperties(RefreshProperties.All)]
        [DefaultValue(0)]
        public virtual int DecimalPlaces
        {
            get { return _upDown.DecimalPlaces; }
            set
            {
                _upDown.DecimalPlaces = value;
                _trackBar.DecimalPlaces = value;

                if (null != _floatingBar)
                {
                    _floatingBar.DecimalPlaces = value;
                }
            }
        }

        /// <summary>
        /// 必要に応じて桁区切り記号を表示するかどうかを示す値を取得または設定します。
        /// </summary>
        [Category("Behavior")]
        [RefreshProperties(RefreshProperties.Repaint)]
        [DefaultValue(false)]
        public bool ThousandsSeparator
        {
            get { return _upDown.ThousandsSeparator; }
            set
            {
                _upDown.ThousandsSeparator = value;

                if (null != _floatingBar)
                {
                    _floatingBar.ThousandsSeparator = value;
                }
            }
        }

        /// <summary>
        /// コントロールに描画された目盛り間のデルタを指定する値を取得または設定します。
        /// </summary>
        [Category("Appearance")]
        [RefreshProperties(RefreshProperties.Repaint)]
        [DefaultValue(5)]
        public virtual decimal TickFrequency
        {
            get { return _tickFrequency; }
            set
            {
                if (value == _tickFrequency) { return; }

                _tickFrequency = value;
                _trackBar.TickFrequency = value;

                if (null != _floatingBar)
                {
                    _floatingBar.TickFrequency = value;
                }
            }
        }

        /// <summary>
        /// トラック バー上に目盛りを表示する方法を示す値を取得または設定します。
        /// </summary>
        [Category("Appearance")]
        public TickStyle TickStyle
        {
            get { return _tickStyle; }
            set
            {
                if (value == _tickStyle) { return; }

                _tickStyle = value;
                _trackBar.TickStyle = value;

                if (null != _floatingBar)
                {
                    _floatingBar.TickStyle = value;
                }

                Relayout();
            }
        }

        /// <summary>
        /// 値の同期用コンポーネント
        /// </summary>
        [Browsable(false)]
        public ValueSynchronizable ValueSynchronized
        {
            get { return _valueSync; }
            set { _valueSync = value; }
        }

        private TextBox TextBox
        {
            get { return _upDown.Controls[0] as TextBox; }
        }

        /// <summary>
        /// 初期化フラグ
        /// </summary>
        private bool Initialized
        {
            get { return _initialized; }
        }

        #endregion

        #region ** プロパティのオーバーライド

        /// <summary>
        /// System.Windows.Forms.Control.GetPreferredSize(System.Drawing.Size) が指定できる下限のサイズを取得または設定します。
        /// </summary>
        public override Size MinimumSize
        {
            get
            {
                Size minSize = base.MinimumSize;

                switch (_editMode)
                {
                    case NumericEditMode.UpDownOnly:
                    default:
                        minSize.Width = _upDown.Right;
                        break;

                    case NumericEditMode.TrackBar:
                    case NumericEditMode.FloatingBar:
                        minSize.Width = _floatingButton.Right;
                        break;
                }

                return minSize;
            }
        }

        /// <summary>
        /// コントロールの前景色を取得または設定します。
        /// </summary>
        public override Color ForeColor
        {
            get { return _upDown.ForeColor; }
            set { _upDown.ForeColor = value; }
        }

        #endregion

        #region ** イベント

        public event ValueChangedEventHandler ValueChanged
        {
            add { _valueSync.ValueChanged += value; }
            remove { _valueSync.ValueChanged -= value; }
        }

        #endregion

        #region ** イベントハンドラ

        private void OnFloatingBarClosed(object sender, FormClosedEventArgs e)
        {
            _floatingBar = null;
        }

        private void OnFloatingBarButtonMouseDown(object sender, MouseEventArgs e)
        {
            _showFloatingBar = (null != _floatingBar);
        }

        private void OnFloatingBarButtonClick(object sender, EventArgs e)
        {
            if (_showFloatingBar) { return; }
            ShowFloatingBar();
        }

        private void OnValueChanged(object sender, ValueChangedEventArgs e)
        {
            _valueSync.DispatchValueChangedEvent();
        }

        #endregion

        #region ** イベントハンドラのオーバーライド

        protected override void OnLoad(EventArgs e)
        {
            _initialized = true;

            // コンポーネントの再配置を行う
            // コンストラクタだとTrackBarのVisualStyleが剥げてしまうため、ここで実施する。
            // ※TrackBar.Visibleをtrue→false→trueとすると必ず発生する。
            Relayout();
        }

        protected override void OnSizeChanged(EventArgs e)
        {
            base.OnSizeChanged(e);
            Relayout();
        }

        #endregion

        #region ** メソッド

        private void Relayout()
        {
            if (!Initialized) { return; }

            SuspendLayout();

            // 各コントロールの表示を設定する
            switch (_editMode)
            {
                case NumericEditMode.UpDownOnly:
                    _floatingButton.Hide();
                    _trackBar.Hide();
                    break;

                case NumericEditMode.TrackBar:
                    _floatingButton.Hide();
                    _trackBar.Show();
                    break;

                case NumericEditMode.FloatingBar:
                    _floatingButton.Show();
                    _trackBar.Hide();
                    break;

                default:
                    Debug.Assert(false);
                    break;
            }

            // UpDownコントロールの幅を指定値に設定する
            _upDown.Size = new Size(_upDownEditWidth, _upDown.Height);

            // 各コントロールを上下中央に配置する
            // Floatingボタン、TrackBar のX座標、TrackBar の幅も合わせて設定する
            _upDown.Location = new Point(0, (Height - _upDown.Height) / 2);
            _floatingButton.Location = new Point(_upDown.Right + 3, (Height - _floatingButton.Height) / 2);
            _trackBar.Location = new Point(_upDown.Right + 3, (Height - _trackBar.Height) / 2);
            _trackBar.Size = new Size(Width - _trackBar.Left, Math.Min(Height, _trackBar.VisibleHeight));

            ResumeLayout();
        }

        private void ShowFloatingBar()
        {
            if (null != _floatingBar) { return; }

            _floatingBar = new FloatingTrackBar();
            Debug.Assert(null != _floatingBar);

            // パラメータ設定
            _floatingBar.DecimalPlaces = _upDown.DecimalPlaces;
            _floatingBar.Increment = _upDown.Increment;
            _floatingBar.Maximum = _upDown.Maximum;
            _floatingBar.Minimum = _upDown.Minimum;
            _floatingBar.Value = _upDown.Value;
            _floatingBar.SmallChange = _smallChange;
            _floatingBar.LargeChange = _largeChange;
            _floatingBar.TickFrequency = _tickFrequency;
            _floatingBar.TickStyle = _tickStyle;

            // イベントハンドラ設定
            _floatingBar.FormClosed += OnFloatingBarClosed;

            // 同期設定
            _upDown.ValueSynchronized += _floatingBar;
            _trackBar.ValueSynchronized += _floatingBar;

            // 表示位置を設定する
            Rectangle buttonRect = _floatingButton.Parent.RectangleToScreen(_floatingButton.Bounds);

            switch (_floatingBarAlign)
            {
                case HorizontalAlignment.Center:
                    _floatingBar.Location = new Point(buttonRect.Right - (buttonRect.Width + _floatingBar.Width) / 2,
                                                       buttonRect.Bottom + 2);
                    break;

                case HorizontalAlignment.Left:
                    _floatingBar.Location = new Point(buttonRect.Left, buttonRect.Bottom + 2);
                    break;

                case HorizontalAlignment.Right:
                    _floatingBar.Location = new Point(buttonRect.Right - _floatingBar.Width, buttonRect.Bottom + 2);
                    break;
            }


            // 表示する
            _floatingBar.Show(FindForm());
        }

        private void HideFloatingBar()
        {
            if (null == _floatingBar) { return; }

            // 同期設定を解除する
            _upDown.ValueSynchronized -= _floatingBar;
            _trackBar.ValueSynchronized -= _floatingBar;

            _floatingBar = null;
        }

        #endregion
    }
}
