﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Windows.CommandBars
{
    using System;
    using System.Drawing;

    using NintendoWare.SoundFoundation.Commands;

    /// <summary>
    /// 実行可能なコマンドバーアイテムの情報を管理します。
    /// </summary>
    public class CommandBarExecutableItem : CommandBarItem
    {
        private Command _command;                     // 関連づけられたコマンド

        private string _text = string.Empty;        // 関連付けられたテキスト
        private string _toolTipText = string.Empty; // 関連付けられたテキスト
        private Image _image = null;                // 関連付けられたイメージ

        private bool _textCustomized = false;
        private bool _toolTipTextCustomized = false;
        private bool _imageCustomized = false;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="name">アイテムURI。</param>
        /// <param name="command">関連付けるコマンド。</param>
        public CommandBarExecutableItem(string uri, Command command)
            : base(uri)
        {
            if (null == command) { throw new ArgumentNullException("command"); }
            _command = command;
        }

        /// <summary>
        /// コンストラクタ。
        /// 派生クラスには、Command = null を許可する。
        /// </summary>
        protected CommandBarExecutableItem(string uri) : base(uri) { }

        /// <summary>
        /// ローカライズされたテキストを取得します。
        /// </summary>
        public override string Text
        {
            get
            {
                return (_textCustomized || null == Command) ? _text : Command.Text;
            }
            set
            {
                _text = value;
                _textCustomized = true;
            }
        }

        /// <summary>
        /// ローカライズされたツールチップテキストを取得します。
        /// </summary>
        public virtual string ToolTipText
        {
            get
            {
                return ((_toolTipTextCustomized || null == Command) ?
                        _toolTipText : Command.ToolTipText);
            }
            set
            {
                _toolTipText = value;
                _toolTipTextCustomized = true;
            }
        }

        /// <summary>
        /// ショートカットキーテキストを取得します。
        /// </summary>
        public virtual string ShortcutKeyText
        {
            get
            {
                if (null == Command) { return string.Empty; }
                return Command.ShortcutKeyText;
            }
        }

        /// <summary>
        /// イメージを取得します。
        /// </summary>
        public virtual Image Image
        {
            get
            {
                return (_imageCustomized || null == Command) ? _image : Command.Image;
            }
            set
            {
                _image = value;
                _imageCustomized = true;
            }
        }

        /// <summary>
        /// 関連付けられたコマンドを取得します。
        /// </summary>
        public virtual Command Command
        {
            get { return _command; }
        }
    }
}
