﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.ComponentModel;
    using System.Linq;
    using System.Reflection;
    using System.Text;
    using System.Xml;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.ToolDevelopmentKit;
    using NintendoWare.ToolDevelopmentKit.Reflection;

    public class StaticParameterProvider<TValue> : IParameterProvider
        where TValue : class, INotifyPropertyChanged
    {
        private static IClassAccessor classAccessor = null;

        private TValue value;
        private ParameterDictionary parameterDictionary = new ParameterDictionary();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="value">パラメータの値を指定します。</param>
        public StaticParameterProvider(TValue value)
        {
            this.RebuildParameterDictionary(this.parameterDictionary);
            this.Value = value;
        }

        public event EventHandler ValueChanged;

        /// <summary>
        /// 値を取得または設定します。
        /// </summary>
        public TValue Value
        {
            get { return this.value; }
            set
            {
                if ((object)value == (object)this.value) { return; }

                this.UnregisterEventListener(this.value);
                this.RegisterEventListener(value);

                this.UpdateAllParameterDictionaryValues(value);

                this.value = value;

                OnValueChanged(EventArgs.Empty);
            }
        }

        /// <summary>
        /// パラメータの辞書を取得します。
        /// </summary>
        public IParameterDictionary Parameters
        {
            get { return this.parameterDictionary; }
        }

        /// <summary>
        /// パラメータの値からテキストを取得します。
        /// </summary>
        /// <returns>テキストを返します。</returns>
        public override string ToString()
        {
            StringBuilder stringBuilder = new StringBuilder();

            XmlWriterSettings settings = new XmlWriterSettings()
            {
                ConformanceLevel = ConformanceLevel.Fragment,
                Indent = false,
                OmitXmlDeclaration = true,
            };

            using (XmlWriter writer = XmlWriter.Create(stringBuilder, settings))
            {
                writer.WriteStartElement("Parameters");

                foreach (string key in this.Parameters.Keys)
                {
                    writer.WriteStartElement(key);
                    writer.WriteValue(this.Parameters[key].ToString());
                    writer.WriteEndElement();
                }

                writer.WriteEndElement();
            }

            return stringBuilder.ToString();
        }

        /// <summary>
        /// テキストをパラメータに展開します。
        /// </summary>
        /// <param name="text">テキストを指定します。</param>
        public void Parse(string text)
        {
            Ensure.Argument.NotNull(text);

            XmlDocument document = new XmlDocument();
            document.LoadXml(text);

            if (document.DocumentElement == null ||
                document.DocumentElement.Name != "Parameters")
            {
                return;
            }

            foreach (XmlNode xmlNode in document.DocumentElement.ChildNodes)
            {
                if (!(xmlNode is XmlElement))
                {
                    continue;
                }

                if (!this.Parameters.ContainsKey(xmlNode.Name))
                {
                    continue;
                }

                this.Parameters[xmlNode.Name].Parse(xmlNode.InnerText);
            }
        }

        protected virtual void OnValueChanged(EventArgs e)
        {
            Assertion.Argument.NotNull(e);

            if (this.ValueChanged != null)
            {
                this.ValueChanged(this, e);
            }
        }

        /// <summary>
        /// 現在の値型に対応するクラスアクセサを取得します。
        /// </summary>
        private IClassAccessor GetClassAccessor()
        {
            if (StaticParameterProvider<TValue>.classAccessor == null)
            {
                StaticParameterProvider<TValue>.classAccessor =
                    new AccessorManager(
                        BindingFlags.Public | BindingFlags.Instance |
                        BindingFlags.GetProperty | BindingFlags.SetProperty).GetAccessor(typeof(TValue));
            }

            return StaticParameterProvider<TValue>.classAccessor;
        }

        /// <summary>
        /// 指定したプロパティ名に対応するプロパティアクセサを取得します。
        /// </summary>
        private IPropertyAccessor GetPropertyAccessor(string propertyName)
        {
            Assertion.Argument.NotNull(propertyName);
            return this.GetClassAccessor().GetMembers(propertyName).FirstOrDefault()
                as IPropertyAccessor;
        }

        /// <summary>
        /// パラメータの辞書を再構築します。
        /// </summary>
        /// <param name="parameterDictionary">パラメータの辞書を指定します。</param>
        private void RebuildParameterDictionary(IParameterDictionary parameterDictionary)
        {
            this.parameterDictionary.ParameterValueChanged -= OnParameterDictionaryValueChanged;

            this.parameterDictionary.Clear();

            foreach (IPropertyAccessor accessor in this.GetClassAccessor().Properties)
            {
                ValueRangeAttribute range = accessor.GetAttribute<ValueRangeAttribute>();

                IParameterValue parameterValue = ParameterValueFactory.Create(
                    accessor.ValueType,
                    null,
                    (range == null) ? null : range.Mininum,
                    (range == null) ? null : range.Maximum);

                this.parameterDictionary.AddValue(accessor.Name, parameterValue);
            }

            this.parameterDictionary.ParameterValueChanged += OnParameterDictionaryValueChanged;
        }

        /// <summary>
        /// イベントリスナーを登録します。
        /// </summary>
        /// <param name="value">値を指定します。</param>
        private void RegisterEventListener(TValue value)
        {
            if (value == null) { return; }
            value.PropertyChanged += OnSourceValuePropertyChanged;
        }

        /// <summary>
        /// イベントリスナー登録を解除します。
        /// </summary>
        /// <param name="value">値を指定します。</param>
        private void UnregisterEventListener(TValue value)
        {
            if (value == null) { return; }
            value.PropertyChanged -= OnSourceValuePropertyChanged;
        }

        /// <summary>
        /// ソースに含まれる指定した名前の値を取得します。
        /// </summary>
        /// <param name="srcValue">ソース値を指定します。</param>
        /// <param name="name">パラメータの名前を指定します。</param>
        /// <returns>パラメータの値を返します。</returns>
        private object GetSourceValue(TValue srcValue, string name)
        {
            Assertion.Argument.NotNull(srcValue);
            Assertion.Argument.NotNull(name);

            return this.GetPropertyAccessor(name).GetValue(srcValue);
        }

        /// <summary>
        /// ソースに含まれる指定した名前の値を設定します。
        /// </summary>
        /// <param name="name">パラメータの名前を指定します。</param>
        /// <returns>パラメータの値を返します。</returns>
        private void SetSourceValue(string name, object value)
        {
            Assertion.Argument.NotNull(name);

            this.GetPropertyAccessor(name).SetValue(this.Value, value);
        }

        /// <summary>
        /// パラメータ辞書の値をソースに反映します。
        /// </summary>
        /// <param name="name">パラメータ名を指定します。</param>
        /// <param name="value">新しいパラメータの値を指定します。</param>
        private void UpdateSourceValue(string name, object value)
        {
            Assertion.Argument.NotNull(name);

            this.Value.PropertyChanged -= OnSourceValuePropertyChanged;

            this.SetSourceValue(name, value);

            this.Value.PropertyChanged += OnSourceValuePropertyChanged;
        }

        /// <summary>
        /// すべてのソースの値をパラメータ辞書に反映します。
        /// </summary>
        /// <param name="srcValue">ソース値を指定します。</param>
        private void UpdateAllParameterDictionaryValues(TValue srcValue)
        {
            Assertion.Argument.NotNull(srcValue);

            this.parameterDictionary.ParameterValueChanged -= OnParameterDictionaryValueChanged;

            foreach (string key in this.parameterDictionary.Keys)
            {
                this.parameterDictionary[key].Value = this.GetSourceValue(srcValue, key);
            }

            this.parameterDictionary.ParameterValueChanged += OnParameterDictionaryValueChanged;
        }

        /// <summary>
        /// ソースの値をパラメータ辞書に反映します。
        /// </summary>
        /// <param name="propertyName">パラメータ名を指定します。</param>
        private void UpdateParameterDictionaryValue(string name)
        {
            Assertion.Argument.NotNull(name);

            this.parameterDictionary.ParameterValueChanged -= OnParameterDictionaryValueChanged;

            this.parameterDictionary[name].Value = this.GetSourceValue(this.Value, name);

            this.parameterDictionary.ParameterValueChanged += OnParameterDictionaryValueChanged;
        }

        /// <summary>
        /// ソースの値が変更されると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元を指定します。</param>
        /// <param name="e">イベントパラメータを指定します。</param>
        private void OnSourceValuePropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            this.UpdateParameterDictionaryValue(e.PropertyName);

            this.OnValueChanged(
                new ParameterEventArgs(e.PropertyName, this.parameterDictionary[e.PropertyName]));
        }

        /// <summary>
        /// パラメータ辞書の値が変更されると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元を指定します。</param>
        /// <param name="e">イベントパラメータを指定します。</param>
        private void OnParameterDictionaryValueChanged(object sender, ParameterEventArgs e)
        {
            this.UpdateSourceValue(e.Key, e.ParameterValue.Value);
        }
    }
}
