﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using NintendoWare.SoundFoundation.Operations;

namespace NintendoWare.SoundFoundation.Projects
{
    /// <summary>
    /// コンポーネントを挿入するオペレーションです。
    /// </summary>
    public class InsertComponentOperation : OperationImpl
    {
        private Component _target = null;   // 挿入先のコンポーネント
        private int _insertIndex = 0;      // 挿入先インデックス
        private Component[] _components = null;   // 挿入するコンポーネント

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="target">挿入先のコンポーネント。</param>
        /// <param name="component">挿入するコンポーネント。</param>
        public InsertComponentOperation(Component target, Component component) : this(target, -1, component) { }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="target">挿入先のコンポーネント。</param>
        /// <param name="insertIndex">挿入先インデックス。</param>
        /// <param name="component">挿入するコンポーネント。</param>
        public InsertComponentOperation(Component target, int insertIndex, Component component)
        {
            if (null == target) { throw new ArgumentNullException("target"); }

            if (-1 > insertIndex) { throw new ArgumentException("insertIndex must not be less than -1."); }
            if (target.Children.Count < insertIndex)
            {
                throw new ArgumentException("insertIndex must be less than target.Children.Count.");
            }

            if (null == component) { throw new ArgumentNullException("component"); }
            if (null != component.Parent)
            {
                throw new ArgumentException("component must not have a parent");
            }

            _target = target;
            _insertIndex = (-1 == insertIndex) ? target.Children.Count : insertIndex;
            _components = new Component[] { component };
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="target">挿入先のコンポーネント。</param>
        /// <param name="insertTarget">挿入位置のコンポーネント。</param>
        /// <param name="component">挿入するコンポーネント。</param>
        public InsertComponentOperation(Component target, Component insertTarget, Component component)
            : this(target, (null != insertTarget) ? target.Children.IndexOf(insertTarget) : target.Children.Count, component) { }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="target">挿入先のコンポーネント。</param>
        /// <param name="insertIndex">挿入先インデックス。</param>
        /// <param name="components">挿入するコンポーネントの IEnumerable。</param>
        public InsertComponentOperation(Component target, int insertIndex, IEnumerable<Component> components)
        {
            if (null == target) { throw new ArgumentNullException("target"); }

            if (-1 > insertIndex) { throw new ArgumentException("insertIndex must not be less than -1."); }
            if (target.Children.Count < insertIndex)
            {
                throw new ArgumentException("insertIndex must be less than target.Children.Count.");
            }

            if (null == components) { throw new ArgumentNullException("components"); }

            foreach (Component component in components)
            {
                if (null != component.Parent)
                {
                    throw new ArgumentException("component must not have a parent");
                }
            }

            _target = target;
            _insertIndex = (-1 == insertIndex) ? target.Children.Count : insertIndex;
            _components = components.ToArray();
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="target">挿入先のコンポーネント。</param>
        /// <param name="insertTarget">挿入位置のコンポーネント。</param>
        /// <param name="components">挿入するコンポーネントの IEnumerable。</param>
        public InsertComponentOperation(Component target, Component insertTarget, IEnumerable<Component> components)
            : this(target, (null != insertTarget) ? target.Children.IndexOf(insertTarget) : target.Children.Count, components) { }

        /// <summary>
        /// 挿入先のコンポーネントを取得します。
        /// </summary>
        public Component Target
        {
            get { return _target; }
        }

        /// <summary>
        /// 挿入先インデックスを取得します。
        /// </summary>
        public int InsertIndex
        {
            get { return _insertIndex; }
        }

        /// <summary>
        /// 挿入するコンポーネントの IEnumerator を取得します。
        /// </summary>
        public IEnumerable<Component> Components
        {
            get { return _components; }
        }

        /// <summary>
        /// オペレーションを実行します。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        protected override bool ExecuteInternal()
        {
            int index = _insertIndex;

            _target.Children.BeginCompressEvent();

            foreach (Component component in _components)
            {
                _target.Children.Insert(index, component);
                index++;
            }

            _target.Children.EndCompressEvent();
            return true;
        }

        /// <summary>
        /// オペレーションをロールバックします。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        protected override bool RollbackInternal()
        {
            _target.Children.BeginCompressEvent();

            foreach (Component component in _components)
            {
                component.Parent.Children.Remove(component);
            }

            _target.Children.EndCompressEvent();
            return true;
        }
    }
}
