﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.IO;
    using System.Xml;
    using System.Xml.Serialization;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate;
    using NintendoWare.ToolDevelopmentKit;

    public class SoundSetDocumentReader : IDocumentReader, ISoundFileDocumentReader
    {
        private readonly IObjectFactory<Type, Component> componentFactory;

        public SoundSetDocumentReader(IObjectFactory<Type, Component> componentFactory)
        {
            Ensure.Argument.NotNull(componentFactory);
            this.componentFactory = componentFactory;

            this.DocumentTypeName = Platforms.Any.SoundSetDocument;
            this.DocumentVersion = Platforms.Any.SoundSetDocumentVersion;
            this.Platform = Platforms.Any.PlatformName;
        }

        /// <summary>
        /// 対応するドキュメントの種類を取得します。
        /// </summary>
        public string DocumentTypeName { get; protected set; }

        public string DocumentVersion { get; protected set; }

        public string Platform { get; protected set; }

        public string Version { get; protected set; }

        public virtual string GetFileVersion(string filePath)
        {
            string version = string.Empty;

            if (string.IsNullOrEmpty(filePath) == false && File.Exists(filePath) == true)
            {
                using (FileStream stream = File.Open(filePath, FileMode.Open, FileAccess.Read, FileShare.Read))
                {
                    using (XmlReader reader = XmlReader.Create(stream))
                    {
                        try
                        {
                            version = XmlSoundSetFile.GetFileVersion(reader);
                        }
                        catch
                        {
                        }
                    }
                }
            }

            return version;
        }

        public virtual bool IsSupported(IStreamResource resource)
        {
            if (!(resource is FileResource))
            {
                return false;
            }

            using (FileStream stream =
                    File.Open(resource.Key, FileMode.Open, FileAccess.Read, FileShare.Read))
            {
                using (XmlReader reader = XmlReader.Create(stream))
                {
                    try
                    {
                        XmlSoundSetFile.Validate(reader, this.Platform, this.DocumentVersion);
                    }
                    catch (NotSupportedException)
                    {
                        return false;
                    }
                    catch
                    {
                        throw;
                    }
                }
            }

            return true;
        }

        public virtual void Read(IStreamResource resource, Document document)
        {
            Ensure.Argument.NotNull(resource);
            Ensure.Argument.True(resource is FileResource);
            Ensure.Argument.True(document is SoundSetDocument);

            SoundSetDocument soundSetDocument = document as SoundSetDocument;

            XmlSoundSetFile xmlSoundSetFile = CreateXmlSoundSetFile(resource);

            SoundSetXml2ModelTranslator translator =
                new SoundSetXml2ModelTranslator(this.componentFactory);

            translator.PathResolver.BasePath =
                Path.GetDirectoryName(resource.Key.GetFullPath());

            soundSetDocument.SoundSet = translator.Run(xmlSoundSetFile.Body.SoundSet);
        }

        private XmlSoundSetFile CreateXmlSoundSetFile(IStreamResource resource)
        {
            Assertion.Argument.NotNull(resource);
            Assertion.Argument.True(resource is FileResource);

            using (FileStream fileStream = File.Open(resource.Key, FileMode.Open, FileAccess.Read, FileShare.Read))
            {
                return new XmlSerializer(typeof(XmlSoundSetFile)).Deserialize(fileStream) as XmlSoundSetFile;
            }
        }
    }
}
