﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareReport;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.ToolDevelopmentKit;
    using LegacyFormat = NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat;

    /// <summary>
    /// RVL サウンドプロジェクトドキュメントをサウンドプロジェクトドキュメントに変換します。
    /// </summary>
    public class SoundProjectRvlToSoundProjectDocumentConverter : IDocumentConverter
    {
        private readonly SoundIntermediateOutputTraits intermediateOutputTraits;
        private IObjectFactory<Type, Component> componentFactory;

        public SoundProjectRvlToSoundProjectDocumentConverter(
            SoundIntermediateOutputTraits intermediateOutputTraits)
            : this(intermediateOutputTraits, new SoundProjectComponentFactory())
        {
        }

        public SoundProjectRvlToSoundProjectDocumentConverter(
            SoundIntermediateOutputTraits intermediateOutputTraits,
            IObjectFactory<Type, Component> componentFactory)
        {
            Ensure.Argument.NotNull(intermediateOutputTraits);
            Ensure.Argument.NotNull(componentFactory);

            this.intermediateOutputTraits = intermediateOutputTraits;
            this.componentFactory = componentFactory;
        }

        public string InputDocumentTypeName
        {
            get { return Platforms.Rvl.SoundProjectDocument; }
        }

        public string OutputDocumentTypeName
        {
            get { return this.intermediateOutputTraits.SoundProjectDocumentTypeName; }
        }

        public void Convert(Document inputDocument, Document outputDocument)
        {
            Ensure.Argument.NotNull(inputDocument);
            Ensure.Argument.NotNull(outputDocument);
            this.ConvertInternal(
                inputDocument as SoundProjectRvlDocument,
                outputDocument as SoundProjectDocument);
        }

        private void ConvertInternal(SoundProjectRvlDocument inputDocument, SoundProjectDocument outputDocument)
        {
            Ensure.Argument.NotNull(inputDocument);
            Ensure.Argument.NotNull(outputDocument);

            outputDocument.Project = this.ConvertSoundProject(inputDocument.Project);

            if (inputDocument.Resource is FileResource)
            {
                outputDocument.Resource = new FileResource(
                    Path.ChangeExtension(
                        inputDocument.Resource.Key,
                        this.intermediateOutputTraits.SoundProjectFileExtension
                        )
                    );
            }
        }

        private SoundProject ConvertSoundProject(LegacyFormat.Nw4rSoundMakerProject source)
        {
            Assertion.Argument.NotNull(source);

            SoundProject dest = this.componentFactory.Create(typeof(SoundProject)) as SoundProject;
            Ensure.Operation.ObjectNotNull(dest);

            dest.Name = Path.GetFileNameWithoutExtension(source.FilePath);

            string projectDirectoryPath = Path.GetDirectoryName(source.FilePath);

            // サウンドセットエントリのコンバート
            foreach (LegacyFormat.Nw4rSoundMakerProject.SoundSet sourceSoundSet in source.SoundSets)
            {
                var soundSetFilePath = this.GetSoundSetFilePath(projectDirectoryPath, sourceSoundSet);
                dest.Children.Add(new ProjectSoundSetComponent(sourceSoundSet.Name, soundSetFilePath));
            }

            // プロジェクト設定のコンバート
            this.ConvertProjectConvertSettings(source, dest);
            this.ConvertItemSettings(source, dest);
            this.ConvertHtmlSoundListSettings(source, dest);

            return dest;
        }

        private string GetSoundSetFilePath(
            string projectDirectoryPath,
            LegacyFormat.Nw4rSoundMakerProject.SoundSet soundSet)
        {
            return Path.Combine(
                projectDirectoryPath,
                Path.ChangeExtension(
                    soundSet.FilePath,
                    this.intermediateOutputTraits.SoundSetFileExtension
                    )
                ).GetFullPath();
        }

        //-----------------------------------------------------------------
        // プロジェクト設定のコンバート
        //-----------------------------------------------------------------

        private void ConvertProjectConvertSettings(LegacyFormat.Nw4rSoundMakerProject source, SoundProject dest)
        {
            Assertion.Argument.NotNull(source);
            Assertion.Argument.NotNull(dest);

            // コンバート設定
            dest.OutputDirectoryPath = source.OutputDir;

            // サウンドアーカイブの出力パスが "."と同等なのかを調べます。
            if (FilePathValidator.ValidateRelativePathIsCurrent
                (dest.OutputDirectoryPath) == true)
            {
                dest.OutputDirectoryPath = "output";
            }

            //
            dest.ExternalFileOutputDirectoryPath = source.ExtFileDir;
            dest.IntermediateOutputPath = source.IntermediateOutputDir;
            dest.KeepIntermediateSequenceFile = source.KeepIntermediateRseq;
            dest.SmfConvertTimeBase = source.SmfConvertTimebase;
            dest.DoWarnUnreferencedItems =
                (source.SoundArchiveConvertSettings.WarningFlags &
                LegacyFormat.Model.Nw4rSoundArchiveConvertWarningFlags.WarnUnreferencedItem) != 0;

            // コンバート前後コマンド
            dest.IsPreConvertCommandsEnabled = source.PreConvertCommandEnabled;
            dest.IsPostConvertCommandsEnabled = source.PostConvertCommandEnabled;
            dest.PreConvertCommands = source.PreConvertCommandList;
            dest.PostConvertCommands = source.PostConvertCommandList;

            // サウンドアーカイブプレイヤー設定
            dest.SoundArchivePlayerStreamSoundCount = source.SoundArchivePlayerSettings.StrmSoundCount;
            dest.SoundArchivePlayerStreamChannelCount = source.SoundArchivePlayerSettings.StrmChannelCount;
            dest.SoundArchivePlayerWaveSoundCount = source.SoundArchivePlayerSettings.WaveSoundCount;
            dest.SoundArchivePlayerSequenceSoundCount = source.SoundArchivePlayerSettings.SeqSoundCount;
            dest.SoundArchivePlayerSequenceTrackCount = source.SoundArchivePlayerSettings.SeqTrackCount;
        }

        private void ConvertItemSettings(LegacyFormat.Nw4rSoundMakerProject source, SoundProject dest)
        {
            Assertion.Argument.NotNull(source);
            Assertion.Argument.NotNull(dest);

            // アイテム名のプリフィックス
            dest.ItemNamingHasPrefix = !source.FileNameConvertSettings.WithoutPrefix;
            dest.StreamSoundNamePrefix = source.PrefixStrmSound;
            dest.WaveSoundSetNamePrefix = source.PrefixWaveSoundSet;
            dest.WaveSoundNamePrefix = source.PrefixWaveSound;
            dest.SequenceSoundSetNamePrefix = source.PrefixSeqSoundSet;
            dest.SequenceSoundNamePrefix = source.PrefixSeqSound;
            dest.BankNamePrefix = source.PrefixBank;
            dest.PlayerNamePrefix = source.PrefixPlayer;
            dest.GroupNamePrefix = source.PrefixGroup;
            dest.InstrumentNamePrefix = source.PrefixInst;

            // 命名規則
            dest.ItemNamingCaseChange = this.ConvertCaseChangeType(source.FileNameConvertSettings.ChangeCase);
            dest.ItemNamingInvalidCharChange = this.ConvertInvalidCharChangeType(source.FileNameConvertSettings.ChangeInvalidChar);
        }

        private void ConvertHtmlSoundListSettings(LegacyFormat.Nw4rSoundMakerProject source, SoundProject dest)
        {
            Assertion.Argument.NotNull(source);
            Assertion.Argument.NotNull(dest);

            foreach (LegacyFormat.Nw4rSoundList.OutputSetting sourceSoundListOutput in source.SoundListOutputSettings)
            {
                dest.SoundListOutputSettings.ListOutputs.Add(
                    this.ConvertHtmlSoundListOutput(source, sourceSoundListOutput)
                    );
            }
        }

        //-----------------------------------------------------------------
        // HTMLサウンドリスト設定のコンバート
        //-----------------------------------------------------------------

        private SoundListOutput ConvertHtmlSoundListOutput(
            LegacyFormat.Nw4rSoundMakerProject sourceProject,
            LegacyFormat.Nw4rSoundList.OutputSetting source)
        {
            Assertion.Argument.NotNull(sourceProject);
            Assertion.Argument.NotNull(source);

            SoundListOutput dest = new SoundListOutput()
            {
                Name = source.SettingName,
                Title = source.Title,
                FilePath = PathEx.MakeRelative(
                    source.OutputFilePath,
                    Path.GetDirectoryName(sourceProject.FilePath)
                    ),
                OutputType = source.CombineSoundSet ?
                    SoundListOutputType.EntireSoundProject : SoundListOutputType.EachSoundSet,
                IsIgnoreDisableItems = false,
            };

            SortedList<int, ListOutput>
                sourceListOutputs = new SortedList<int, ListOutput>();

            ListOutput soundListOutput = this.ConvertSoundListColumns(source.SoundListColumnInfo);
            soundListOutput.IsOutput = source.ListOutputInfo.SoundList.OutputFlag;
            sourceListOutputs.Add(source.ListOutputInfo.SoundList.Order, soundListOutput);

            ListOutput bankListOutput = this.ConvertBankListColumns(source.BankListColumnInfo);
            bankListOutput.IsOutput = source.ListOutputInfo.BankList.OutputFlag;
            sourceListOutputs.Add(source.ListOutputInfo.BankList.Order, bankListOutput);

            ListOutput playerListOutput = this.ConvertPlayerListColumns(source.PlayerListColumnInfo);
            playerListOutput.IsOutput = source.ListOutputInfo.PlayerList.OutputFlag;
            sourceListOutputs.Add(source.ListOutputInfo.PlayerList.Order, playerListOutput);

            ListOutput groupListOutput = this.ConvertGroupListColumns(source.GroupListColumnInfo);
            groupListOutput.IsOutput = source.ListOutputInfo.GroupList.OutputFlag;
            sourceListOutputs.Add(source.ListOutputInfo.GroupList.Order, groupListOutput);

            foreach (ListOutput listOutput in sourceListOutputs.Values)
            {
                dest.ListOutputs.Add(listOutput);
            }

            return dest;
        }

        private ListOutput ConvertSoundListColumns(
            LegacyFormat.Nw4rSoundList.OutputSetting.TSoundListColumnInfo source)
        {
            Assertion.Argument.NotNull(source);

            SortedList<int, ListColumnOutput>
                sourceListColumns = new SortedList<int, ListColumnOutput>();

            sourceListColumns.Add(
                source.ActorPlayer.Order,
                this.ConvertColumnOutput(
                    SoundProjectReportTraits.ColumnName.ActorPlayer,
                    source.ActorPlayer)
                );
            sourceListColumns.Add(
                source.Comment.Order,
                this.ConvertColumnOutput(
                    SoundProjectReportTraits.ColumnName.Comment,
                    source.Comment)
                );
            sourceListColumns.Add(
                source.Index.Order,
                this.ConvertColumnOutput(
                    SoundProjectReportTraits.ColumnName.ItemIndex,
                    source.Index)
                );
            sourceListColumns.Add(
                source.Name.Order,
                this.ConvertColumnOutput(
                    SoundProjectReportTraits.ColumnName.Name,
                    source.Name)
                );
            sourceListColumns.Add(
                source.Player.Order,
                this.ConvertColumnOutput(
                    SoundProjectReportTraits.ColumnName.PlayerReference,
                    source.Player)
                );
            sourceListColumns.Add(
                source.PlayerPrio.Order,
                this.ConvertColumnOutput(
                    SoundProjectReportTraits.ColumnName.PlayerPriority,
                    source.PlayerPrio)
                );
            sourceListColumns.Add(
                source.Type.Order,
                this.ConvertColumnOutput(
                    SoundProjectReportTraits.ColumnName.ItemType,
                    source.Type)
                );
            sourceListColumns.Add(
                source.Volume.Order,
                this.ConvertColumnOutput(
                    SoundProjectReportTraits.ColumnName.Volume,
                    source.Volume)
                );

            ListOutput dest = new ListOutput()
            {
                Name = SoundProjectReportExporter.SoundListName,
            };

            foreach (ListColumnOutput columnOutput in sourceListColumns.Values)
            {
                dest.ColumnOutputs.Add(columnOutput);
            }

            return dest;
        }

        private ListOutput ConvertBankListColumns(
            LegacyFormat.Nw4rSoundList.OutputSetting.TBankListColumnInfo source)
        {
            Assertion.Argument.NotNull(source);

            SortedList<int, ListColumnOutput>
                sourceListColumns = new SortedList<int, ListColumnOutput>();

            sourceListColumns.Add(
                source.Comment.Order,
                this.ConvertColumnOutput(
                    SoundProjectReportTraits.ColumnName.Comment,
                    source.Comment)
                );
            sourceListColumns.Add(
                source.Index.Order,
                this.ConvertColumnOutput(
                    SoundProjectReportTraits.ColumnName.ItemIndex,
                    source.Index)
                );
            sourceListColumns.Add(
                source.Name.Order,
                this.ConvertColumnOutput(
                    SoundProjectReportTraits.ColumnName.Name,
                    source.Name)
                );

            ListOutput dest = new ListOutput()
            {
                Name = SoundProjectReportExporter.SoundSetBankListName,
            };

            foreach (ListColumnOutput columnOutput in sourceListColumns.Values)
            {
                dest.ColumnOutputs.Add(columnOutput);
            }

            return dest;
        }

        private ListOutput ConvertPlayerListColumns(
            LegacyFormat.Nw4rSoundList.OutputSetting.TPlayerListColumnInfo source)
        {
            Assertion.Argument.NotNull(source);

            SortedList<int, ListColumnOutput>
                sourceListColumns = new SortedList<int, ListColumnOutput>();

            sourceListColumns.Add(
                source.Comment.Order,
                this.ConvertColumnOutput(
                    SoundProjectReportTraits.ColumnName.Comment,
                    source.Comment)
                );
            sourceListColumns.Add(
                source.Index.Order,
                this.ConvertColumnOutput(
                    SoundProjectReportTraits.ColumnName.ItemIndex,
                    source.Index)
                );
            sourceListColumns.Add(
                source.Name.Order,
                this.ConvertColumnOutput(
                    SoundProjectReportTraits.ColumnName.Name,
                    source.Name)
                );
            sourceListColumns.Add(
                source.SoundLimit.Order,
                this.ConvertColumnOutput(
                    SoundProjectReportTraits.ColumnName.PlayerSoundLimit,
                    source.SoundLimit)
                );

            ListOutput dest = new ListOutput()
            {
                Name = SoundProjectReportExporter.PlayerListName,
            };

            foreach (ListColumnOutput columnOutput in sourceListColumns.Values)
            {
                dest.ColumnOutputs.Add(columnOutput);
            }

            return dest;
        }

        private ListOutput ConvertGroupListColumns(
            LegacyFormat.Nw4rSoundList.OutputSetting.TGroupListColumnInfo source)
        {
            Assertion.Argument.NotNull(source);

            SortedList<int, ListColumnOutput>
                sourceListColumns = new SortedList<int, ListColumnOutput>();

            sourceListColumns.Add(
                source.Comment.Order,
                this.ConvertColumnOutput(
                    SoundProjectReportTraits.ColumnName.Comment,
                    source.Comment)
                );
            sourceListColumns.Add(
                source.Index.Order,
                this.ConvertColumnOutput(
                    SoundProjectReportTraits.ColumnName.ItemIndex,
                    source.Index)
                );
            sourceListColumns.Add(
                source.Name.Order,
                this.ConvertColumnOutput(
                    SoundProjectReportTraits.ColumnName.Name,
                    source.Name)
                );

            ListOutput dest = new ListOutput()
            {
                Name = SoundProjectReportExporter.GroupListName,
            };

            foreach (ListColumnOutput columnOutput in sourceListColumns.Values)
            {
                dest.ColumnOutputs.Add(columnOutput);
            }

            return dest;
        }

        private ListColumnOutput ConvertColumnOutput(
            string sourceName, LegacyFormat.Nw4rSoundList.OutputSetting.TColumnInfo source)
        {
            Assertion.Argument.StringNotEmpty(sourceName);
            Assertion.Argument.NotNull(source);

            return new ListColumnOutput()
            {
                Name = sourceName,
                IsOutput = source.OutputFlag,
            };
        }

        //-----------------------------------------------------------------
        // 列挙値のコンバート
        //-----------------------------------------------------------------

        private CaseChangeType ConvertCaseChangeType(LegacyFormat.Nw4rSoundMakerProject.ChangeCase source)
        {
            switch (source)
            {
                case LegacyFormat.Nw4rSoundMakerProject.ChangeCase.UpperCase:
                    return CaseChangeType.ToUpper;

                case LegacyFormat.Nw4rSoundMakerProject.ChangeCase.LowerCase:
                    return CaseChangeType.ToLower;

                case LegacyFormat.Nw4rSoundMakerProject.ChangeCase.NoChange:
                    return CaseChangeType.None;
            }

            return CaseChangeType.None;
        }

        private InvalidCharChangeType ConvertInvalidCharChangeType(
            LegacyFormat.Nw4rSoundMakerProject.ChangeInvalidChar source)
        {
            switch (source)
            {
                case LegacyFormat.Nw4rSoundMakerProject.ChangeInvalidChar.ReplaceUnderscore:
                    return InvalidCharChangeType.ReplaceToUnderscore;

                case LegacyFormat.Nw4rSoundMakerProject.ChangeInvalidChar.RemoveChar:
                    return InvalidCharChangeType.RemoveChar;
            }

            return InvalidCharChangeType.ReplaceToUnderscore;
        }
    }
}
