﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.IO;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.ToolDevelopmentKit;
    using LegacyFormat = NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat;

    /// <summary>
    /// RVL バンクドキュメントをバンクドキュメントに変換します。
    /// </summary>
    public class BankRvlToBankDocumentConverter : IDocumentConverter
    {
        private readonly string outputDocumentTypeName = string.Empty;
        private readonly string outputFileExtension = string.Empty;
        private IObjectFactory<Type, Component> componentFactory;

        public BankRvlToBankDocumentConverter()
            : this(Platforms.Any.BankDocument, SoundFileExtensions.BankFile, new BankComponentFactory())
        {
        }

        public BankRvlToBankDocumentConverter(string outputDocumentTypeName, string outputFileExtension)
            : this(outputDocumentTypeName, outputFileExtension, new BankComponentFactory())
        {
        }

        public BankRvlToBankDocumentConverter(
            string outputDocumentTypeName,
            string outputFileExtension,
            IObjectFactory<Type, Component> componentFactory)
        {
            Ensure.Argument.StringNotEmpty(outputDocumentTypeName);
            Ensure.Argument.NotNull(componentFactory);

            this.outputDocumentTypeName = outputDocumentTypeName;
            this.outputFileExtension = outputFileExtension;
            this.componentFactory = componentFactory;
        }

        public string InputDocumentTypeName
        {
            get { return Platforms.Rvl.BankDocument; }
        }

        public string OutputDocumentTypeName
        {
            get { return this.outputDocumentTypeName; }
        }

        public void Convert(Document inputDocument, Document outputDocument)
        {
            Ensure.Argument.NotNull(inputDocument);
            Ensure.Argument.NotNull(outputDocument);
            this.ConvertInternal(
                inputDocument as BankRvlDocument,
                outputDocument as BankDocument);
        }

        private void ConvertInternal(BankRvlDocument inputDocument, BankDocument outputDocument)
        {
            Ensure.Argument.NotNull(inputDocument);
            Ensure.Argument.NotNull(outputDocument);

            string bankDictionaryPath = Path.GetDirectoryName(inputDocument.Resource.Key);

            outputDocument.Bank = this.ConvertBank(inputDocument.Bank, bankDictionaryPath);

            if (inputDocument.Resource is FileResource)
            {
                outputDocument.Resource = new FileResource(
                    Path.ChangeExtension(inputDocument.Resource.Key, this.outputFileExtension));
            }
        }

        private Bank ConvertBank(LegacyFormat.Nw4rXmlBank source, string basePath)
        {
            Assertion.Argument.NotNull(source);
            Assertion.Argument.NotNull(basePath);

            Bank dest = this.componentFactory.Create(typeof(Bank)) as Bank;
            Ensure.Operation.ObjectNotNull(dest);

            foreach (LegacyFormat.Nw4rXmlBankInst sourceInstrument in source.InstList)
            {
                dest.Children.Add(this.ConvertInstrument(sourceInstrument, basePath));
            }

            return dest;
        }

        //-----------------------------------------------------------------
        // バンクアイテムのコンバート
        //-----------------------------------------------------------------

        private Instrument ConvertInstrument(LegacyFormat.Nw4rXmlBankInst source, string basePath)
        {
            Assertion.Argument.NotNull(source);
            Assertion.Argument.NotNull(basePath);

            ImportLog.WriteLine("  Instrument: ({0})", source.Label);
            Instrument dest = new Instrument()
            {
                ColorIndex = source.ColorIndex,
                Comment = source.Comment == null ? string.Empty : source.Comment,
                EnvelopeMode = this.ConvertInstrumentEnvelopeMode(source.AdsrEnvelopeSelect),
                IsEnabled = source.Enabled,
                Name = source.Label,
                PitchCents = source.FineTune,
                PitchSemitones = source.CoarseTune,
                ProgramNo = source.PrgNo,
                Volume = source.Volume,
            };

            bool envModeVel = false;
            foreach (LegacyFormat.Nw4rXmlBankInstKeyRegion sourceKeyRegion in source)
            {
                bool mode = false;
                dest.Children.Add(this.ConvertKeyRegion(sourceKeyRegion, basePath, out mode));
                if (mode == true)
                {
                    envModeVel = true;
                }
            }

            if (source.AdsrEnvelope == null)
            {
                if (envModeVel == true && 1 < source.KeyRegionCount)
                {
                    dest.EnvelopeMode = InstrumentEnvelopeMode.VelocityRegion;
                }
                else
                {
                    dest.EnvelopeMode = InstrumentEnvelopeMode.Instrument;
                }
            }
            else
            {
                dest.Envelope = this.ConvertEnvelope(source.AdsrEnvelope);
            }

            return dest;
        }

        private KeyRegion ConvertKeyRegion(
                                           LegacyFormat.Nw4rXmlBankInstKeyRegion source,
                                           string basePath,
                                           out bool envModeVel
                                           )
        {
            Assertion.Argument.NotNull(source);
            Assertion.Argument.NotNull(basePath);

            ImportLog.WriteLine("    KeyRegion: ({0}, {1})", source.RangeMin, source.RangeMax);
            KeyRegion dest = new KeyRegion()
            {
                Comment = source.Comment == null ? string.Empty : source.Comment,
                KeyMax = source.RangeMax,
                KeyMin = source.RangeMin,
            };

            envModeVel = false;
            foreach (LegacyFormat.Nw4rXmlBankInstVelRegion sourceVelocityRegion in source)
            {
                bool mode = false;
                dest.Children.Add(this.ConvertVelocityRegion(
                                                             sourceVelocityRegion,
                                                             basePath,
                                                             out mode,
                                                             source
                                                             ));
                if (mode == true)
                {
                    envModeVel = true;
                }
            }

            return dest;
        }

        private VelocityRegion ConvertVelocityRegion(
                                                     LegacyFormat.Nw4rXmlBankInstVelRegion source,
                                                     string basePath,
                                                     out bool envModeVel,
                                                     LegacyFormat.Nw4rXmlBankInstKeyRegion keySrc
                                                     )
        {
            Assertion.Argument.NotNull(source);
            Assertion.Argument.NotNull(basePath);
            Assertion.Argument.NotNull(keySrc);

            ImportLog.WriteLine(
                                "      VelocityRegion: ({0}, {1}) ({2})",
                                source.RangeMin,
                                source.RangeMax,
                                Path.Combine(basePath, source.FilePath).GetFullPath()
                                );
            VelocityRegion dest = new VelocityRegion()
            {
                Encoding = this.ConvertWaveEncoding(source.FileFormat),
                FilePath = Path.Combine(basePath, source.FilePath).GetFullPath(),
                KeyGroup = source.AlternateAssign,
                InterpolationType = InterpolationType.Polyphase,
                OriginalKey = source.OriginalKey,
                Pan = source.Pan,
                PercussionMode = (source.NoteOffType == LegacyFormat.Nw4rXmlBankNoteOffType.Ignore),
                PitchCents = source.FineTune,
                PitchSemitones = source.CoarseTune,
                VelocityMax = source.RangeMax,
                VelocityMin = source.RangeMin,
                Volume = source.Volume,
            };

            if (source.AdsrEnvelope != null)
            {
                dest.Envelope = this.ConvertEnvelope(source.AdsrEnvelope);
                envModeVel = true;
            }
            else if (keySrc.AdsrEnvelope != null)
            {
                dest.Envelope = this.ConvertEnvelope(keySrc.AdsrEnvelope);
                envModeVel = true;
            }
            else
            {
                dest.Envelope = new Envelope();
                envModeVel = false;
            }

            return dest;
        }

        //-----------------------------------------------------------------
        // パラメータのコンバート
        //-----------------------------------------------------------------

        private Envelope ConvertEnvelope(LegacyFormat.Nw4rAdsrEnvelope source)
        {
            Assertion.Argument.NotNull(source);

            return new Envelope()
            {
                Attack = source.Attack,
                Decay = source.Decay,
                Hold = source.Hold,
                Release = source.Release,
                Sustain = source.Sustain,
            };
        }

        //-----------------------------------------------------------------
        // 列挙値のコンバート
        //-----------------------------------------------------------------

        private InstrumentEnvelopeMode ConvertInstrumentEnvelopeMode(
            LegacyFormat.Nw4rXmlBankAdsrEnvelopeSelect source)
        {
            switch (source)
            {
                case LegacyFormat.Nw4rXmlBankAdsrEnvelopeSelect.Inst:
                    return InstrumentEnvelopeMode.Instrument;

                case LegacyFormat.Nw4rXmlBankAdsrEnvelopeSelect.KeyRegion:
                    return InstrumentEnvelopeMode.KeyRegion;

                case LegacyFormat.Nw4rXmlBankAdsrEnvelopeSelect.VelRegion:
                    return InstrumentEnvelopeMode.VelocityRegion;

                case LegacyFormat.Nw4rXmlBankAdsrEnvelopeSelect.LowerPriority:
                    return InstrumentEnvelopeMode.Lower;
            }

            return InstrumentEnvelopeMode.Instrument;
        }

        private WaveEncoding ConvertWaveEncoding(LegacyFormat.Nw4rXmlBankPcmFileFormat source)
        {
            switch (source)
            {
                case LegacyFormat.Nw4rXmlBankPcmFileFormat.Adpcm:
                    return WaveEncoding.Adpcm;

                case LegacyFormat.Nw4rXmlBankPcmFileFormat.Pcm16:
                    return WaveEncoding.Pcm16;

                case LegacyFormat.Nw4rXmlBankPcmFileFormat.Pcm8:
                    return WaveEncoding.Pcm8;
            }

            return WaveEncoding.Adpcm;
        }
    }
}
