﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using Core;
    using Core.Parameters;

    public class WaveSoundSetBase : SoundSetItem
    {
        private bool suspendParameterValueChangedEventBubbling = false;

        private string savedWaveArchive = String.Empty;
        private WaveArchiveBase targetWaveArchive = null;

        public WaveSoundSetBase()
        {
            this.IsListenQueryComponentDirty = true;
            this.IsListenQuerySoundProjectItemBinding = true;
            this.IsListenResetComponentDirty = true;
            this.IsListenUpdateSoundProjectItemBinding = true;
        }

        /// <summary>
        /// 参照する波形アーカイブを取得します。
        /// </summary>
        public WaveArchiveBase TargetWaveArchive
        {
            get { return this.targetWaveArchive; }
        }

        /// <summary>
        /// 波形アーカイブ参照を取得または設定します。
        /// </summary>
        public string WaveArchiveReference
        {
            get { return GetTextValue(ProjectParameterNames.SoundSetItem.WaveArchiveReference); }
            set { SetTextValue(ProjectParameterNames.SoundSetItem.WaveArchiveReference, value); }
        }

        public bool LinkedWaveArchive
        {
            get
            {
                return (this.targetWaveArchive != null ||
                        this.WaveArchiveReference == WaveArchiveConsts.AutoShared ||
                        this.WaveArchiveReference == WaveArchiveConsts.AutoIndividual);
            }
        }

        /// <summary>
        /// 参照している波形アーカイブかどうか。
        /// </summary>
        public override bool IsReference(Component component)
        {
            if (base.IsReference(component) == true)
            {
                return true;
            }

            if (this.TargetWaveArchive == null) { return false; }
            return this.TargetWaveArchive == component ? true : false;
        }

        /// <summary>
        /// パラメータを初期化します。
        /// </summary>
        protected override void InitializeParameters()
        {
            base.InitializeParameters();

            SetInstanceTextValue(ProjectParameterNames.SoundSetItem.WaveArchiveReference, WaveArchiveConsts.AutoShared);

            // ウェーブサウンドバイナリとサウンドアーカイブバイナリに依存するパラメータ
            this.SetParameterAttributes(
                ProjectParameterNames.SoundSetItem.WaveArchiveReference,
                ParameterAttributes.ComputeHash,
                ParameterAttributes.ComputeSoundArchiveHash);
        }

        protected override void OnParameterValueChanged(ParameterEventArgs e)
        {
            base.OnParameterValueChanged(e);

            if (this.suspendParameterValueChangedEventBubbling)
            {
                return;
            }

            if (e.Key == ProjectParameterNames.SoundSetItem.WaveArchiveReference)
            {
                if (this.targetWaveArchive != null && this.targetWaveArchive.Name != WaveArchiveReference)
                {
                    this.targetWaveArchive = null;
                }

                Router.BubbleEvent(new RoutingEventArgs
                                    (new RequestUpdateSoundProjectItemBindingEvent(), this));
            }
        }

        protected override void OnUpdateSoundProjectItemBinding(ComponentManagerEventArgs e)
        {
            base.OnUpdateSoundProjectItemBinding(e);

            string[] waveArchiveNames = (null == this.targetWaveArchive) ?
                                    new string[] { WaveArchiveReference } : new string[] { WaveArchiveReference, this.targetWaveArchive.Name };

            this.targetWaveArchive = FindComponent<WaveArchiveBase>(e.ComponentManager, waveArchiveNames);

            if (this.targetWaveArchive != null)
            {
                this.suspendParameterValueChangedEventBubbling = true;
                this.WaveArchiveReference = this.targetWaveArchive.Name;
                this.suspendParameterValueChangedEventBubbling = false;
            }
        }

        protected override void OnQuerySoundProjectItemBinding(QuerySoundProjectItemBindingEventArgs e)
        {
            base.OnQuerySoundProjectItemBinding(e);

            if (e.BindingComponents.Contains(this)) { return; }

            if (e.Component == this.targetWaveArchive)
            {
                e.BindingComponents.Add(this);
            }
        }

        protected override void OnQueryComponentDirty(QueryComponentDirtyEventArgs e)
        {
            base.OnQueryComponentDirty(e);

            if (this.savedWaveArchive != WaveArchiveReference)
            {
                e.Result = true;
            }
        }

        protected override void OnResetComponentDirty(RoutingEventArgs e)
        {
            base.OnResetComponentDirty(e);

            this.savedWaveArchive = WaveArchiveReference;
        }
    }
}
