﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// サウンドセットアイテムを表します。
    /// </summary>
    public class SoundSetItem : Component
    {
        private RoutingEventBinding<ComponentManagerEventArgs> updateSoundProjectItemBinding;
        private RoutingEventBinding<QuerySoundProjectItemBindingEventArgs> querySoundProjectItemBinding;
        private RoutingEventBinding<QueryComponentDirtyEventArgs> queryComponentDirty;
        private RoutingEventBinding<RoutingEventArgs> resetComponentDirty;
        private RoutingEventBinding<ComponentManagerRenamedComponentEventArgs> updateRenamedSoundProjectItemBinding;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SoundSetItem()
        {
            updateSoundProjectItemBinding =
                new RoutingEventBinding<ComponentManagerEventArgs>(
                typeof(UpdateSoundProjectItemBindingEvent),
                null, OnUpdateSoundProjectItemBindingInternal);

            updateRenamedSoundProjectItemBinding =
                new RoutingEventBinding<ComponentManagerRenamedComponentEventArgs>(
                typeof(UpdateRenamedSoundProjectItemBindingEvent),
                null, OnUpdateRenamedSoundProjectItemBindingInternal);

            querySoundProjectItemBinding =
                new RoutingEventBinding<QuerySoundProjectItemBindingEventArgs>(
                    typeof(QuerySoundProjectItemBindingEvent),
                    null, OnQuerySoundProjectItemBindingInternal);

            queryComponentDirty =
                new RoutingEventBinding<QueryComponentDirtyEventArgs>(
                    typeof(QueryComponentDirtyEvent),
                    null, OnQueryComponentDirtyInternal);

            resetComponentDirty =
                new RoutingEventBinding<RoutingEventArgs>(
                    typeof(ResetComponentDirtyEvent),
                    null, OnResetComponentDirtyInternal);
        }

        /// <summary>
        /// 所属するサウンドセットを取得します。
        /// </summary>
        public SoundSet SoundSet
        {
            get
            {
                Component parent = Parent;

                while (null != parent)
                {
                    if (parent is SoundSet) { return parent as SoundSet; }
                    parent = parent.Parent;
                }

                return null;
            }
        }

        protected bool IsListenUpdateSoundProjectItemBinding
        {
            get { return IsListenEvent<UpdateSoundProjectItemBindingEvent>(); }
            set { SetEventListening(this.updateSoundProjectItemBinding, value); }
        }

        protected bool IsListenUpdateRenamedSoundProjectItemBinding
        {
            get { return IsListenEvent<UpdateRenamedSoundProjectItemBindingEvent>(); }
            set { SetEventListening(this.updateRenamedSoundProjectItemBinding, value); }
        }

        protected bool IsListenQuerySoundProjectItemBinding
        {
            get { return IsListenEvent<QuerySoundProjectItemBindingEvent>(); }
            set { SetEventListening(this.querySoundProjectItemBinding, value); }
        }

        protected bool IsListenQueryComponentDirty
        {
            get { return IsListenEvent<QueryComponentDirtyEvent>(); }
            set { SetEventListening(this.queryComponentDirty, value); }
        }

        protected bool IsListenResetComponentDirty
        {
            get { return IsListenEvent<ResetComponentDirtyEvent>(); }
            set { SetEventListening(this.resetComponentDirty, value); }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual bool IsReference(Component component)
        {
            return false;
        }

        protected virtual void OnUpdateSoundProjectItemBinding(ComponentManagerEventArgs e)
        {
        }

        protected virtual void OnUpdateRenamedSoundProjectItemBinding(ComponentManagerRenamedComponentEventArgs e)
        {
        }

        protected virtual void OnQuerySoundProjectItemBinding(QuerySoundProjectItemBindingEventArgs e)
        {
        }

        protected virtual void OnQueryComponentDirty(QueryComponentDirtyEventArgs e)
        {
        }

        protected virtual void OnResetComponentDirty(RoutingEventArgs e)
        {
        }

        private bool IsListenEvent<TEvent>()
            where TEvent : RoutingEvent
        {
            return this.Router.Bindings.ContainsKey(typeof(TEvent));
        }

        private void SetEventListening(RoutingEventBinding binding, bool isListen)
        {
            Assertion.Argument.NotNull(binding);

            if (isListen == this.Router.Bindings.ContainsKey(binding.EventType)) { return; }

            if (isListen)
            {
                this.Router.Bindings.Add(binding);
            }
            else
            {
                this.Router.Bindings.Remove(binding);
            }
        }

        private void OnUpdateSoundProjectItemBindingInternal(object sender, ComponentManagerEventArgs e)
        {
            OnUpdateSoundProjectItemBinding(e);
        }

        private void OnUpdateRenamedSoundProjectItemBindingInternal(object sender, ComponentManagerRenamedComponentEventArgs e)
        {
            OnUpdateRenamedSoundProjectItemBinding(e);
        }

        private void OnQuerySoundProjectItemBindingInternal(object sender, QuerySoundProjectItemBindingEventArgs e)
        {
            OnQuerySoundProjectItemBinding(e);
        }

        private void OnQueryComponentDirtyInternal(object sender, QueryComponentDirtyEventArgs e)
        {
            OnQueryComponentDirty(e);
        }

        private void OnResetComponentDirtyInternal(object sender, RoutingEventArgs e)
        {
            OnResetComponentDirty(e);
        }
    }
}
