﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Parameters;

    public class GroupItemBase : Component
    {
        ///
        private string _StandardTargetName = String.Empty;

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public GroupItemBase()
        {
            Router.Bindings.Add(
                new RoutingEventBinding<ComponentManagerEventArgs>(
                    typeof(UpdateSoundProjectItemBindingEvent),
                    null, OnUpdateSoundProjectItemBinding));

            Router.Bindings.Add(
                new RoutingEventBinding<QuerySoundProjectItemBindingEventArgs>(
                    typeof(QuerySoundProjectItemBindingEvent),
                    null, OnQuerySoundProjectItemBinding));

            Router.Bindings.Add(
                new RoutingEventBinding<QueryComponentDirtyEventArgs>(
                    typeof(QueryComponentDirtyEvent),
                    null, OnQueryComponentDirty));

            Router.Bindings.Add(
                new RoutingEventBinding<RoutingEventArgs>(
                    typeof(ResetComponentDirtyEvent),
                    null, OnResetComponentDirty));

            Router.Bindings.Add(
                new RoutingEventBinding<ComponentManagerRenamedComponentEventArgs>(
                    typeof(UpdateRenamedSoundProjectItemBindingEvent),
                    null, OnUpdateRenamedSoundProjectItemBinding));

            //
            TargetSoundSet = String.Empty;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string TargetSoundSet { get; set; }

        ///--------------------------------
        /// <summary>
        /// ターゲット
        /// </summary>
        public SoundSetItem Target { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string TargetItemReference
        {
            get { return GetTextValue(ProjectParameterNames.TargetItemReference); }
            set { SetTextValue(ProjectParameterNames.TargetItemReference, value); }
        }

        ///--------------------------------
        /// <summary>
        /// 関連アイテム登録設定を取得または設定します。
        /// </summary>
        public GroupItemRegisterType RegisterType
        {
            get { return GetValue<GroupItemRegisterType>(ProjectParameterNames.GroupItem.RegisterType); }
            set { SetValue<GroupItemRegisterType>(ProjectParameterNames.GroupItem.RegisterType, value); }
        }

        ///--------------------------------
        /// <summary>
        /// パラメータを初期化します。
        /// </summary>
        protected override void InitializeParameters()
        {
            base.InitializeParameters();

            SetInstanceTextValue(ProjectParameterNames.TargetItemReference, string.Empty);
            SetInstanceTextValue("GroupItemTargetSoundSet", string.Empty);

            this.Parameters.AddValue(
                                     ProjectParameterNames.GroupItem.RegisterType,
                                     new GroupItemRegisterTypeParameterValue()
                                     );

            // グループバイナリ依存パラメータ
            this.SetParameterAttributes(ProjectParameterNames.TargetItemReference, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.GroupItem.RegisterType, ParameterAttributes.ComputeHash);

            ((ParameterDictionary)Parameters).ParameterValueChanged += OnParameterValueChanged;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool SuspendSendEvent { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnParameterValueChanged(object sender, ParameterEventArgs e)
        {
            if (SuspendSendEvent != false)
            {
                return;
            }

            if (e.Key == ProjectParameterNames.TargetItemReference)
            {
                Target = null;
                Router.BubbleEvent(new RoutingEventArgs
                                    (new RequestUpdateSoundProjectItemBindingEvent(), this));
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnUpdateSoundProjectItemBinding(object sender, ComponentManagerEventArgs e)
        {
            this.UpdateTargetItemReference(e.ComponentManager);
        }

        private void OnQuerySoundProjectItemBinding(object sender, QuerySoundProjectItemBindingEventArgs e)
        {
            if (e.BindingComponents.Contains(this)) { return; }

            if (e.Component.Name != this.TargetItemReference) { return; }
            if (e.Component != this.Target) { return; }

            e.BindingComponents.Add(this);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnQueryComponentDirty(object sender, QueryComponentDirtyEventArgs e)
        {
            if (_StandardTargetName != TargetItemReference)
            {
                e.Result = true;
            }

        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnResetComponentDirty(object sender, RoutingEventArgs e)
        {
            _StandardTargetName = TargetItemReference;
        }

        /// <summary>
        /// アイテムの名前が変更された時に呼ばれます。
        /// </summary>
        private void OnUpdateRenamedSoundProjectItemBinding(object sender, ComponentManagerRenamedComponentEventArgs e)
        {
            this.UpdateTargetItemReference(e.ComponentManager);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateTargetItemReference(ComponentManager componentManager)
        {
            string[] itemNames = (null == Target) ?
                new string[] { TargetItemReference } : new string[] { Target.Name, TargetItemReference };
            this.Target = FindComponent<SoundSetItem>(componentManager, itemNames);

            if (null != this.Target)
            {
                SuspendSendEvent = true;
                TargetItemReference = Target.Name;
                TargetSoundSet = Target.SoundSet.Name;
                SuspendSendEvent = false;
            }
        }
    }
}
