﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.Linq;
using NintendoWare.SoundFoundation.Core;
using NintendoWare.SoundFoundation.Core.Collections;

namespace NintendoWare.SoundFoundation.Projects
{
    /// <summary>
    /// Component のコレクションです。
    /// </summary>
    public class ComponentList : TreeObjectCollectionBase<Component>, ICollection<Component>, IReadOnlyEventRoutableCollection
    {
        private InnerCollectionImpl _components = new InnerCollectionImpl();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="owner">コレクションの所有者。</param>
        public ComponentList(Component owner) : base(owner) { }

        /// <summary>
        /// 内部コレクションを取得します。
        /// </summary>
        protected override IList<Component> InnerCollection
        {
            get { return _components; }
        }

#if false
        /// <summary>
        /// コレクション内でアイテムを移動します。
        /// </summary>
        /// <param name="item">移動するアイテム。</param>
        /// <param name="targetItem">移動先のアイテム。</param>
        public void Move( Component item, Component targetItem )
        {
            if ( null == item ) { throw new ArgumentNullException( "item" ); }
            if ( null == targetItem ) { throw new ArgumentNullException( "targetItem" ); }

            if ( null == targetItem.Parent || this == targetItem.Parent.Children ) {
                throw new ArgumentException( "targetItem's parant is invalid." );
            }

            MoveInternal( IndexOf( item ), IndexOf( targetItem ), true );
        }

        /// <summary>
        /// コレクション内でアイテムを移動します。
        /// </summary>
        /// <param name="index">移動するアイテムのインデックス。</param>
        /// <param name="targetItem">移動先のアイテム。</param>
        public void Move( int index, Component targetItem )
        {
            if ( null == targetItem ) { throw new ArgumentNullException( "targetItem" ); }

            if ( null == targetItem.Parent || this == targetItem.Parent.Children ) {
                throw new ArgumentException( "targetItem's parant is invalid." );
            }

            MoveInternal( index, IndexOf( targetItem ), true );
        }

        /// <summary>
        /// コレクション内でアイテムを移動します。
        /// </summary>
        /// <param name="index">移動するアイテムのインデックス。</param>
        /// <param name="targetIndex">移動先のインデックス。</param>
        public void Move( int index, int targetIndex )
        {
            MoveInternal( index, targetIndex, true );
        }
        /// <summary>
        /// コレクション内でアイテムを移動します。
        /// </summary>
        /// <param name="items">移動するアイテム。</param>
        /// <param name="targetItem">移動先のアイテム。</param>
        public void Move( IEnumerable<Component> items, Component targetItem )
        {
            if ( null == items ) { throw new ArgumentNullException( "items" ); }
            if ( null == targetItem ) { throw new ArgumentNullException( "targetItem" ); }

            if ( null == targetItem.Parent || this == targetItem.Parent.Children ) {
                throw new ArgumentException( "targetItem's parant is invalid." );
            }

            Move( items, IndexOf( targetItem ) );
        }
#endif

        /// <summary>
        /// コレクション内でアイテムを移動します。
        /// </summary>
        /// <param name="item">移動するアイテム。</param>
        /// <param name="targetIndex">移動先のインデックス。</param>
        public void Move(Component item, int targetIndex)
        {
            if (null == item) { throw new ArgumentNullException("item"); }

            if (null == item.Parent || this != item.Parent.Children)
            {
                throw new ArgumentException("item's parant is invalid.");
            }

            MoveInternal(IndexOf(item), targetIndex, true);
        }

        /// <summary>
        /// コレクション内でアイテムを移動します。
        /// </summary>
        /// <param name="items">移動するアイテム。</param>
        /// <param name="targetIndex">移動先のインデックス。</param>
        public void Move(IEnumerable<Component> items, int targetIndex)
        {
            if (null == items) { throw new ArgumentNullException("items"); }

            int currentTargetIndex = targetIndex;

            foreach (Component item in items)
            {
                if (null == item.Parent || this != item.Parent.Children)
                {
                    throw new ArgumentException("item's parant is invalid.");
                }

                int index = IndexOf(item);
                if (index == currentTargetIndex) { continue; }

                MoveInternal(index, currentTargetIndex, true);

                if (index < currentTargetIndex)
                {
                    currentTargetIndex++;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void Sort<TKey>(Func<Component, TKey> keySelector)
        {
            if (null == keySelector) { throw new ArgumentNullException("keySelector"); }

            InnerCollectionImpl components = new InnerCollectionImpl();

            foreach (Component component in _components.OrderBy<Component, TKey>(keySelector))
            {
                components.Add(component);
            }

            _components = components;
        }

        /// <summary>
        /// 指定アイテムがコレクションに含まれているかどうかを調べます。
        /// </summary>
        /// <param name="item">アイテム。</param>
        /// <returns>含まれている場合は true、含まれていない場合は false。</returns>
        bool IReadOnlyEventRoutableCollection.Contains(IEventRoutable item)
        {
            return Contains(item as Component);
        }

        /// <summary>
        /// アイテムの列挙子を取得します。
        /// </summary>
        /// <returns>アイテムの列挙子</returns>
        IEnumerator<IEventRoutable> IEnumerable<IEventRoutable>.GetEnumerator()
        {
            return InnerCollection.Cast<IEventRoutable>().GetEnumerator();
        }

        /// <summary>
        /// コレクション内でアイテムを移動します。
        /// </summary>
        /// <param name="index">移動するアイテムのインデックス。</param>
        /// <param name="targetIndex">移動先のインデックス。</param>
        /// <param name="raiseEvent">イベントを発行する場合は true、発行しない場合は false。</param>
        private void MoveInternal(int index, int targetIndex, bool raiseEvent)
        {
            if (InnerCollection.Count <= index)
            {
                throw new ArgumentException("index must be less than Count.");
            }
            if (InnerCollection.Count <= targetIndex)
            {
                throw new ArgumentException("targetIndex must be less than Count.");
            }
            if (index == targetIndex) { return; }

            Component item = this[index];

            InnerCollection.Remove(item);
            InnerCollection.Insert(targetIndex, item);

            if (!raiseEvent) { return; }

            OnCollectionChanged(new NotifyCollectionChangedEventArgs
                                 (NotifyCollectionChangedAction.Move, item, targetIndex, index));
        }

        /// <summary>
        /// コレクション内部実装クラス
        /// </summary>
        private class InnerCollectionImpl : KeyedCollection<Component, Component>
        {
            /// <summary>
            /// アイテムからキーを取得します。
            /// </summary>
            /// <param name="item">アイテム。</param>
            /// <returns>アイテムのキー。</returns>
            protected override Component GetKeyForItem(Component item)
            {
                return item;
            }
        }
    }
}
