﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using NintendoWare.SoundFoundation.Core.Parameters;
using NintendoWare.SoundFoundation.Operations;
using NintendoWare.SoundFoundation.Projects;

namespace NintendoWare.SoundFoundation.Parameters
{
    /// <summary>
    /// パラメータを挿入するオペレーションです。
    /// </summary>
    public class InsertParameterOperation : OperationImpl
    {
        private IList<ComponentReference> target = null; // 挿入先のパラメータコレクション
        private int insertIndex = 0; // 挿入先インデックス
        private IParameterProvider[] providers = null; // 挿入するパラメータ

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="target">挿入先のパラメータコレクション。</param>
        /// <param name="provider">挿入するパラメータ。</param>
        public InsertParameterOperation(
                                        IList<ComponentReference> target,
                                        ComponentReference provider
                                        )
            : this(target, -1, provider)
        {
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="target">挿入先のパラメータコレクション。</param>
        /// <param name="insertIndex">挿入先インデックス。</param>
        /// <param name="provider">挿入するパラメータ。</param>
        public InsertParameterOperation(
                                        IList<ComponentReference> target,
                                        int insertIndex,
                                        ComponentReference provider
                                        )
        {
            if (null == target) { throw new ArgumentNullException("target"); }

            if (-1 > insertIndex)
            {
                throw new ArgumentException("insertIndex must not be less than -1.");
            }
            if (target.Count() < insertIndex)
            {
                throw new ArgumentException(
                                            "insertIndex must be less than target.Count."
                                            );
            }

            if (null == provider) { throw new ArgumentNullException("provider"); }

            this.target = target;
            this.insertIndex = (-1 == insertIndex) ? target.Count() : insertIndex;
            this.providers = new IParameterProvider[] { provider };
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="target">挿入先のパラメータコレクション。</param>
        /// <param name="insertTarget">挿入位置のパラメータ。</param>
        /// <param name="provider">挿入するパラメータ。</param>
        public InsertParameterOperation(
                                        IList<ComponentReference> target,
                                        ComponentReference insertTarget,
                                        ComponentReference provider
                                        )
            : this(
                   target,
                   ((null != insertTarget) ?
                    target.IndexOf(insertTarget) :
                    target.Count()),
                   provider
                   )
        {
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="target">挿入先のパラメータコレクション。</param>
        /// <param name="insertIndex">挿入先インデックス。</param>
        /// <param name="providers">挿入するパラメータの IEnumerable。</param>
        public InsertParameterOperation(
                                        IList<ComponentReference> target,
                                        int insertIndex,
                                        IEnumerable<ComponentReference> providers
                                        )
        {
            if (null == target) { throw new ArgumentNullException("target"); }

            if (-1 > insertIndex)
            {
                throw new ArgumentException("insertIndex must not be less than -1.");
            }
            if (target.Count() < insertIndex)
            {
                throw new ArgumentException(
                                            "insertIndex must be less than target.Count."
                                            );
            }

            if (null == providers) { throw new ArgumentNullException("providers"); }

            this.target = target;
            this.insertIndex = (-1 == insertIndex) ? target.Count() : insertIndex;
            this.providers = providers.ToArray();
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="target">挿入先のパラメータコレクション。</param>
        /// <param name="insertTarget">挿入位置のパラメータ。</param>
        /// <param name="providers">挿入するパラメータの IEnumerable。</param>
        public InsertParameterOperation(
                                        IList<ComponentReference> target,
                                        ComponentReference insertTarget,
                                        IEnumerable<ComponentReference> providers
                                        )
            : this(
                   target,
                   ((null != insertTarget) ?
                    target.IndexOf(insertTarget) :
                    target.Count()),
                   providers
                   )
        {
        }

        /// <summary>
        /// 挿入先のパラメータコレクションを取得します。
        /// </summary>
        public IList<ComponentReference> Target
        {
            get { return this.target; }
        }

        /// <summary>
        /// 挿入先インデックスを取得します。
        /// </summary>
        public int InsertIndex
        {
            get { return this.insertIndex; }
        }

        /// <summary>
        /// 挿入するパラメータの IEnumerator を取得します。
        /// </summary>
        public IEnumerable<IParameterProvider> Providers
        {
            get { return this.providers; }
        }

        /// <summary>
        /// オペレーションを実行します。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        protected override bool ExecuteInternal()
        {
            int index = this.insertIndex;

            foreach (ComponentReference provider in this.providers)
            {
                this.target.Insert(index, provider);
                index++;
            }

            return true;
        }

        /// <summary>
        /// オペレーションをロールバックします。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        protected override bool RollbackInternal()
        {
            foreach (ComponentReference provider in this.providers)
            {
                this.target.Remove(provider);
            }

            return true;
        }
    }
}
