﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.IO;
    using Core;
    using Core.Parameters;
    using Projects;
    using ToolDevelopmentKit;
    using ToolDevelopmentKit.Conversion;

    public abstract class Xml2ModelTranslator
    {
        private Converter converter = new Converter();
        private PathResolver pathResolver = new PathResolver();

        private Xml2ModelTranslationContext context;

        private IObjectFactory<Type, Component> componentFactory;
        private CommonItemParamTranslator itemParamTranslator;

        protected Xml2ModelTranslator(
            IObjectFactory<Type, Component> objectFactory,
            CommonItemParamTranslator itemParamTranslator)
        {
            Ensure.Argument.NotNull(objectFactory);
            Ensure.Argument.NotNull(itemParamTranslator);

            this.componentFactory = objectFactory;
            this.itemParamTranslator = itemParamTranslator;
            this.itemParamTranslator.PathResolver = pathResolver;
        }

        public PathResolver PathResolver
        {
            get { return this.pathResolver; }
        }

        protected TDest Run<TDest>(object value)
            where TDest : class
        {
            this.context = new Xml2ModelTranslationContext();

            return this.converter.Convert(value, typeof(TDest)) as TDest;
        }

        protected Xml2ModelTranslationContext Context
        {
            get { return this.context; }
        }

        protected void AddConverter<TSrc, TDest>()
            where TSrc : XmlComponent
            where TDest : Component
        {
            this.converter.TypeConverters.Add(
                new TypeCouple(typeof(TSrc), typeof(TDest)),
                this.XmlComponentToComponent<TSrc, TDest>);
        }

        protected void AddConverter<TSrc, TDest>(ConvertHandler handler)
        {
            Assertion.Argument.NotNull(handler);
            this.converter.TypeConverters.Add(new TypeCouple(typeof(TSrc), typeof(TDest)), handler);
        }

        protected void AddDefaultConverter<TSrc, TDest>()
        {
            this.converter.TypeConverters.Add(
                new TypeCouple(typeof(TSrc), typeof(TDest)), DefaultConverters.ConvertClass);
        }

        protected Component TranslateComponent<TXmlComponent, TComponent>(
            TXmlComponent src, Type dstType, Converter converter)
            where TXmlComponent : XmlComponent
            where TComponent : Component
        {
            Ensure.Argument.NotNull(src);
            Ensure.Argument.NotNull(converter);

            TComponent dest = this.componentFactory.Create(typeof(TComponent)) as TComponent;

            if (src.Name != null)
            {
                dest.Name = src.Name;
            }

            if (src is IXmlParameterProvider)
            {
                this.TranslateParameters(src as IXmlParameterProvider, dest, converter);
            }

            if (src is IXmlComponentParent)
            {
                this.TranslateChildren(src as IXmlComponentParent, dest, converter);
            }

            return dest;
        }

        protected void TranslateParameters(
            IXmlParameterProvider src, IParameterProvider dest, Converter converter)
        {
            Assertion.Argument.NotNull(src);
            Assertion.Argument.NotNull(dest);
            Assertion.Argument.NotNull(converter);

            foreach (XmlParameter srcParameter in src.Parameters)
            {
                this.itemParamTranslator.Xml2Model(srcParameter, dest);
            }
        }

        protected void TranslateChildren(IXmlComponentParent src, Component dest, Converter converter)
        {
            TranslateChildren(src, dest, null, converter);
        }

        protected void TranslateChildren(
            IXmlComponentParent src, Component dest, Type destType, Converter converter)
        {
            Assertion.Argument.NotNull(src);
            Assertion.Argument.NotNull(dest);
            Assertion.Argument.NotNull(converter);

            foreach (XmlComponent item in src.Items)
            {
                Component destItem = null;

                if (destType == null)
                {
                    destItem = converter.Convert(item, destType) as Component;
                }
                else
                {
                    TypeCouple typeCouple = new TypeCouple(item.GetType(), destType);

                    if (converter.TypeConverters.ContainsKey(typeCouple))
                    {
                        destItem = converter.TypeConverters[typeCouple](item, destType, converter) as Component;
                    }
                }

                if (destItem == null)
                {
                    throw new Exception("invalid type");
                }

                dest.Children.Add(destItem);
            }
        }

        /// <summary>
        /// XmlComponent から Component に変換します。
        /// </summary>
        /// <typeparam name="TXmlComponent">変換元の型を指定します。</typeparam>
        /// <typeparam name="TComponent">変換後の型を指定します。</typeparam>
        /// <param name="src">変換元のオブジェクトを指定します。</param>
        /// <param name="dstType">変換後の型を指定します。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>変換後のオブジェクトを返します。</returns>
        private object XmlComponentToComponent<TXmlComponent, TComponent>(
            object src, Type dstType, Converter converter)
            where TXmlComponent : XmlComponent
            where TComponent : Component
        {
            return this.TranslateComponent<TXmlComponent, TComponent>(
                src as TXmlComponent, dstType, converter);
        }
    }
}
