﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate
{
    using NintendoWare.SoundFoundation.Documents;
    using Projects;
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using ToolDevelopmentKit;
    using ToolDevelopmentKit.Conversion;

    /// <summary>
    ///
    /// </summary>
    public static class SoundProjectParameterTranslateFilter
    {
        private static HashSet<string> excludeNames = new HashSet<string>();

        public static void Add(string name)
        {
            excludeNames.Add(name);
        }

        internal static string[] Filter(string[] names)
        {
            return names
                .Where(n => excludeNames.Contains(n) == false)
                .ToArray();
        }
    }

    /// <summary>
    ///
    /// </summary>
    public static class SoundProjectParameterTranslateSndEdit
    {
        public static string[] SndEditParamNames { get; set; }
    }

    public static class SoundProjectTranslatorSettings
    {
        static SoundProjectTranslatorSettings()
        {
            EnabledSoundArchiveOutputType = false;
        }

        public static bool EnabledSoundArchiveOutputType { get; set; }
    }

    internal class SoundProjectModel2XmlTranslator : Model2XmlTranslator
    {
        Dictionary<SoundSet, string> _dictionary = null;

        public SoundProjectModel2XmlTranslator(Document[] documents)
            : base(new SoundProjectItemParamTranslator())
        {
            Assertion.Argument.NotNull(documents);
            _dictionary = this.GetSoundSetFilePathDictionary(documents);

            this.AddConverter<SoundProject, XmlSoundProject>(SoundProjectToXmlSoundProject);
            this.AddConverter<ProjectSoundSetComponent, XmlProjectSoundSet>(ProjectSoundSetComponentToXmlProjectSoundSet);
            this.AddConverter<FolderComponent, XmlFolder>(FolderToXmlFolder);
            this.AddConverter<SoundProject, XmlSoundArchivePlayer>(SoundProjectToXmlSoundArchivePlayer);
            this.AddConverter<SoundProject, XmlConvert>(SoundProjectToXmlConvert);
            this.AddConverter<SoundProject, XmlItemNaming>(SoundProjectToXmlItemNaming);
            this.AddConverter<SoundProject, XmlCommentColumnText>(SoundProjectToXmlCommentColumnText);

            this.AddConverter<SoundListOutputType, string>(SoundListOutputTypeToString);
            this.AddDefaultConverter<SoundListOutput, XmlSoundListOutput>();
            this.AddDefaultConverter<ListOutput, XmlListOutput>();
            this.AddDefaultConverter<ListColumnOutput, XmlListColumnOutput>();
            this.AddDefaultConverter<UserCommand, XmlUserCommand>();

            this.AddConverter<StructureTypes, string>(StructureTypesToString);
            this.AddDefaultConverter<UserParameterStructureSetting, XmlUserParameterSetting>();

            this.AddConverter<SoundProject, XmlSndEdit>(SoundProjectToXmlSndEdit);
        }

        public XmlSoundProject Run(SoundProject soundProject)
        {
            return this.Run<XmlSoundProject>(soundProject) as XmlSoundProject;
        }

        private void TranslateProjectSoundSetRoot(SoundProject soundProject, XmlSoundProject dest, Converter converter)
        {
            Assertion.Argument.NotNull(soundProject);
            Assertion.Argument.NotNull(dest);
            Assertion.Argument.NotNull(converter);

            var root = new XmlProjectSoundSetRoot();
            TranslateProjectItems(soundProject, soundProject, root.Items, converter);
            dest.ProjectSoundSetRoot = root;
        }

        private void TranslateProjectItems(SoundProject soundProject, Component component, List<object> items, Converter converter)
        {
            foreach (var childComponent in component.Children)
            {
                if (childComponent is FolderComponent)
                {
                    var item = converter.Convert(childComponent, typeof(XmlFolder)) as XmlFolder;
                    items.Add(item);

                    TranslateProjectItems(soundProject, childComponent, item.Items, converter);
                }
                else if (childComponent is SoundSet)
                {
                    var soundSet = childComponent as SoundSet;
                    Debug.Assert(_dictionary.ContainsKey(soundSet) == true);
                    var filePath = _dictionary[soundSet];
                    Debug.Assert(string.IsNullOrEmpty(filePath) == false);
                    // プロジェクトからサウンドセットへの参照は、
                    // サウンドセット名とファイルパスを保存します。
                    var item = converter.Convert(new ProjectSoundSetComponent(soundSet.Name, filePath), typeof(XmlProjectSoundSet));
                    items.Add(item);
                }
                else if (childComponent is ProjectSoundSetComponent)
                {
                    var item = converter.Convert(childComponent as ProjectSoundSetComponent, typeof(XmlProjectSoundSet));
                    items.Add(item);
                }
            }
        }

        /// <summary>
        /// SoundProject から XmlSoundProject に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object SoundProjectToXmlSoundProject(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is SoundProject);
            Assertion.Argument.True(dstType == typeof(XmlSoundProject));

            SoundProject srcItem = src as SoundProject;

            XmlSoundProject dest = new XmlSoundProject();

            // フォルダ構成とサウンドセットの情報
            TranslateProjectSoundSetRoot(srcItem, dest, converter);

            // サウンドアーカイブプレイヤー設定
            dest.SoundArchivePlayer = this.SoundProjectToXmlSoundArchivePlayer(
                srcItem, typeof(XmlSoundArchivePlayer), converter) as XmlSoundArchivePlayer;

            // コンバート設定
            dest.Convert = this.SoundProjectToXmlConvert(
                srcItem, typeof(XmlConvert), converter) as XmlConvert;
            dest.Convert.PreConvertCommands = srcItem.PreConvertCommands;
            dest.Convert.PostConvertCommands = srcItem.PostConvertCommands;

            // アイテムの命名設定
            dest.ItemNaming = this.SoundProjectToXmlItemNaming(
                srcItem, typeof(XmlItemNaming), converter) as XmlItemNaming;

            // サウンドリスト出力設定
            dest.SoundListOutputs = new List<XmlSoundListOutput>();
            foreach (SoundListOutput soundListOutput in srcItem.SoundListOutputSettings.ListOutputs)
            {
                XmlSoundListOutput xmlSoundListOutput =
                    converter.Convert(soundListOutput, typeof(XmlSoundListOutput)) as XmlSoundListOutput;
                dest.SoundListOutputs.Add(xmlSoundListOutput);
            }

            // コメントカラム名の設定
            dest.CommentColumnText = this.SoundProjectToXmlCommentColumnText(srcItem, typeof(XmlCommentColumnText), converter) as XmlCommentColumnText;

            // カラーコメントの設定
            dest.ColorComment = this.SoundProjectToXmlColorComment
                (srcItem, typeof(XmlColorComment), converter) as XmlColorComment;

            // ユーザーコマンドの設定
            dest.UserCommands = new List<XmlUserCommand>();
            foreach (UserCommand userCommand in srcItem.UserCommands)
            {
                dest.UserCommands.Add
                    (converter.Convert(userCommand, typeof(XmlUserCommand)) as XmlUserCommand);
            }

            // ユーザーパラメータの設定
            dest.UserParameterSettings = new List<XmlUserParameterSetting>();
            foreach (UserParameterStructureSetting setting in srcItem.UserDataStructureSettings.Settings)
            {
                XmlUserParameterSetting xmlUserParameterSetting = converter.Convert
                    (setting, typeof(XmlUserParameterSetting)) as XmlUserParameterSetting;
                dest.UserParameterSettings.Add(xmlUserParameterSetting);
            }

            // プロジェクト設定
            dest.ProjectSetting = SoundProjectToXmlProjectSetting
                (srcItem, typeof(XmlProjectSetting), converter) as XmlProjectSetting;

            // ファイルイベントの設定
            dest.FileEvent = SoundProjectToXmlFileEvent
                (srcItem, typeof(XmlFileEvent), converter) as XmlFileEvent;

            // sndedit 設定
            dest.SndEdit = SoundProjectToXmlSndEdit
                (srcItem, typeof(XmlSndEdit), converter) as XmlSndEdit;

            // 接続、切断の設定
            dest.Connection = SoundProjectToXmlConnection
                (srcItem, typeof(XmlConnection), converter) as XmlConnection;

            // 統計の設定
            dest.Statistics = SoundProjectToXmlStatistics
                (srcItem, typeof(XmlStatistics), converter) as XmlStatistics;

            //
            if (SoundProjectTranslatorSettings.EnabledSoundArchiveOutputType == true)
            {
                var soundSets = new HashSet<SoundSet>(GetSoundSets(srcItem));
                dest.SoundArchiveSettings = new List<XmlSoundArchiveSetting>();
                foreach (var setting in srcItem.SoundArchiveOutputTypes)
                {
                    if (soundSets.Contains(setting.Key) == true)
                    {
                        var xmlSetting = new XmlSoundArchiveSetting();
                        xmlSetting.Name = setting.Key.Name;
                        xmlSetting.OutputType = (setting.Value).ToText();
                        dest.SoundArchiveSettings.Add(xmlSetting);
                    }
                }
            }

            return dest;
        }

        private IEnumerable<SoundSet> GetSoundSets(Component component)
        {
            if (component is SoundSet)
            {
                yield return component as SoundSet;
                yield break;
            }

            foreach (var childComponent in component.Children)
            {
                foreach (var resultComponent in GetSoundSets(childComponent))
                {
                    yield return resultComponent as SoundSet;
                }
            }
        }

        /// <summary>
        /// SoundListOutputType から String に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object SoundListOutputTypeToString(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is SoundListOutputType);
            Assertion.Argument.True(dstType == typeof(string));

            return ((SoundListOutputType)src).ToText();
        }

        /// <summary>
        /// StructureTypes から String に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object StructureTypesToString(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is StructureTypes);
            Assertion.Argument.True(dstType == typeof(string));

            return ((StructureTypes)src).ToText();
        }

        /// <summary>
        /// ProjectSoundSetComponent から XmlProjectSoundSet に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object ProjectSoundSetComponentToXmlProjectSoundSet(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is ProjectSoundSetComponent);
            Assertion.Argument.True(dstType == typeof(XmlProjectSoundSet));

            ProjectSoundSetComponent srcItem = src as ProjectSoundSetComponent;
            XmlProjectSoundSet dest = new XmlProjectSoundSet()
            {
                Name = srcItem.Name,
                Parameters = new List<XmlParameter>(),
            };

            dest.Parameters.Add(
                new XmlFilePathParam()
                {
                    Value = this.PathResolver.GetRelativePath(srcItem.TargetFilePath),
                });

            return dest;
        }

        private object FolderToXmlFolder(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is FolderComponent);
            Assertion.Argument.True(dstType == typeof(XmlFolder));

            var folder = src as FolderComponent;
            var xmlFolder = new XmlFolder() { Name = folder.Name };
            return xmlFolder;
        }

        /// <summary>
        /// SoundProject から XmlSoundArchivePlayer に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object SoundProjectToXmlSoundArchivePlayer(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is SoundProject);
            Assertion.Argument.True(dstType == typeof(XmlSoundArchivePlayer));

            string[] targetParamNames = new string[]
            {
                ProjectParameterNames.SoundArchivePlayer.SequenceSoundCount,
                ProjectParameterNames.SoundArchivePlayer.SequenceTrackCount,
                ProjectParameterNames.SoundArchivePlayer.StreamChannelCount,
                ProjectParameterNames.SoundArchivePlayer.StreamBufferTimes,
                ProjectParameterNames.SoundArchivePlayer.StreamSoundCount,
                ProjectParameterNames.SoundArchivePlayer.WaveSoundCount,
            };

            targetParamNames = SoundProjectParameterTranslateFilter.Filter(targetParamNames);
            XmlSoundArchivePlayer xmlSoundArchivePlayer = new XmlSoundArchivePlayer();
            TranslateParameters(src as SoundProject, targetParamNames, xmlSoundArchivePlayer, converter);

            return xmlSoundArchivePlayer;
        }

        /// <summary>
        /// SoundProject から XmlConvert に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object SoundProjectToXmlConvert(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is SoundProject);
            Assertion.Argument.True(dstType == typeof(XmlConvert));

            List<string> targetParamNamesList = new List<string>()
            {
                ProjectParameterNames.Conversion.DoWarnUnreferencedItems,
                ProjectParameterNames.Conversion.DoWarnDisableGroupItemTargets,
                ProjectParameterNames.Conversion.DoWarnPCBinariesForAACNotFound,
                ProjectParameterNames.Conversion.ExcludeStringTable,
                ProjectParameterNames.Conversion.DoOutputPCBinariesForAAC,
                ProjectParameterNames.Conversion.DoUseHardwareOpusDecoder,
                ProjectParameterNames.Conversion.ExternalFileDirectoryPath,
                ProjectParameterNames.Conversion.UserManagementFileDirectoryPath,
                ProjectParameterNames.Conversion.IntermediateOutputDirectoryPath,
                ProjectParameterNames.Conversion.InGameEditCacheOutputDirectoryPath,
                ProjectParameterNames.Conversion.IsPreConvertCommandsEnabled,
                ProjectParameterNames.Conversion.IsPostConvertCommandsEnabled,
                ProjectParameterNames.Conversion.KeepIntermediateTextSequence,
                ProjectParameterNames.Conversion.OutputLabel,
                ProjectParameterNames.Conversion.OutputDirectoryPath,
                ProjectParameterNames.Conversion.SmfTimebase,
                ProjectParameterNames.Conversion.DoUseBuiltInWavePreprocessExe,
                ProjectParameterNames.Conversion.WavePreprocessExePath,
            };

            // 非公開機能が有効の場合
            if ((src as SoundProject).DoDeleteGarbageOutputBinaries == true)
            {
                targetParamNamesList.Add(ProjectParameterNames.Conversion.DoDeleteGarbageOutputBinaries);
            }

            string[] targetParamNames = SoundProjectParameterTranslateFilter.Filter(targetParamNamesList.ToArray());
            XmlConvert xmlConvert = new XmlConvert();
            TranslateParameters(src as SoundProject, targetParamNames, xmlConvert, converter);

            return xmlConvert;
        }

        /// <summary>
        /// SoundProject から XmlItemNaming に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object SoundProjectToXmlItemNaming(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is SoundProject);
            Assertion.Argument.True(dstType == typeof(XmlItemNaming));

            string[] targetParamNames = new string[]
            {
                ProjectParameterNames.ItemNaming.BankNamePrefix,
                ProjectParameterNames.ItemNaming.CaseChange,
                ProjectParameterNames.ItemNaming.GroupNamePrefix,
                ProjectParameterNames.ItemNaming.HasPrefix,
                ProjectParameterNames.ItemNaming.InstrumentNamePrefix,
                ProjectParameterNames.ItemNaming.InvalidCharChange,
                ProjectParameterNames.ItemNaming.PlayerNamePrefix,
                ProjectParameterNames.ItemNaming.SequenceSoundNamePrefix,
                ProjectParameterNames.ItemNaming.SequenceSoundSetNamePrefix,
                ProjectParameterNames.ItemNaming.StreamSoundNamePrefix,
                ProjectParameterNames.ItemNaming.WaveArchiveNamePrefix,
                ProjectParameterNames.ItemNaming.WaveSoundNamePrefix,
                ProjectParameterNames.ItemNaming.WaveSoundSetNamePrefix,
                ProjectParameterNames.ItemNaming.ItemPastePostfix,
                ProjectParameterNames.ItemNaming.EnabledNameDelimiter,
                ProjectParameterNames.ItemNaming.NameDelimiter,
            };

            XmlItemNaming xmlItemNaming = new XmlItemNaming();
            TranslateParameters(src as SoundProject, targetParamNames, xmlItemNaming, converter);

            return xmlItemNaming;
        }

        /// <summary>
        /// SoundProject から XmlCommentColumnText に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object SoundProjectToXmlCommentColumnText(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is SoundProject);
            Assertion.Argument.True(dstType == typeof(XmlCommentColumnText));

            string[] targetParamNames = new string[]
            {
                ProjectParameterNames.CommentColumnText.Comment,
                ProjectParameterNames.CommentColumnText.Comment1,
                ProjectParameterNames.CommentColumnText.Comment2,
                ProjectParameterNames.CommentColumnText.Comment3,
                ProjectParameterNames.CommentColumnText.Comment4,
                ProjectParameterNames.CommentColumnText.Comment5,
                ProjectParameterNames.CommentColumnText.Comment6,
                ProjectParameterNames.CommentColumnText.Comment7,
                ProjectParameterNames.CommentColumnText.Comment8,
                ProjectParameterNames.CommentColumnText.Comment9,
            };

            XmlCommentColumnText xmlCommentColumnText = new XmlCommentColumnText();
            TranslateParameters(src as SoundProject, targetParamNames, xmlCommentColumnText, converter);

            return xmlCommentColumnText;
        }

        /// <summary>
        /// SoundProject から XmlColorComment に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object SoundProjectToXmlColorComment(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is SoundProject);
            Assertion.Argument.True(dstType == typeof(XmlColorComment));

            string[] targetParamNames = new string[]
            {
                ProjectParameterNames.ColorComment.Comment0,
                ProjectParameterNames.ColorComment.Comment1,
                ProjectParameterNames.ColorComment.Comment2,
                ProjectParameterNames.ColorComment.Comment3,
                ProjectParameterNames.ColorComment.Comment4,
                ProjectParameterNames.ColorComment.Comment5,
                ProjectParameterNames.ColorComment.Comment6,
                ProjectParameterNames.ColorComment.Comment7,
                ProjectParameterNames.ColorComment.Comment8,
            };

            XmlColorComment xmlColorComment = new XmlColorComment();
            TranslateParameters
                (src as SoundProject, targetParamNames, xmlColorComment, converter);

            return xmlColorComment;
        }

        /// <summary>
        /// SoundProject から XmlProjectSetting に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object SoundProjectToXmlProjectSetting(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is SoundProject);
            Assertion.Argument.True(dstType == typeof(XmlProjectSetting));

            string[] targetParamNames = new string[]
            {
                ProjectParameterNames.Project.ProjectComment,
            };

            XmlProjectSetting xmlProjectSetting = new XmlProjectSetting();
            TranslateParameters(src as SoundProject, targetParamNames,
                                 xmlProjectSetting, converter);

            return xmlProjectSetting;
        }

        /// <summary>
        /// SoundProject から XmlFileEvent に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object SoundProjectToXmlFileEvent(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is SoundProject);
            Assertion.Argument.True(dstType == typeof(XmlFileEvent));

            string[] targetParamNames = new string[]
            {
                ProjectParameterNames.FileEvent.IsFileSavePreCommandEnabled,
                ProjectParameterNames.FileEvent.IsFileSavePostCommandEnabled,
                ProjectParameterNames.FileEvent.FileSavePreCommandPath,
                ProjectParameterNames.FileEvent.FileSavePostCommandPath,
            };

            XmlFileEvent xmlFileEvent = new XmlFileEvent();
            TranslateParameters(src as SoundProject, targetParamNames, xmlFileEvent, converter);

            return xmlFileEvent;
        }

        /// <summary>
        /// SoundProject から XmlSndEdit に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object SoundProjectToXmlSndEdit(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is SoundProject);
            Assertion.Argument.True(dstType == typeof(XmlSndEdit));

            string[] paramNames = SoundProjectParameterTranslateSndEdit.SndEditParamNames;
            XmlSndEdit xmlSndEdit = new XmlSndEdit();
            TranslateParameters(src as SoundProject, paramNames, xmlSndEdit, converter);

            return xmlSndEdit;
        }

        /// <summary>
        /// SoundProject から XmlConnection に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object SoundProjectToXmlConnection(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is SoundProject);
            Assertion.Argument.True(dstType == typeof(XmlConnection));

            SoundProject project = src as SoundProject;
            List<string> paramNames = new List<string>();

            if (project.EnabledConnectionTimeOut == true)
            {
                paramNames.Add(ProjectParameterNames.Connection.ConnectionTimeOut);
            }

            if (paramNames.Count() > 0)
            {
                XmlConnection xmlConnection = new XmlConnection();
                TranslateParameters(src as SoundProject, paramNames.ToArray(), xmlConnection, converter);

                return xmlConnection;
            }

            return null;
        }

        /// <summary>
        /// SoundProject から XmlStatistics に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object SoundProjectToXmlStatistics(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is SoundProject);
            Assertion.Argument.True(dstType == typeof(XmlStatistics));

            string[] targetParamNames = new string[]
            {
                ProjectParameterNames.Statistics.EnabledIntegratedLoudnessThreshold,
                ProjectParameterNames.Statistics.IntegratedLoudnessThreshold,
                ProjectParameterNames.Statistics.SequenceMaxMeasureDuration,
            };

            XmlStatistics xmlStatistics = new XmlStatistics();
            TranslateParameters(src as SoundProject, targetParamNames, xmlStatistics, converter);

            return xmlStatistics;
        }

        private Dictionary<SoundSet, string> GetSoundSetFilePathDictionary(Document[] documents)
        {
            Dictionary<SoundSet, string> dictionary = new Dictionary<SoundSet, string>();

            foreach (var document in documents.OfType<SoundSetDocument>())
            {
                var soundSet = document.TopComponents[0] as SoundSet;

                if (dictionary.ContainsKey(soundSet) == false)
                {
                    dictionary.Add(soundSet, document.Resource.Key);
                }
            }

            return dictionary;
        }
    }
}
