﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate
{
    using System;

    /// <summary>
    /// シーケンスサウンドテキストを解析します。
    /// </summary>
    public partial class SequenceSoundTextLexer
    {
        /// <summary>
        /// 式を解析します。
        /// </summary>
        public class ExpressionParser
        {
            public delegate bool ParseSymbolCallback(string symbol, out long val);

            public long ParseTerm(SequenceSoundTextLexer lexer, ParseSymbolCallback symbolParser)
            {
                Token token = lexer.ReadToken();
                long x;

                switch (token)
                {
                    case Token.Number:
                        return lexer.NumberValue;

                    case Token.Minus:
                        return -ParseTerm(lexer, symbolParser);

                    case Token.Plus:
                        return +ParseTerm(lexer, symbolParser);

                    case Token.LeftParen:
                        x = ParseOr(lexer, symbolParser);
                        token = lexer.ReadToken();
                        if (token != Token.RightParen)
                        {
                            throw new LexerException("Unmatch ( )", lexer);
                        }
                        return x;

                    case Token.Symbol:
                        string symbol = lexer.StringValue;
                        if (symbolParser == null || !symbolParser(symbol, out x))
                        {
                            throw new LexerException("Undefined symbol \"" + symbol + "\"", lexer);
                        }
                        return x;

                    default:
                        throw new LexerException("Syntax error \"" + lexer.LastTokenString + "\"", lexer);
                }
            }

            public long ParseMul(SequenceSoundTextLexer lexer, ParseSymbolCallback symbolParser)
            {
                long left = ParseTerm(lexer, symbolParser);

                while (true)
                {
                    Token token = lexer.PeekToken();

                    switch (token)
                    {
                        case Token.Mul:
                            lexer.ReadToken(); // eat
                            left *= ParseTerm(lexer, symbolParser);
                            break;
                        case Token.Div:
                            lexer.ReadToken(); // eat
                            long d = ParseTerm(lexer, symbolParser);
                            if (d == 0)
                            {
                                throw new LexerException("Devide by zero", lexer);
                            }
                            left /= d;
                            break;

                        default:
                            return left;
                    }
                }
            }

            public long ParsePlus(SequenceSoundTextLexer lexer, ParseSymbolCallback symbolParser)
            {
                long left = ParseMul(lexer, symbolParser);

                while (true)
                {
                    Token token = lexer.PeekToken();

                    switch (token)
                    {
                        case Token.Plus:
                            lexer.ReadToken(); // eat
                            left += ParseMul(lexer, symbolParser);
                            break;
                        case Token.Minus:
                            lexer.ReadToken(); // eat
                            left -= ParseMul(lexer, symbolParser);
                            break;

                        default:
                            return left;
                    }
                }
            }

            public long ParseShift(SequenceSoundTextLexer lexer, ParseSymbolCallback symbolParser)
            {
                long left = ParsePlus(lexer, symbolParser);

                while (true)
                {
                    Token token = lexer.PeekToken();

                    switch (token)
                    {
                        case Token.LeftShift:
                            lexer.ReadToken(); // eat
                            left <<= (int)ParsePlus(lexer, symbolParser);
                            break;
                        case Token.RightShift:
                            lexer.ReadToken(); // eat
                            left >>= (int)ParsePlus(lexer, symbolParser);
                            break;

                        default:
                            return left;
                    }
                }
            }

            public long ParseCompare(SequenceSoundTextLexer lexer, ParseSymbolCallback symbolParser)
            {
                long left = ParseShift(lexer, symbolParser);

                while (true)
                {
                    Token token = lexer.PeekToken();

                    switch (token)
                    {
                        case Token.LessThan:
                            lexer.ReadToken(); // eat
                            left = Convert.ToInt64(left < ParseShift(lexer, symbolParser));
                            break;
                        case Token.LessEqual:
                            lexer.ReadToken(); // eat
                            left = Convert.ToInt64(left <= ParseShift(lexer, symbolParser));
                            break;
                        case Token.GreaterThan:
                            lexer.ReadToken(); // eat
                            left = Convert.ToInt64(left > ParseShift(lexer, symbolParser));
                            break;
                        case Token.GreaterEqual:
                            lexer.ReadToken(); // eat
                            left = Convert.ToInt64(left >= ParseShift(lexer, symbolParser));
                            break;

                        default:
                            return left;
                    }
                }
            }

            public long ParseEqual(SequenceSoundTextLexer lexer, ParseSymbolCallback symbolParser)
            {
                long left = ParseCompare(lexer, symbolParser);

                while (true)
                {
                    Token token = lexer.PeekToken();

                    switch (token)
                    {
                        case Token.Equal:
                            lexer.ReadToken(); // eat
                            left = (left == ParseCompare(lexer, symbolParser)) ? 1 : 0;
                            break;

                        default:
                            return left;
                    }
                }
            }

            public long ParseAnd(SequenceSoundTextLexer lexer, ParseSymbolCallback symbolParser)
            {
                long left = ParseEqual(lexer, symbolParser);

                while (true)
                {
                    Token token = lexer.PeekToken();

                    switch (token)
                    {
                        case Token.And:
                            lexer.ReadToken(); // eat
                            left &= ParseEqual(lexer, symbolParser);
                            break;

                        default:
                            return left;
                    }
                }
            }

            public long ParseOr(SequenceSoundTextLexer lexer, ParseSymbolCallback symbolParser)
            {
                long left = ParseAnd(lexer, symbolParser);

                while (true)
                {
                    Token token = lexer.PeekToken();

                    switch (token)
                    {
                        case Token.Or:
                            lexer.ReadToken(); // eat
                            left |= ParseAnd(lexer, symbolParser);
                            break;

                        default:
                            return left;
                    }
                }
            }

            public long Parse(SequenceSoundTextLexer lexer)
            {
                return ParseOr(lexer, null);
            }

            public long Parse(SequenceSoundTextLexer lexer, ParseSymbolCallback symbolParser)
            {
                return ParseOr(lexer, symbolParser);
            }
        }
    }
}
