﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;

namespace NintendoWare.SoundFoundation.Core
{
    /// <summary>
    /// イベントのルーティング機能を提供します。
    /// </summary>
    public class EventRouter
    {
        // リンク情報
        private IEventRoutable _owner;

        // バインディングディクショナリ
        private RoutingEventBindingMap _bindingMap = new RoutingEventBindingMap();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="owner">イベントルー他の所有者。</param>
        public EventRouter(IEventRoutable owner)
        {
            if (null == owner) { throw new ArgumentNullException("owner"); }
            _owner = owner;
        }

        /// <summary>
        /// イベントルータの所有者を取得します。
        /// </summary>
        public IEventRoutable Owner
        {
            get { return _owner; }
        }

        /// <summary>
        /// イベントとイベントハンドラマップを取得します。
        /// </summary>
        public RoutingEventBindingMap Bindings
        {
            get { return _bindingMap; }
        }

        /// <summary>
        /// バブリングイベントを発行します。
        /// </summary>
        /// <param name="e">ルーティングイベントデータ。</param>
        public void BubbleEvent(RoutingEventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            DoBubblingEvent(Owner, e, false);
        }

        /// <summary>
        /// イベントを発行します。
        /// </summary>
        /// <param name="e">ルーティングイベントデータ。</param>
        public void DispatchEvent(RoutingEventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            DoEvent(Owner, e, false);
        }

        /// <summary>
        /// バブリングイベントを処理します。
        /// </summary>
        /// <param name="sender">イベントの発行元。</param>
        /// <param name="e">ルーティングイベントデータ。</param>
        /// <param name="isInvoke">イベントを実行するかどうか。</param>
        private void DoBubblingEvent(object sender, RoutingEventArgs e, bool isInvoke)
        {
            if (null == sender) { throw new ArgumentNullException("sender"); }
            if (null == e) { throw new ArgumentNullException("e"); }

            Debug.Assert(!e.Cancel, "RoutingEventArgs.Cancel must be false.");

            if (isInvoke && _bindingMap.ContainsKey(e.Event.GetType()))
            {

                foreach (RoutingEventBinding binding in _bindingMap[e.Event.GetType()])
                {
                    binding.InvokeBubblingEvent(sender, e);
                }

            }

            if (e.Cancel || e.Destination == this) { return; }

            if (null != Owner.Parent)
            {
                Owner.Parent.Router.DoBubblingEvent(sender, e, true);
            }
        }

        /// <summary>
        /// イベントを処理します。
        /// </summary>
        /// <param name="sender">イベントの発行元。</param>
        /// <param name="e">ルーティングイベントデータ。</param>
        /// <param name="isInvoke">イベントを実行するかどうか。</param>
        private void DoEvent(object sender, RoutingEventArgs e, bool isInvoke)
        {
            if (null == sender) { throw new ArgumentNullException("sender"); }
            if (null == e) { throw new ArgumentNullException("e"); }

            Debug.Assert(!e.Cancel, "RoutingEventArgs.Cancel must be false.");

            if (isInvoke && _bindingMap.ContainsKey(e.Event.GetType()))
            {

                foreach (RoutingEventBinding binding in _bindingMap[e.Event.GetType()])
                {
                    binding.InvokeEvent(sender, e);
                }
            }

            if (e.Cancel) { return; }

            // 宛先指定
            if (null != e.Destination)
            {

                IEventRoutable target = e.Destination;

                while (null != target)
                {

                    if (Owner.Children.Contains(target))
                    {
                        target.Router.DoEvent(sender, e, true);
                        return;
                    }

                    target = target.Parent;

                }

                throw new Exception("invalid routing event destination.");

            }
            // ブロードキャスト
            else
            {

                foreach (IEventRoutable child in Owner.Children)
                {
                    child.Router.DoEvent(sender, e, true);
                }

            }
        }
    }
}
