﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Core.Parameters
{
    using System;
    using System.Collections.Generic;
    using ToolDevelopmentKit;

    /// <summary>
    /// int 型のパラメータ値を格納します。
    /// </summary>
    public class IntParameterValue : ParameterValue<int>
    {
        private int maxValue = int.MaxValue;
        private int minValue = int.MinValue;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public IntParameterValue()
            : base(0)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="value">パラメータの値を指定します。</param>
        public IntParameterValue(int value)
            : base(value)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="value">パラメータの値を指定します。</param>
        /// <param name="min">最小値を指定します。</param>
        /// <param name="max">最大値を指定します。</param>
        public IntParameterValue(int value, int min, int max)
            : base(value)
        {
            if (min > max) { throw new ArgumentException("min must be over max."); }

            this.minValue = min;
            this.maxValue = max;
        }

        /// <summary>
        /// 最小値を取得します。
        /// </summary>
        public int MinValue
        {
            get { return this.minValue; }
        }

        /// <summary>
        /// 最大値を取得します。
        /// </summary>
        public int MaxValue
        {
            get { return this.maxValue; }
        }

        /// <summary>
        /// パラメータの値を取得するキャストオペレータです。
        /// </summary>
        /// <param name="value">対象となるパラメータ値を指定します。</param>
        /// <returns>パラメータの値を返します。</returns>
        public static implicit operator int(IntParameterValue value)
        {
            Ensure.Argument.NotNull(value);
            return value.Value;
        }

        /// <summary>
        /// テキストからパラメータ値を取得します。
        /// </summary>
        /// <param name="text">テキストを指定します。</param>
        protected override int ParseInternal(string text)
        {
            return int.Parse(text);
        }

        /// <summary>
        /// 指定した値を検証します。
        /// </summary>
        /// <param name="value">検証する値を指定します。</param>
        /// <returns>検証結果を返します。</returns>
        protected override ValidationResult ValidateInternal(int value)
        {
            if (value < this.minValue || value > this.maxValue)
            {
                return new OutOfRangeValidationResult(this.minValue, this.maxValue);
            }

            return ValidationResult.NoError;
        }

        /// <summary>
        /// パラメータ値のバイト列を取得します。
        /// </summary>
        /// <returns>バイト列の列挙子を返します。</returns>
        protected override IEnumerable<byte> GetBytes()
        {
            return BitConverter.GetBytes(this.Value);
        }
    }
}
