﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Core.IO
{
    using System;
    using System.Collections.Generic;
    using System.Collections.Specialized;

    /// <summary>
    /// コマンドライン引数を解析します。
    /// </summary>
    public class CommandLineParser
    {
        private static readonly string[] OptionPrefixes = new string[] { "--", "-", "/" };
        private Dictionary<string, List<string>> arguments =
        new Dictionary<string, List<string>>();

        /// <summary>
        /// コマンドライン引数の値が有効化どうかを調べます。
        /// </summary>
        /// <returns>有効な場合は true、無効な場合は false。</returns>
        public bool IsValueEnabled()
        {
            return this.ContainsOption(string.Empty);
        }

        /// <summary>
        /// オプションの数を取得します。
        /// </summary>
        /// <returns>オプションの数</returns>
        public int OptionCount
        {
            get
            {
                int count = this.arguments.Count - 1;
                if (this.arguments.Count < 0)
                {
                    return 0;
                }
                else
                {
                    return count;
                }
            }
        }

        /// <summary>
        /// 指定オプションが含まれているかどうかを調べます。
        /// </summary>
        /// <param name="option">オプションの名前。</param>
        /// <returns>有効な場合は true、無効な場合は false。</returns>
        public bool ContainsOption(string option)
        {
            return this.arguments.ContainsKey(option);
        }

        /// <summary>
        /// 指定オプションの値を取得します。
        /// </summary>
        /// <param name="option">オプションの名前。</param>
        /// <returns>オプションの値。</returns>
        public string GetOptionValue(string option)
        {
            string[] strings = GetOptionValues(option);
            if (0 < strings.Length)
            {
                return strings[0];
            }

            return string.Empty;
        }

        /// <summary>
        /// 指定オプションの値らを取得します。
        /// </summary>
        /// <param name="option">オプションの名前。</param>
        /// <returns>オプションの値ら。</returns>
        public string[] GetOptionValues(string option)
        {
            if (this.arguments.ContainsKey(option) == false)
            {
                return new List<string>().ToArray();
            }

            return this.arguments[option].ToArray();
        }

        /// <summary>
        /// コマンドライン引数の値を取得します。
        /// </summary>
        /// <returns>コマンドライン引数の値。</returns>
        public string GetValue()
        {
            return this.GetOptionValue(string.Empty);
        }

        /// <summary>
        /// コマンドライン引数の値らを取得します。
        /// </summary>
        /// <returns>コマンドライン引数の値ら。</returns>
        public string[] GetValues()
        {
            return this.GetOptionValues(string.Empty);
        }

        /// <summary>
        /// コマンドライン引数を解析します。
        /// </summary>
        /// <param name="arguments">コマンドライン引数の配列を指定します。</param>
        public void ParseArguments(string[] arguments)
        {
            if (null == arguments) { return; }
            if (0 == arguments.Length) { return; }

            int current = 0;
            string currentOption = string.Empty;

            while (current < arguments.Length)
            {

                string arg = arguments[current];
                if (null == arg) { continue; }

                string option = GetOptionName(arg);

                if (0 < option.Length)
                {
                    this.arguments.Add(option, new List<string>());
                    currentOption = option;
                }
                else
                {
                    if (this.arguments.ContainsKey(currentOption))
                    {
                        this.arguments[currentOption].Add(arg);
                    }
                    else
                    {
                        List<string> list = new List<string>();
                        list.Add(arg);
                        this.arguments.Add(currentOption, list);
                    }

                    currentOption = string.Empty;
                }

                current++;
            }
        }

        private string GetOptionName(string argument)
        {
            if (null == argument) { throw new Exception(); }

            if (2 > argument.Length) { return string.Empty; }

            foreach (string optionPrefix in CommandLineParser.OptionPrefixes)
            {
                if (argument.StartsWith(optionPrefix))
                {
                    return argument.Substring(optionPrefix.Length);
                }
            }

            return string.Empty;
        }
    }
}
