﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Core
{
    using System;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// ハッシュ値を示す構造体です。
    /// </summary>
    public struct HashCode
    {
        /// <summary>
        /// 空のハッシュ値です。
        /// </summary>
        public static readonly HashCode Empty = new HashCode(new byte[0]);

        private readonly byte[] value;

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="value">ハッシュ値を指定します。</param>
        public HashCode(byte[] value)
        {
            Assertion.Argument.NotNull(value);
            this.value = value;
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        public byte[] Value
        {
            get
            {
                return this.value;
            }
        }

        /// <summary>
        /// ハッシュ値の長さを取得します。
        /// </summary>
        public int Length
        {
            get
            {
                return this.Value.Length;
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 指定ハッシュ値の同値比較を行います。
        /// </summary>
        /// <param name="hashCode1">１つ目のハッシュ値を指定します。</param>
        /// <param name="hashCode2">２つ目のハッシュ値を指定します。</param>
        /// <returns>等しい場合は true、等しくない場合は false を返します。</returns>
        public static bool operator ==(HashCode hashCode1, HashCode hashCode2)
        {
            return hashCode1.Equals(hashCode2);
        }

        /// <summary>
        /// 指定ハッシュ値の同値比較を行います。
        /// </summary>
        /// <param name="hashCode1">１つ目のハッシュ値を指定します。</param>
        /// <param name="hashCode2">２つ目のハッシュ値を指定します。</param>
        /// <returns>等しくない場合は true、等しい場合は false を返します。</returns>
        public static bool operator !=(HashCode hashCode1, HashCode hashCode2)
        {
            return !hashCode1.Equals(hashCode2);
        }

        /// <summary>
        /// ハッシュ値の XOR を計算します。
        /// </summary>
        /// <param name="hashCode1">１つ目のハッシュ値を指定します。</param>
        /// <param name="hashCode2">２つ目のハッシュ値を指定します。</param>
        /// <returns>計算結果を返します。</returns>
        public static HashCode operator ^(HashCode hashCode1, HashCode hashCode2)
        {
            Ensure.Operation.True(hashCode1.Length == hashCode2.Length);

            var newValue = new byte[hashCode1.Length];

            for (int i = 0; i < hashCode1.Length; i++)
            {
                newValue[i] = (byte)(hashCode1.Value[i] ^ hashCode2.Value[i]);
            }

            return new HashCode(newValue);
        }

        /// <summary>
        /// 文字列を解析して HashCode を作成します。
        /// </summary>
        /// <param name="text">文字列を指定します。</param>
        /// <returns>作成した HashCode を返します。</returns>
        public static HashCode Parse(string text)
        {
            if (string.IsNullOrEmpty(text))
            {
                return HashCode.Empty;
            }

            byte[] bytes = new byte[text.Length / 2];
            int textPosition = 0;

            for (int i = 0; i < bytes.Length; i++)
            {
                bytes[i] = Convert.ToByte(text.Substring(textPosition, 2), 16);
                textPosition += 2;
            }

            return new HashCode(bytes);
        }

        /// <summary>
        /// 文字列に変換します。
        /// </summary>
        /// <returns>返還後の文字列を返します。</returns>
        public override string ToString()
        {
            return BitConverter.ToString(this.Value).Replace("-", string.Empty);
        }

        /// <summary>
        /// 指定オブジェクトとの同値比較を行います。
        /// </summary>
        /// <param name="obj">オブジェクトを指定します。</param>
        /// <returns>等しい場合は true、等しくない場合は false を返します。</returns>
        public override bool Equals(object obj)
        {
            if (!(obj is HashCode))
            {
                return false;
            }

            HashCode other = (HashCode)obj;

            if (this.Length != other.Length)
            {
                return false;
            }

            for (int i = 0; i < this.Length; i++)
            {
                if (this.Value[i] != other.Value[i])
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        /// <returns>ハッシュ値を返します。</returns>
        public override int GetHashCode()
        {
            return this.Value.GetHashCode();
        }
    }
}
