﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion
{
    using NintendoWare.SoundFoundation.Projects;
    using System.Collections.Generic;

    public class SoundProjectConvertPartsSettings : SoundProjectConvertBaseSettings
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="platform">対象プラットフォーム。</param>
        /// <param name="targets">コンバータ対象アイテムの列挙子。</param>
        /// <param name="outputDirectoryPath">出力先のディレクトリパス。</param>
        /// <param name="doGarbageCorrection">ガベージコレクションの有無。</param>
        /// <param name="reconvertFilter">再コンバータフィルタ。</param>
        public SoundProjectConvertPartsSettings(
            string platform,
            IEnumerable<SoundSetItem> targets,
            string outputDirectoryPath,
            bool doGarbageCorrection,
            SoundProjectReconvertFilter reconvertFilter)
            : base(platform, reconvertFilter)
        {
            this.Targets = targets;
            this.OutputDirectoryPath = outputDirectoryPath;
            this.DoGarbageCorrection = doGarbageCorrection;
        }

        /// <summary>
        /// コンバータ対象アイテムの列挙子を取得します。
        /// </summary>
        public IEnumerable<SoundSetItem> Targets { get; private set; }

        /// <summary>
        /// 出力先のディレクトリパスを取得します。
        /// </summary>
        public string OutputDirectoryPath { get; private set; }

        /// <summary>
        /// ガベージコレクションの有無を取得します。
        /// </summary>
        public bool DoGarbageCorrection { get; private set; }

        /// <summary>
        /// 通常コンバート用の設定を生成します。
        /// </summary>
        /// <param name="platform">対象プラットフォーム。</param>
        /// <param name="targets">コンバータ対象アイテムの列挙子。</param>
        /// <param name="outputDirectoryPath">出力先のディレクトリパス。</param>
        /// <param name="doGarbageCorrection">ガベージコレクションの有無。</param>
        public static SoundProjectConvertPartsSettings CreateForConvert(
            string platform,
            IEnumerable<SoundSetItem> targets,
            string outputDirectoryPath,
            bool doGarbageCorrection)
        {
            return new SoundProjectConvertPartsSettings(
                platform,
                targets,
                outputDirectoryPath,
                doGarbageCorrection,
                SoundProjectReconvertFilter.None);
        }

        /// <summary>
        /// 再コンバート用の設定を生成します。
        /// </summary>
        /// <param name="platform">対象プラットフォーム。</param>
        /// <param name="targets">コンバータ対象アイテムの列挙子。</param>
        /// <param name="outputDirectoryPath">出力先のディレクトリパス。</param>
        /// <param name="doGarbageCorrection">ガベージコレクションの有無。</param>
        public static SoundProjectConvertPartsSettings CreateForReconvert(
            string platform,
            IEnumerable<SoundSetItem> targets,
            string outputDirectoryPath,
            bool doGarbageCorrection)
        {
            return CreateForReconvert(
                platform,
                targets,
                outputDirectoryPath,
                doGarbageCorrection,
                SoundProjectReconvertFilter.All);
        }

        /// <summary>
        /// 再コンバート用の設定を生成します。
        /// </summary>
        /// <param name="platform">対象プラットフォーム。</param>
        /// <param name="targets">コンバータ対象アイテムの列挙子。</param>
        /// <param name="outputDirectoryPath">出力先のディレクトリパス。</param>
        /// <param name="doGarbageCorrection">ガベージコレクションの有無。</param>
        /// <param name="reconvertFilter">再コンバータフィルタ。</param>
        public static SoundProjectConvertPartsSettings CreateForReconvert(
            string platform,
            IEnumerable<SoundSetItem> targets,
            string outputDirectoryPath,
            bool doGarbageCorrection,
            SoundProjectReconvertFilter reconvertFilter)
        {
            return new SoundProjectConvertPartsSettings(
                platform,
                targets,
                outputDirectoryPath,
                doGarbageCorrection,
                reconvertFilter);
        }
    }
}
