﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using System.Xml.Serialization;

    /// <summary>
    /// 依存関係管理クラス
    /// </summary>
    public partial class DependencyManager : IXmlSerializable
    {
        /// <summary>
        /// アイテムアイテムコレクション クラス
        /// </summary>
        private class DependentOutputCollection : KeyedCollection<string, IDependentOutput>, IDependentOutputCollection
        {
            private Dictionary<string, Dictionary<string, IDependentOutput>> _userKeyMap
                                            = new Dictionary<string, Dictionary<string, IDependentOutput>>();

            private Dictionary<string, HashSet<IDependentOutputItem>> _outputNameMap
                                    = new Dictionary<string, HashSet<IDependentOutputItem>>(NoCaseStringCompare.Instance);

            /// <summary>
            /// ユーザキーマップIDコレクションを取得します。
            /// </summary>
            public Dictionary<string, Dictionary<string, IDependentOutput>>.KeyCollection UserKeyMapIDs
            {
                get { return _userKeyMap.Keys; }
            }

            /// <summary>
            /// 出力コレクションを取得します。
            /// </summary>
            public ReadOnlyCollection<IDependentOutput> Values
            {
                get { return new ReadOnlyCollection<IDependentOutput>(this); }
            }

            //-----------------------------------------------------------------
            // インデクサ
            //-----------------------------------------------------------------

            /// <summary>
            /// 指定したユーザキーを持つ出力を取得します。
            /// </summary>
            /// <param name="userKeyMapID">ユーザキーマップID。</param>
            /// <param name="userKey">取得する出力のユーザキー。</param>
            /// <returns>指定したユーザキーを持つ出力。</returns>
            public IDependentOutput this[string userKeyMapID, string userKey]
            {
                get { return GetValueFromUserKey(userKeyMapID, userKey); }
            }

            //-----------------------------------------------------------------
            // ユーザキーの操作
            //-----------------------------------------------------------------

            /// <summary>
            /// ユーザキーコレクションを取得します。
            /// </summary>
            /// <param name="userKeyMapID">ユーザキーマップID。</param>
            public Dictionary<string, IDependentOutput>.KeyCollection GetUserKeys(string userKeyMapID)
            {
                return _userKeyMap[userKeyMapID].Keys;
            }

            /// <summary>
            /// 指定したユーザキーマップの出力が含まれているかどうかを確認します。
            /// </summary>
            /// <param name="userKeyMapID">ユーザキーマップID。</param>
            /// <returns>指定したキーを持つ出力を含む場合は true。それ以外の場合は false。</returns>
            public bool ContainsUserKeyMap(string userKeyMapID)
            {
                return _userKeyMap.ContainsKey(userKeyMapID);
            }

            /// <summary>
            /// 指定したユーザキーを持つ出力が含まれているかどうかを確認します。
            /// </summary>
            /// <param name="userKeyMapID">ユーザキーマップID。</param>
            /// <param name="userKey">検索されるユーザキー。</param>
            /// <returns>指定したユーザキーを持つ出力を含む場合は true。それ以外の場合は false。</returns>
            public bool ContainsUserKey(string userKeyMapID, string userKey)
            {
                if (!_userKeyMap.ContainsKey(userKeyMapID)) { return false; }
                if (!_userKeyMap[userKeyMapID].ContainsKey(userKey)) { return false; }

                return (GetValueFromUserKey(userKeyMapID, userKey) != null);
            }

            public bool ContainsOutputName(string outputName)
            {
                return _outputNameMap.ContainsKey(outputName);
            }

            /// <summary>
            /// 既存の出力にユーザキーを設定します。
            /// </summary>
            /// <param name="userKeyMapID">ユーザキーマップID。</param>
            /// <param name="userKey">設定するユーザキー。</param>
            /// <param name="output">既存の出力。</param>
            public void SetUserKey(string userKeyMapID, string userKey, IDependentOutput output)
            {
                if (null == userKeyMapID) { throw new ArgumentNullException("userKeyMapID"); }
                if (0 == userKeyMapID.Length) { throw new ArgumentException("userKeyMapID"); }
                if (null == userKey) { throw new ArgumentNullException("userKey"); }
                if (0 == userKey.Length) { throw new ArgumentException("userKey"); }
                if (null == output) { throw new ArgumentNullException("output"); }

                if (!Contains(output.Key)) { throw new DependencyOutputNotFound(output.Key); }

                // 出力の参照カウントをインクリメントしてユーザキーを追加
                (output as DependentOutput).AddReference();

                if (!_userKeyMap.ContainsKey(userKeyMapID))
                {
                    _userKeyMap.Add(userKeyMapID, new UserKeyDictionary());
                }

                if (!_userKeyMap[userKeyMapID].ContainsKey(userKey))
                {
                    _userKeyMap[userKeyMapID].Add(userKey, output);
                }
                else
                {
                    _userKeyMap[userKeyMapID][userKey] = output;
                }
            }

            /// <summary>
            /// 指定したユーザキーを削除します。
            /// </summary>
            /// <param name="userKeyMapID">ユーザキーマップID。</param>
            /// <param name="userKey">削除する出力のユーザキー。</param>
            public void RemoveByUserKey(string userKeyMapID, string userKey)
            {
                if (null == userKeyMapID) { throw new ArgumentNullException("userKeyMapID"); }
                if (0 == userKeyMapID.Length) { throw new ArgumentException("userKeyMapID"); }
                if (null == userKey) { throw new ArgumentNullException("userKey"); }
                if (0 == userKey.Length) { throw new ArgumentException("userKey"); }

                DependentOutput output = GetValueFromUserKey(userKeyMapID, userKey) as DependentOutput;

                // ユーザキーを削除して出力の参照カウントをデクリメント
                _userKeyMap[userKeyMapID].Remove(userKey);
                output.RemoveReference();

                // 誰からも参照されなくなったら出力を削除する
                if (0 >= output.ReferenceCount)
                {
                    base.Remove(output.Key);
                }
            }

            /// <summary>
            /// 指定したキーの出力が含まれているかどうかを確認します。
            /// </summary>
            /// <param name="key">取得する出力のキー。</param>
            /// <returns>指定したキーを持つ出力を含む場合は true。それ以外の場合は false。</returns>
            bool IDependentOutputCollection.ContainsKey(string key)
            {
                return Contains(key);
            }

            //-----------------------------------------------------------------
            // その他のメソッド
            //-----------------------------------------------------------------

            public void CollectGarbage()
            {
                Collection<UserKey> invalidKeys = new Collection<UserKey>();
                Collection<string> invalidKeyMaps = new Collection<string>();

                foreach (string userKeyMapID in _userKeyMap.Keys)
                {

                    foreach (string userKey in _userKeyMap[userKeyMapID].Keys)
                    {

                        if (GetValueFromUserKey(userKeyMapID, userKey) == null)
                        {
                            invalidKeys.Add(new UserKey(userKeyMapID, userKey));
                        }

                    }

                    if (0 == _userKeyMap[userKeyMapID].Count)
                    {
                        invalidKeyMaps.Add(userKeyMapID);
                    }

                }

                foreach (UserKey userKey in invalidKeys)
                {
                    _userKeyMap[userKey.MapID].Remove(userKey.Key);
                }

                foreach (string mapID in invalidKeyMaps)
                {
                    _userKeyMap.Remove(mapID);
                }
            }

            protected override string GetKeyForItem(IDependentOutput item)
            {
                return item.Key;
            }

            protected override void InsertItem(int index, IDependentOutput item)
            {
                base.InsertItem(index, item);

                foreach (IDependentOutputItem outputItem in item.OutputItems)
                {
                    HashSet<IDependentOutputItem> outputItems = null;

                    if (!_outputNameMap.TryGetValue(outputItem.Name, out outputItems))
                    {
                        outputItems = new HashSet<IDependentOutputItem>();
                        _outputNameMap.Add(outputItem.Name, outputItems);
                    }

                    outputItems.Add(outputItem);
                }

                DependentOutput itemImpl = item as DependentOutput;
                itemImpl.ItemInserting += OnOutputInfoInserting;
                itemImpl.ItemInserted += OnOutputInfoInserted;
                itemImpl.ItemRemoved += OnOutputInfoRemoved;
                itemImpl.ItemClearing += OnOutputInfoClearing;
            }

            protected override void RemoveItem(int index)
            {
                DependentOutput itemImpl = (Items[index] as DependentOutput);

                itemImpl.ItemInserting -= OnOutputInfoInserting;
                itemImpl.ItemInserted -= OnOutputInfoInserted;
                itemImpl.ItemRemoved -= OnOutputInfoRemoved;
                itemImpl.ItemClearing -= OnOutputInfoClearing;

                foreach (IDependentOutputItem outputItem in itemImpl.OutputItems)
                {
                    HashSet<IDependentOutputItem> outputItems = null;

                    if (!_outputNameMap.TryGetValue(outputItem.Name, out outputItems))
                    {
                        continue;
                    }

                    outputItems.Remove(outputItem);

                    if (outputItems.Count == 0)
                    {
                        _outputNameMap.Remove(outputItem.Name);
                    }
                }

                itemImpl.ResetReference();
                base.RemoveItem(index);
            }

            protected override void ClearItems()
            {
                _userKeyMap.Clear();
                _outputNameMap.Clear();
                base.ClearItems();
            }

            /// <summary>
            /// 指定したユーザキーを持つ出力を取得します。
            /// </summary>
            /// <param name="userKeyMapID">ユーザキーマップID。</param>
            /// <param name="userKey">検索されるユーザキー。</param>
            /// <returns>指定したユーザキーを持つ出力。</returns>
            private IDependentOutput GetValueFromUserKey(string userKeyMapID, string userKey)
            {
                IDependentOutput output = _userKeyMap[userKeyMapID][userKey];
                if (!Contains(output)) { return null; }
                return output;
            }

            //-----------------------------------------------------------------
            // イベントハンドラ
            //-----------------------------------------------------------------

            private void OnOutputInfoInserting(object sender, OutputItemEventArgs e)
            {
                HashSet<IDependentOutputItem> outputItems = null;

                if (!_outputNameMap.TryGetValue(e.OutputItem.Name, out outputItems))
                {
                    return;
                }

                if (outputItems.Contains(e.OutputItem)) { throw new Exception(); }
            }

            private void OnOutputInfoInserted(object sender, OutputItemEventArgs e)
            {
                HashSet<IDependentOutputItem> outputItems = null;

                if (!_outputNameMap.TryGetValue(e.OutputItem.Name, out outputItems))
                {
                    outputItems = new HashSet<IDependentOutputItem>();
                    _outputNameMap.Add(e.OutputItem.Name, outputItems);
                }

                outputItems.Add(e.OutputItem);
            }

            private void OnOutputInfoRemoved(object sender, OutputItemEventArgs e)
            {
                HashSet<IDependentOutputItem> outputItems = null;

                if (!_outputNameMap.TryGetValue(e.OutputItem.Name, out outputItems))
                {
                    return;
                }

                outputItems.Remove(e.OutputItem);

                if (outputItems.Count == 0)
                {
                    _outputNameMap.Remove(e.OutputItem.Name);
                }
            }

            private void OnOutputInfoClearing(object sender, EventArgs e)
            {
                DependentOutput itemImpl = sender as DependentOutput;

                foreach (IDependentOutputItem outputItem in itemImpl.OutputItems)
                {
                    _outputNameMap.Remove(outputItem.Name);
                }
            }

            //-----------------------------------------------------------------
            // 隠蔽するメソッド
            //-----------------------------------------------------------------

            private new void Insert(int index, IDependentOutput value)
            {
                // 使用しない
                Debug.Assert(false, "internal error");
            }

            #region ** キーマップ

            public class UserKeyDictionary : Dictionary<string, IDependentOutput> { }

            private struct UserKey
            {
                public string MapID;
                public string Key;

                public UserKey(string mapID, string key)
                {
                    MapID = mapID;
                    Key = key;
                }
            }

            #endregion
        }
    }
}
