﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion
{
    using System;
    using System.Collections.ObjectModel;
    using System.IO;
    using System.Xml;
    using System.Xml.Serialization;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// 依存関係管理クラス
    /// </summary>
    public partial class DependencyManager : IXmlSerializable
    {
        /// <summary>
        /// 出力クラス
        /// </summary>
        private class DependentOutput : IDependentOutput
        {
            #region ** 固定値

            // 要素名
            public const string ElementName = "OutputItem";
            private const string OutputsElementName = "Outputs";
            private const string DependenciesElementName = "Dependencies";

            // 属性名
            private const string KeyAttributeName = "Key";

            #endregion

            private DependencyManager _owner = null;
            private int _referenceCount = 0;
            private string _key = string.Empty;
            private bool _forceDirty = false;
            private bool _mark = false; // 循環参照を防ぐためのマーク付け

            private OutputInfoCollection _outputs = null;
            private DependencyFileInfoCollection _dependencies = null;

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="owner">DependencyOutputItemInfoを所有するDependencyManager</param>
            /// <param name="key">出力を一意に識別するキー</param>
            public DependentOutput(DependencyManager owner, string key)
            {
                if (null == owner) { throw new ArgumentNullException("owner"); }
                if (null == key) { throw new ArgumentNullException("key"); }
                if (0 == key.Length) { throw new ArgumentException("key"); }

                _owner = owner;
                _key = key;

                _outputs = new OutputInfoCollection(this);
                _dependencies = new DependencyFileInfoCollection(this);

                _outputs.ItemInserting += OnOutputInfoInserting;
                _outputs.ItemInserted += OnOutputInfoInserted;
                _outputs.ItemRemoved += OnOutputInfoRemoved;
                _outputs.ItemClearing += OnOutputInfoClearing;
            }

            public event OutputItemEventHandler ItemInserting;
            public event OutputItemEventHandler ItemInserted;
            public event OutputItemEventHandler ItemRemoved;
            public event EventHandler ItemClearing;

            /// <summary>
            /// DependencyManagerを取得します。
            /// </summary>
            public DependencyManager Owner
            {
                get { return _owner; }
            }

            /// <summary>
            /// 参照カウントを取得します。
            /// </summary>
            public int ReferenceCount
            {
                get { return _referenceCount; }
            }

            /// <summary>
            /// キーを取得します。
            /// </summary>
            public string Key
            {
                get { return _key; }
            }

            /// <summary>
            /// 出力情報コレクションを取得します。
            /// </summary>
            public IDependentOutputItemCollection OutputItems
            {
                get { return _outputs; }
            }

            /// <summary>
            /// 依存情報コレクションを取得します。
            /// </summary>
            public IDependedFileInfoCollection Dependencies
            {
                get { return _dependencies; }
            }

            /// <summary>
            /// 出力ファイルの有効性を確認します。
            /// </summary>
            public bool Dirty
            {
                get
                {
                    // HACK : 複数スレッドからコールされるので、ロックしています。
                    //        本来は利用側で対処すべきか？
                    lock (this.OutputItems)
                    {
                        try
                        {
                            if (_mark == true) { return false; } // 循環参照だったら循環前にまかせる
                            _mark = true; // 循環参照を防ぐためマーキング

                            if (_forceDirty) { return true; }

                            // 出力ファイルの存在確認
                            foreach (IDependentOutputItem item in OutputItems)
                            {
                                if (!File.Exists(item.AbsoluteFilePath)) { return true; }
                            }

                            // 依存データの有効性確認
                            foreach (IDependedFileInfo dependencyInfo in _dependencies)
                            {
                                // ハッシュ値が存在し、
                                // キャッシュのハッシュ値と一致する場合は、変更なしとみなします。
                                // キャッシュのハッシュ値と一致しない場合は、Dirty とみなします。
                                if (dependencyInfo.CurrentHashCode != HashCode.Empty)
                                {
                                    if (dependencyInfo.CurrentHashCode == dependencyInfo.LastUpdatedHashCode)
                                    {
                                        continue;
                                    }
                                    else
                                    {
                                        return true;
                                    }
                                }

                                if (dependencyInfo is DependedOutputItemInfo)
                                {
                                    IDependentOutput dependentOutput = (dependencyInfo as DependedOutputItemInfo).Output;

                                    if (null == dependentOutput || dependentOutput.Dirty) { return true; }
                                    _mark = true; // finally でマークがクリアされるのでマーキング（これはしかたない）

                                    foreach (IDependentOutputItem outputItem in OutputItems)
                                    {
                                        foreach (IDependentOutputItem dependentOutputItem in dependentOutput.OutputItems)
                                        {
                                            if (outputItem.LastUpdatedTime.CompareTo(dependentOutputItem.LastUpdatedTime) < 0) { return true; }
                                        }
                                    }
                                }
                                else
                                {
                                    foreach (IDependentOutputItem outputItem in OutputItems)
                                    {
                                        if (!File.Exists(dependencyInfo.AbsoluteFilePath)) { return true; }

                                        if (outputItem.LastUpdatedTime.CompareTo(dependencyInfo.LastUpdatedTime) < 0) { return true; }
                                    }
                                }
                            }

                            return false;
                        }
                        finally
                        {
                            _mark = false; // 必ずマークをクリアするために finally に置く。
                        }
                    }
                }
            }

            /// <summary>
            /// 強制的に無効にする設定を行います。
            /// </summary>
            public bool ForceDirty
            {
                set { _forceDirty = value; }
            }

            /// <summary>
            /// 依存ファイルが存在しているかを確認します。
            /// </summary>
            public bool DependencyFilesExists
            {
                get
                {
                    try
                    {
                        ValidateDependencies();
                    }
                    catch (Exception)
                    {
                        return false;
                    }

                    return true;
                }
            }

            //-----------------------------------------------------------------
            // XML入出力
            //-----------------------------------------------------------------

            /// <summary>
            /// XML要素から出力情報を読み込みます。
            /// </summary>
            /// <param name="owner">DependencyOutputItemInfoを所有するDependencyManager</param>
            /// <param name="xmlElement">XML要素</param>
            /// <returns></returns>
            public static DependentOutput FromXml(DependencyManager owner, XmlElement xmlElement)
            {
                if (null == owner) { throw new ArgumentNullException("owner"); }
                if (null == xmlElement) { throw new ArgumentNullException("xmlElement"); }

                if (ElementName != xmlElement.Name) { throw new DependencyManagerInvalidFileFormatException(); }
                if (!xmlElement.HasAttribute(KeyAttributeName)) { throw new DependencyManagerInvalidFileFormatException(); }

                // Outputs
                XmlElement outputsElement = xmlElement[OutputsElementName];
                if (null == outputsElement) { return null; }

                // Dependencies
                XmlElement dependenciesElement = xmlElement[DependenciesElementName];


                DependentOutput newOutput = new DependentOutput(owner, xmlElement.GetAttribute(KeyAttributeName));

                // OutputInfo
                foreach (XmlElement outputInfoElement in
                    outputsElement.GetElementsByTagName(DependentOutputItem.ElementName))
                {
                    DependentOutputItem newOutputItem = DependentOutputItem.FromXml(outputInfoElement);
                    if (null == newOutputItem)
                    {
                        continue;
                    }

                    newOutput.OutputItems.Add(newOutputItem);
                }

                if (null != dependenciesElement)
                {
                    // DependencyFileInfo
                    foreach (XmlElement dependeFileInfoElement in
                        dependenciesElement.GetElementsByTagName(DependedFileInfo.ElementName))
                    {
                        DependedFileInfo newDependencyInfo = DependedFileInfo.FromXml(dependeFileInfoElement);
                        if (null == newDependencyInfo)
                        {
                            continue;
                        }

                        if (newOutput.Dependencies.Contains(newDependencyInfo.FilePath))
                        {
                            continue;
                        }

                        newOutput.Dependencies.Add(newDependencyInfo);
                    }

                    // DependencyOutputItemInfo
                    foreach (XmlElement dependeOutputItemInfoElement in
                        dependenciesElement.GetElementsByTagName(DependedOutputItemInfo.ElementName))
                    {
                        DependedOutputItemInfo newDependencyInfo = DependedOutputItemInfo.FromXml(dependeOutputItemInfoElement);
                        if (null == newDependencyInfo)
                        {
                            continue;
                        }

                        if (newOutput.Dependencies.Contains(newDependencyInfo.FilePath))
                        {
                            continue;
                        }

                        newOutput.Dependencies.Add(newDependencyInfo);
                    }

                }

                return newOutput;
            }

            /// <summary>
            /// 出力情報をXML出力します。
            /// </summary>
            /// <param name="document">XMLドキュメント</param>
            /// <returns>XML要素</returns>
            public XmlElement ToXml(XmlDocument document)
            {
                if (null == document) { throw new ArgumentNullException("document"); }

                XmlElement outputItemElement = document.CreateElement(ElementName);

                // Key
                outputItemElement.SetAttribute(KeyAttributeName, Key);

                // Outputs
                if (0 < OutputItems.Count)
                {

                    XmlElement outputsElement = document.CreateElement(OutputsElementName);

                    foreach (DependentOutputItem outputInfo in OutputItems)
                    {

                        XmlElement outputInfoElement = outputInfo.ToXml(document);
                        if (null == outputInfoElement) { continue; }

                        outputsElement.AppendChild(outputInfoElement);

                    }

                    if (0 < outputsElement.ChildNodes.Count)
                    {
                        outputItemElement.AppendChild(outputsElement);
                    }

                }

                // Dependencies
                if (0 < _dependencies.Count)
                {

                    XmlElement dependenciesElement = document.CreateElement(DependenciesElementName);

                    foreach (IDependedFileInfo dependencyFileInfo in _dependencies)
                    {

                        XmlElement dependencyFileInfoElement = dependencyFileInfo.ToXml(document);
                        if (null == dependencyFileInfo) { continue; }

                        dependenciesElement.AppendChild(dependencyFileInfoElement);

                    }

                    if (0 < dependenciesElement.ChildNodes.Count)
                    {
                        outputItemElement.AppendChild(dependenciesElement);
                    }

                }

                return (0 < outputItemElement.ChildNodes.Count) ? outputItemElement : null;
            }

            //-----------------------------------------------------------------
            // 参照カウントの操作
            //-----------------------------------------------------------------

            /// <summary>
            /// 参照カウントをインクリメントします。
            /// </summary>
            public void AddReference()
            {
                _referenceCount++;
            }

            /// <summary>
            /// 参照カウントをデクリメントします。
            /// </summary>
            public void RemoveReference()
            {
                if (0 < _referenceCount)
                {

                    _referenceCount--;

                    // 参照がなくなったら、出力ファイルを削除する
                    if (0 < _referenceCount) { return; }

                }

                Clean();
            }

            /// <summary>
            /// 参照カウントをクリアします。
            /// </summary>
            public void ResetReference()
            {
                _referenceCount = 0;
            }

            //-----------------------------------------------------------------
            // クリーン
            //-----------------------------------------------------------------

            /// <summary>
            /// 出力ファイルを削除します。
            /// </summary>
            public void Clean()
            {
                foreach (IDependentOutputItem item in OutputItems)
                {
                    item.DeleteFile();
                }
            }

            //-----------------------------------------------------------------
            // 依存関係のチェック
            //-----------------------------------------------------------------

            /// <summary>
            /// 依存ファイルが存在しているかを確認します。
            /// </summary>
            public void ValidateDependencies()
            {
                // 依存ファイルの存在を確認する
                foreach (IDependedFileInfo dependencyInfo in _dependencies)
                {
                    if (dependencyInfo is DependedOutputItemInfo)
                    {
                        if (null == (dependencyInfo as DependedOutputItemInfo).Output)
                        {
                            throw new DependencyOutputNotFound(dependencyInfo.Key);
                        }

                        continue;

                    }

                    if (!File.Exists(dependencyInfo.AbsoluteFilePath))
                    {
                        throw new DependencyFileNotFoundException(dependencyInfo.AbsoluteFilePath);
                    }

                }
            }

            //-----------------------------------------------------------------
            // イベントハンドラ
            //-----------------------------------------------------------------

            private void OnOutputInfoInserting(object sender, OutputItemEventArgs e)
            {
                if (null != ItemInserting)
                {
                    ItemInserting(this, e);
                }
            }

            private void OnOutputInfoInserted(object sender, OutputItemEventArgs e)
            {
                if (null != ItemInserted)
                {
                    ItemInserted(this, e);
                }
            }

            private void OnOutputInfoRemoved(object sender, OutputItemEventArgs e)
            {
                if (null != ItemRemoved)
                {
                    ItemRemoved(this, e);
                }
            }

            private void OnOutputInfoClearing(object sender, EventArgs e)
            {
                if (null != ItemClearing)
                {
                    ItemClearing(this, e);
                }
            }

            #region ** コレクション

            public class DependencyFileInfoCollection :
                KeyedCollection<string, IDependedFileInfo>,
                IDependedFileInfoCollection
            {
                private DependentOutput _owner = null;

                public DependencyFileInfoCollection(DependentOutput owner)
                {
                    Assertion.Argument.NotNull(owner);
                    _owner = owner;
                }

                protected override string GetKeyForItem(IDependedFileInfo item)
                {
                    return item.Key;
                }

                protected override void InsertItem(int index, IDependedFileInfo item)
                {
                    base.InsertItem(index, item);
                    item.Owner = this._owner.Owner;
                }
            }

            private class OutputInfoCollection : KeyedCollection<string, IDependentOutputItem>, IDependentOutputItemCollection
            {
                private DependentOutput _owner = null;

                public OutputInfoCollection(DependentOutput owner)
                {
                    Assertion.Argument.NotNull(owner);
                    _owner = owner;
                }

                #region ** イベント

                public event OutputItemEventHandler ItemInserting;
                public event OutputItemEventHandler ItemInserted;
                public event OutputItemEventHandler ItemRemoved;
                public event EventHandler ItemClearing;

                #endregion

                #region ** メソッド

                protected override string GetKeyForItem(IDependentOutputItem item)
                {
                    return item.Key;
                }

                protected override void InsertItem(int index, IDependentOutputItem item)
                {
                    if (null != ItemInserting)
                    {
                        ItemInserting(this, new OutputItemEventArgs(item));
                    }

                    base.InsertItem(index, item);
                    item.Owner = this._owner.Owner;

                    if (null != ItemInserted)
                    {
                        ItemInserted(this, new OutputItemEventArgs(item));
                    }
                }

                protected override void RemoveItem(int index)
                {
                    IDependentOutputItem item = Items[index];

                    base.RemoveItem(index);

                    if (null != ItemRemoved)
                    {
                        ItemRemoved(this, new OutputItemEventArgs(item));
                    }
                }

                protected override void ClearItems()
                {
                    if (null != ItemClearing)
                    {
                        ItemClearing(this, new EventArgs());
                    }

                    base.ClearItems();
                }

                #endregion
            }

            #endregion
        }
    }
}
