﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareReport
{
    using System;
    using System.Collections.Generic;
    using System.Xml;
    using Projects;
    using ToolDevelopmentKit;

    internal class HtmlReportExporter
    {
        public HtmlReportExporter()
        {
            this.Title = string.Empty;
            this.Generator = string.Empty;
            this.StyleSheetSource = Resources.FileResourceProxy.HtmlReportCss;
            this.JScriptSource = Resources.FileResourceProxy.HtmlReportJScript;
        }

        public string Title { get; set; }

        public string Generator { get; set; }

        public virtual string StyleSheetSource { get; set; }

        public virtual string JScriptSource { get; set; }

        public void WriteHtmlHeader(XmlTextWriter writer)
        {
            Assertion.Argument.NotNull(writer);

            {
                writer.WriteStartElement("head");

                {
                    writer.WriteStartElement("meta");
                    writer.WriteAttributeString("name", "Author");
                    writer.WriteAttributeString("content", this.Generator);
                    writer.WriteFullEndElement();
                }

                {
                    writer.WriteStartElement("title");
                    writer.WriteString(this.Title);
                    writer.WriteFullEndElement();
                }

                {
                    writer.WriteStartElement("style");
                    writer.WriteAttributeString("type", "text/css");
                    writer.WriteComment(StyleSheetSource);
                    writer.WriteFullEndElement();
                }

                {
                    writer.WriteStartElement("script");
                    writer.WriteAttributeString("type", "text/javascript");
                    writer.WriteComment(JScriptSource);
                    writer.WriteFullEndElement();
                }

                writer.WriteFullEndElement();
            }
        }

        public void WriteFooter(XmlTextWriter writer, string filePath)
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(filePath);

            this.WriteHr(writer);

            {
                writer.WriteStartElement("dl");

                {
                    writer.WriteStartElement("dt");
                    writer.WriteString("FilePath:");
                    writer.WriteFullEndElement();
                }

                {
                    writer.WriteStartElement("dd");
                    writer.WriteString(filePath);
                    writer.WriteFullEndElement();
                }

                {
                    writer.WriteStartElement("dt");
                    writer.WriteString("DateTime:");
                    writer.WriteFullEndElement();
                }

                {
                    writer.WriteStartElement("dd");
                    writer.WriteString(DateTime.Now.ToString("yyyy-MM-ddTHH:mm:ss"));
                    writer.WriteFullEndElement();
                }

                writer.WriteFullEndElement(); // dl
            }
        }

        public void WriteTitle(XmlTextWriter writer)
        {
            writer.WriteStartElement("h1");
            writer.WriteString(Title);
            writer.WriteFullEndElement();
        }

        public void WriteStartBodyDiv(XmlTextWriter writer)
        {
            writer.WriteStartElement("div");
            writer.WriteAttributeString("id", "body");
        }

        public void WriteEndBodyDiv(XmlTextWriter writer)
        {
            writer.WriteEndElement();
        }

        public void WriteShowButton(XmlTextWriter writer, string targetIDPrefix, string index, string caption)
        {
            Ensure.Argument.NotNull(writer);
            Ensure.Argument.NotNull(targetIDPrefix);
            Ensure.Argument.NotNull(caption);

            {
                writer.WriteStartElement("input");

                writer.WriteAttributeString("type", "button");
                writer.WriteAttributeString("class", "tableopenclose");
                writer.WriteAttributeString("onclick", "SwitchDisplay('" + targetIDPrefix + index + "', this)");
                writer.WriteAttributeString("title", "close");
                writer.WriteAttributeString("style", "font-size:10px; width:16px; height:16px; cursor:pointer");
                writer.WriteAttributeString("value", "-");

                writer.WriteFullEndElement();
            }

            writer.WriteString(" ");

            {
                writer.WriteStartElement("a");
                writer.WriteAttributeString("name", index);
                writer.WriteString(caption);
                writer.WriteFullEndElement(); // a
            }
        }

        public void WriteItemList<TItem>(
            XmlTextWriter writer, IListInfoProvider<TItem> listInfoProvider,
            IListItemInfoProvider<TItem> itemInfoProvider)
        {
            Ensure.Argument.NotNull(writer);
            Ensure.Argument.NotNull(listInfoProvider);
            Ensure.Argument.NotNull(itemInfoProvider);

            {
                writer.WriteStartElement("table");
                writer.WriteAttributeString("width", "100%");

                this.WriteItemListHeader<TItem>(writer, listInfoProvider);
                this.WriteItemListBody<TItem>(writer, listInfoProvider, itemInfoProvider);

                writer.WriteFullEndElement();
            }
        }

        public void WriteHr(XmlTextWriter writer)
        {
            Assertion.Argument.NotNull(writer);

            writer.WriteStartElement("hr");
            writer.WriteFullEndElement();
        }

        private void WriteItemListHeader<TItem>(
            XmlTextWriter writer, IListInfoProvider<TItem> listInfoProvider)
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(listInfoProvider);

            {
                writer.WriteStartElement("thead");

                {
                    writer.WriteStartElement("tr");

                    bool firstColumn = true;
                    foreach (Column column in listInfoProvider.Columns)
                    {
                        writer.WriteStartElement("th");

                        if (column.IsRaw == true)
                        {
                            writer.WriteAttributeString("nowrap", string.Empty);
                        }

                        if (firstColumn)
                        {
                            writer.WriteAttributeString("class", "SortKey");
                        }

                        writer.WriteAttributeString("onclick", "SortTable('" + column.SortType.ToText() + "', this)");
                        writer.WriteAttributeString("title", "sort");
                        writer.WriteAttributeString("style", "cursor:pointer;");
                        if (column.IsRaw == true)
                        {
                            writer.WriteRaw(column.Text);
                        }
                        else
                        {
                            writer.WriteString(column.Text);
                        }

                        writer.WriteFullEndElement();

                        firstColumn = false;
                    }

                    writer.WriteFullEndElement();
                }

                writer.WriteFullEndElement();
            }
        }

        private void WriteItemListBody<TItem>(
            XmlTextWriter writer, IListInfoProvider<TItem> listInfoProvider,
            IListItemInfoProvider<TItem> itemInfoProvider)
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(listInfoProvider);
            Assertion.Argument.NotNull(itemInfoProvider);

            {
                writer.WriteStartElement("tbody");

                int index = 0;
                foreach (TItem item in listInfoProvider.Items)
                {
                    writer.WriteStartElement("tr");

                    string className = this.GetListItemClassName(item, itemInfoProvider);

                    if (className.Length > 0)
                    {
                        writer.WriteAttributeString("class", className);
                    }

                    bool isEnabled = itemInfoProvider.IsEnabled(item);

                    foreach (Column column in listInfoProvider.Columns)
                    {
                        writer.WriteStartElement("td");

                        string text = itemInfoProvider.GetText(
                            column, isEnabled ? index : -1, item);

                        if (0 < text.Length)
                        {
                            writer.WriteString(text);
                        }
                        else
                        {
                            writer.WriteEntityRef("nbsp");
                        }

                        writer.WriteFullEndElement();
                    }

                    writer.WriteFullEndElement();

                    // アイテムが無効の場合はインデックスを進めません。
                    if (isEnabled)
                    {
                        index++;
                    }
                }

                writer.WriteFullEndElement();
            }
        }

        private string GetListItemClassName<TItem>(
            TItem item, IListItemInfoProvider<TItem> itemInfoProvider)
        {
            Assertion.Argument.NotNull(item);

            string className = string.Empty;

            if (!itemInfoProvider.IsEnabled(item))
            {
                className += "Disabled";
            }

            int colorIndex = itemInfoProvider.GetColorIndex(item);
            if (colorIndex > 0)
            {
                if (className.Length > 0)
                {
                    className += " ";
                }
                className += "ColorType" + colorIndex.ToString();
            }

            return className;
        }
    }
}
