﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;

namespace NintendoWare.SoundFoundation.Commands
{
    /// <summary>
    /// コマンド機能を提供します。
    /// </summary>
    public class CommandService
    {
        private CommandCollection _commands = new CommandCollection();    // コマンドコレクション
        private KeyBindingManager _keyBindingManager = new KeyBindingManager();    // キーバインディングマネージャ

        /// <summary>
        /// コマンドの一覧を取得します。
        /// </summary>
        public ICommandCollection Commands
        {
            get { return _commands; }
        }

        /// <summary>
        /// 第１キーの一覧を取得します。
        /// </summary>
        public IEnumerable<KeyStroke> FirstKeys
        {
            get { return _keyBindingManager.FirstKeys; }
        }

        /// <summary>
        /// キーバインディングの一覧を取得します。
        /// </summary>
        public IEnumerable<KeyBinding> KeyBindings
        {
            get { return _keyBindingManager; }
        }

        /// <summary>
        /// コマンドを追加します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        public void AddCommand(Command command)
        {
            if (null == command) { throw new ArgumentNullException("command"); }

            if (ContainsCommand(command))
            {
                return;
            }

            _commands.Add(command);
        }

        /// <summary>
        /// コマンドを削除します。
        /// </summary>
        /// <param name="commandUri">コマンド。</param>
        public void RemoveCommand(Command command)
        {
            RemoveCommand(command.Uri);
        }

        /// <summary>
        /// 指定URIをもつコマンドを削除します。
        /// </summary>
        /// <param name="commandUri">コマンドURI。</param>
        public void RemoveCommand(string commandUri)
        {
            if (!_commands.Contains(commandUri)) { return; }
            _commands.Remove(commandUri);
        }

        /// <summary>
        /// 同じURIのコマンドが既に存在しているかどうか調べます。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>存在している場合は true、存在しない場合は false。</returns>
        public bool ContainsCommand(Command command)
        {
            if (null == command) { throw new ArgumentNullException("command"); }
            return _commands.Contains(command.Uri);
        }

        /// <summary>
        /// 指定キーに関連付けられたコマンドを検索します。
        /// </summary>
        /// <param name="firstKey">第１キー。</param>
        /// <param name="secondKey">第２キー。</param>
        /// <returns>コマンド。</returns>
        public Command FindCommand(KeyStroke firstKey, KeyStroke secondKey)
        {
            return _keyBindingManager.Find(firstKey, secondKey);
        }

        /// <summary>
        /// キーバインディングを追加します。
        /// </summary>
        /// <param name="secondKey">第２キー。</param>
        /// <param name="command">コマンド。</param>
        public void AddKeyBinding(KeyStroke secondKey, Command command)
        {
            AddKeyBinding(KeyStroke.NullKeyStroke, secondKey, command);
        }

        /// <summary>
        /// キーバインディングを追加します。
        /// </summary>
        /// <param name="firstKey">第１キー。</param>
        /// <param name="secondKey">第２キー。</param>
        /// <param name="command">コマンド。</param>
        public void AddKeyBinding(KeyStroke firstKey, KeyStroke secondKey, Command command)
        {
            _keyBindingManager.Add(firstKey, secondKey, command);
        }

        /// <summary>
        /// キーバインディングを削除します。
        /// </summary>
        /// <param name="secondKey">第２キー。</param>
        public void RemoveKeyBinding(KeyStroke secondKey)
        {
            RemoveKeyBinding(KeyStroke.NullKeyStroke, secondKey);
        }

        /// <summary>
        /// キーバインディングを削除します。
        /// </summary>
        /// <param name="firstKey">第１キー。</param>
        /// <param name="secondKey">第２キー。</param>
        public void RemoveKeyBinding(KeyStroke firstKey, KeyStroke secondKey)
        {
            _keyBindingManager.Remove(firstKey, secondKey);
        }

        /// <summary>
        /// 第１キーかどうかを確認します。
        /// </summary>
        /// <param name="key">キー。</param>
        /// <returns>指定キーが第１キーの場合は true、それ以外の場合は false。</returns>
        public bool IsFirstKey(KeyStroke key)
        {
            return _keyBindingManager.ContainsKey(key);
        }

        /// <summary>
        /// 第１キーの後に続くキーバインディングの IEnumerable を取得します。
        /// </summary>
        /// <param name="firstKey">第１キー。</param>
        /// <returns>キーバインディングの IEnumerable。</returns>
        public IEnumerable<KeyBinding> FindKeyBinding(KeyStroke firstKey)
        {
            return _keyBindingManager.FindKeyBindings(firstKey);
        }
    }
}
