﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;

namespace NintendoWare.SoundFoundation.Commands
{
    /// <summary>
    /// キー入力を処理して対応するコマンドを取得します。
    /// </summary>
    public class CommandKeyProcessor
    {
        private CommandService _commandService;
        private KeyStroke _firstKey = KeyStroke.NullKeyStroke;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="commandService">コマンドサービス。</param>
        public CommandKeyProcessor(CommandService commandService)
        {
            if (null == commandService) { throw new ArgumentNullException("commandService"); }
            _commandService = commandService;
        }

        /// <summary>
        /// 入力された第１キーを取得します。
        /// </summary>
        public KeyStroke FirstKey
        {
            get { return _firstKey; }
            private set
            {
                if (value == _firstKey) { return; }

                _firstKey = value;
                OnFirstKeyChanged(new EventArgs());
            }
        }

        /// <summary>
        /// 入力された第１キーが変更されると発生します。
        /// </summary>
        public event EventHandler FirstKeyChanged;

        /// <summary>
        /// キー入力を処理します。
        /// </summary>
        /// <param name="key">入力されたキー。</param>
        /// <returns>対応するコマンド。</returns>
        public Command InputKey(KeyStroke key)
        {
            if (null == key) { throw new ArgumentNullException("key"); }

            if (KeyStroke.NullKeyStroke == _firstKey)
            {

                if (_commandService.IsFirstKey(key))
                {
                    FirstKey = key;
                    return null;
                }

            }

            Command command = _commandService.FindCommand(_firstKey, key);
            Reset();

            return command;
        }

        /// <summary>
        /// 入力状態をリセットします。
        /// </summary>
        public void Reset()
        {
            FirstKey = KeyStroke.NullKeyStroke;
        }

        protected virtual void OnFirstKeyChanged(EventArgs e)
        {
            if (null != FirstKeyChanged)
            {
                FirstKeyChanged(this, e);
            }
        }
    }
}
