﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Commands
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Drawing;

    /// <summary>
    /// コマンドの基本クラス。
    /// </summary>
    public class Command
    {
        private CommandUri _uri;

        private string _name = string.Empty;   // ローカライズされたコマンド名
        private string _text = string.Empty;   // ローカライズされたコマンドテキスト
        private string _toolTipText = string.Empty;   // ローカライズされたツールチップテキスト
        private Image _image = null;           // コマンドのイメージ
        private string _shortcutKeyText = string.Empty;   // ショートカットキーテキスト

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="uri">コマンドURI。</param>
        /// <param name="name">ローカライズされたコマンド名。</param>
        public Command(string uri, string name)
        {
            if (null == uri) { throw new ArgumentNullException("uri"); }
            if (0 == uri.Length) { throw new ArgumentException("uri.Length must not be 0."); }
            if (null == name) { throw new ArgumentNullException("name"); }

            _uri = new CommandUri(uri);
            _name = name;
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="command">コマンド</param>
        /// <param name="parameters">コマンドのパラメータ</param>
        public Command(Command command, IEnumerable<KeyValuePair<string, string>> parameters)
        {
            Debug.Assert(command != null);
            Debug.Assert(parameters != null);

            _uri = CommandUri.Create(command.ID, parameters);
            _name = command.Name;
        }

        /// <summary>
        /// コマンドURI を取得します。
        /// </summary>
        public string Uri
        {
            get { return _uri.Uri; }
        }

        /// <summary>
        /// コマンドID を取得します。
        /// </summary>
        public string ID
        {
            get { return _uri.ID; }
        }

        /// <summary>
        /// ローカライズされたコマンド名を取得します。
        /// </summary>
        public string Name
        {
            get { return _name; }
        }

        /// <summary>
        /// ローカライズされたコマンドテキストを取得または設定します。
        /// </summary>
        public string Text
        {
            get { return _text; }
            set
            {
                if (null == value) { throw new ArgumentNullException("value"); }
                _text = value;
            }
        }

        /// <summary>
        /// ローカライズされたツールチップテキストを取得または設定します。
        /// </summary>
        public string ToolTipText
        {
            get { return _toolTipText; }
            set
            {
                if (null == value) { throw new ArgumentNullException("value"); }
                _toolTipText = value;
            }
        }

        /// <summary>
        /// コマンドのイメージを取得または設定します。
        /// </summary>
        public Image Image
        {
            get { return _image; }
            set { _image = value; }
        }

        /// <summary>
        /// ショートカットキーテキストを取得または設定します。
        /// </summary>
        public string ShortcutKeyText
        {
            get { return _shortcutKeyText; }
            set
            {
                if (null == value) { throw new ArgumentNullException("value"); }
                _shortcutKeyText = value;
            }
        }

        /// <summary>
        /// 固定のコマンドパラメータを取得します。
        /// </summary>
        /// <returns>固定のコマンドパラメータ。</returns>
        public IQueryCommandParameter Parameters
        {
            get { return _uri.Parameters; }
        }
    }
}
