﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Codecs
{
    using System;
    using System.IO;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// 指定ストリームの内容を読み込むフィルタです。
    /// </summary>
    public class SourceStreamReader : Filter
    {
        private Stream source;
        private WaveFormat sourceFormat;

        private bool isEndOfStream = false;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SourceStreamReader()
            : base(new InputConnectionPoint[0])
        {
            this.SamplesPerPacket = 0;
        }

        /// <summary>
        /// ストリーム終端に達したかどうかを調べます。
        /// </summary>
        public override bool IsEndOfStream
        {
            get { return this.isEndOfStream; }
        }

        /// <summary>
        /// ソースストリームを指定します。
        /// </summary>
        public Stream Source
        {
            get { return this.source; }
            set
            {
                if (this.source == value)
                {
                    return;
                }

                this.source = value;
                this.UpdateOutputs();
            }
        }

        /// <summary>
        /// ソースストリームのフォーマットを指定します。
        /// </summary>
        public WaveFormat SourceFormat
        {
            get { return this.sourceFormat; }
            set
            {
                if (this.SourceFormat.Equals(value))
                {
                    return;
                }

                this.sourceFormat = value;
                this.UpdateOutputs();
            }
        }

        /// <summary>
        /// パケットに含めるサンプル数を取得または設定します。
        /// </summary>
        public int SamplesPerPacket { get; set; }

        /// <summary>
        /// 処理を開始します。
        /// </summary>
        public void Run()
        {
            int sampleSize = this.GetSampleSizePerPacket();

            this.isEndOfStream = false;

            while (true)
            {
                byte[] samples = new byte[sampleSize];
                int readSize = this.Source.Read(samples, 0, sampleSize);

                if (readSize == 0)
                {
                    this.isEndOfStream = true;
                    this.SendToAll(Packet.CreateEndOfStreamPacket());
                    break;
                }

                if (readSize < sampleSize)
                {
                    byte[] newSamples = new byte[readSize];
                    Array.Copy(samples, newSamples, readSize);

                    this.SendToAll(Packet.Create(newSamples));
                }
                else
                {
                    this.SendToAll(Packet.Create(samples));
                }
            }
        }

        /// <summary>
        /// パケットに含めるサンプルのサイズを取得します。
        /// </summary>
        /// <returns>パケットに含めるサンプルのサイズを返します。</returns>
        private int GetSampleSizePerPacket()
        {
            if (this.SamplesPerPacket > 0)
            {
                return this.SamplesPerPacket * this.SourceFormat.SampleLength;
            }

            // 4000 サンプル分
            return this.SourceFormat.FrameLength * 4000;
        }

        /// <summary>
        /// 出力ストリームの長さを取得します。
        /// </summary>
        /// <returns>出力ストリームの長さを返します。</returns>
        private int GetOutputLength()
        {
            return (this.Source == null) ? 0 : (int)this.Source.Length;
        }

        /// <summary>
        /// 出力ストリームのフレーム数を取得します。
        /// </summary>
        /// <returns>出力ストリームのフレーム数を返します。</returns>
        private int GetOutputFrameCount()
        {
            if (this.Source == null)
            {
                return 0;
            }

            if (this.SourceFormat.FrameLength == 0)
            {
                return 0;
            }

            Ensure.Operation.True(this.SourceFormat.FrameLength > 0);
            return (int)this.Source.Length / this.SourceFormat.FrameLength;
        }

        /// <summary>
        /// 出力接続ポイントを更新します。
        /// </summary>
        private void UpdateOutputs()
        {
            this.SetOutputs(
                new OutputConnectionPoint[]
                {
                    new OutputConnectionPoint(
                        new WaveStreamInformation(
                            this.GetOutputLength(),
                            this.GetOutputFrameCount(),
                            this.SourceFormat)
                        )
                });
        }
    }
}
