﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Codecs
{
    using System;
    using System.Collections.Generic;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// サンプルを溜め込み連結するフィルタです。
    /// </summary>
    public class SampleSpooler : Filter
    {
        private byte[] spoolBuffer = null;
        private int spooledSize = 0;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SampleSpooler()
            : base(new InputConnectionPoint())
        {
            this.SpoolSize = 0;
        }

        /// <summary>
        /// パケットに溜めるサイズを取得または設定します。
        /// </summary>
        public int SpoolSize { get; set; }

        /// <summary>
        /// 出力接続ポイントを生成します。
        /// </summary>
        /// <returns>出力接続ポイントの配列を返します。</returns>
        protected override OutputConnectionPoint[] CreateOutputs()
        {
            return new OutputConnectionPoint[]
            {
                new OutputConnectionPoint(this.Inputs[0].StreamInformation),
            };
        }

        /// <summary>
        /// パケットを処理します。
        /// </summary>
        /// <param name="input">入力接続ポイントを指定します。</param>
        /// <param name="packet">パケットを指定します。</param>
        protected override void Process(InputConnectionPoint input, Packet packet)
        {
            Assertion.Argument.NotNull(packet);

            Packet[] sendPackets = this.Spool(packet);

            foreach (Packet sendPacket in sendPackets)
            {
                this.SendToAll(sendPacket);
            }
        }

        /// <summary>
        /// 指定パケットをスプールします。
        /// </summary>
        /// <param name="packet">パケットを指定します。</param>
        /// <returns>スプールバッファが埋まった場合、送信するパケットの配列を返します。</returns>
        protected Packet[] Spool(Packet packet)
        {
            Assertion.Argument.NotNull(packet);

            List<Packet> packets = new List<Packet>();
            int current = 0;

            while (current < packet.ValidSamplesLength)
            {
                var spoolBuffer = this.GetSpoolBuffer();

                if (spoolBuffer == null || spoolBuffer.Length == 0)
                {
                    // ここにくることはないはず。
                    throw new InvalidOperationException();
                }

                int writeSize = System.Math.Min(
                    spoolBuffer.Length - this.spooledSize,
                    packet.ValidSamplesLength - current);

                Array.Copy(packet.Samples, current, spoolBuffer, this.spooledSize, writeSize);
                current += writeSize;
                this.spooledSize += writeSize;

                if (this.spooledSize == spoolBuffer.Length)
                {
                    packets.Add(Packet.Create(spoolBuffer));
                    this.ResetSpoolBuffer();
                }
            }

            return packets.ToArray();
        }

        private byte[] GetSpoolBuffer()
        {
            if (this.spoolBuffer != null)
            {
                return this.spoolBuffer;
            }

            int spoolSize = this.SpoolSize;

            if (spoolSize == 0 ||
                spoolSize > this.Inputs[0].StreamInformation.Length)
            {
                spoolSize = this.Inputs[0].StreamInformation.Length;
            }

            if (spoolSize > 0)
            {
                this.spoolBuffer = new byte[spoolSize];
            }

            return this.spoolBuffer;
        }

        /// <summary>
        /// スプールバッファをリセットします。
        /// </summary>
        private void ResetSpoolBuffer()
        {
            this.spoolBuffer = null;
            this.spooledSize = 0;
        }
    }
}
