﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Binarization
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using Core.IO;
    using NintendoWare.ToolDevelopmentKit;
    using NintendoWare.ToolDevelopmentKit.ComponentModel;

    public class ObjectWriter
    {
        private delegate void WriteObjectHandler(BinaryWriter writer, object obj);

        private static int bufferSize = 1024;

        private static Dictionary<Type, WriteObjectHandler> _handlers =
            new Dictionary<Type, WriteObjectHandler>();

        private BinaryWriter writer;

        //-----------------------------------------------------------------

        static ObjectWriter()
        {
            Initialize();
        }

        public ObjectWriter(BinaryWriter writer)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }
            this.writer = writer;
        }

        //-----------------------------------------------------------------

        public long Position
        {
            get { return this.writer.BaseStream.Position; }
            set { this.writer.BaseStream.Position = value; }
        }

        //-----------------------------------------------------------------

        public static bool IsTypeSupported(Type type)
        {
            return GetTargetType(type) != null;
        }

        public void Write(object obj)
        {
            if (null == obj) { throw new ArgumentNullException("obj"); }

            Type type = GetTargetType(obj.GetType());
            if (null == type)
            {
                throw new ArgumentException("unsupported type.");
            }

            _handlers[type](this.writer, obj);
        }

        public void Align(int alignment)
        {
            this.writer.Align(alignment);
        }

        private static void Initialize()
        {
            _handlers.Add(typeof(Byte), (BinaryWriter writer, object obj) => writer.Write((Byte)obj));
            _handlers.Add(typeof(Address), (BinaryWriter writer, object obj) => writer.Write((UInt32)(Address)obj));
            _handlers.Add(typeof(Int16), (BinaryWriter writer, object obj) => writer.Write((Int16)obj));
            _handlers.Add(typeof(Int32), (BinaryWriter writer, object obj) => writer.Write((Int32)obj));
            _handlers.Add(typeof(Int64), (BinaryWriter writer, object obj) => writer.Write((Int64)obj));
            _handlers.Add(typeof(UInt16), (BinaryWriter writer, object obj) => writer.Write((UInt16)obj));
            _handlers.Add(typeof(UInt32), (BinaryWriter writer, object obj) => writer.Write((UInt32)obj));
            _handlers.Add(typeof(UInt64), (BinaryWriter writer, object obj) => writer.Write((UInt64)obj));
            _handlers.Add(typeof(Single), (BinaryWriter writer, object obj) => writer.Write((Single)obj));
            _handlers.Add(typeof(Boolean), (BinaryWriter writer, object obj) => writer.Write((bool)obj));
            _handlers.Add(typeof(String), (BinaryWriter writer, object obj) => writer.Write((obj as string).ToCharArray()));
            _handlers.Add(typeof(Stream), WriteStream);
            _handlers.Add(typeof(IFileObject), WriteFile);
        }

        private static Type GetTargetType(Type type)
        {
            foreach (Type supportedType in _handlers.Keys)
            {
                if (type.IsSupported(supportedType))
                {
                    return supportedType;
                }
            }

            return null;
        }

        private static void WriteStream(BinaryWriter writer, object obj)
        {
            Assertion.Argument.NotNull(writer);

            Stream stream = obj as Stream;
            if (null == stream) { throw new Exception("internal error"); }

            Byte[] buffer = new Byte[bufferSize];

            while (true)
            {
                int read = stream.Read(buffer, 0, bufferSize);
                if (0 >= read) { break; }

                writer.Write(buffer, 0, read);
            }
        }

        private static void WriteFile(BinaryWriter writer, object obj)
        {
            Assertion.Argument.NotNull(writer);

            IFileObject file = obj as IFileObject;
            if (null == file) { throw new Exception("internal error"); }

            Byte[] buffer = new Byte[bufferSize];

            using (Stream stream = file.OpenRead())
            {
                while (true)
                {
                    int read = stream.Read(buffer, 0, bufferSize);
                    if (0 >= read) { break; }

                    writer.Write(buffer, 0, read);
                }
            }
        }
    }
}
