﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Binarization
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Reflection;
    using NintendoWare.SoundFoundation.Core.Reflection;
    using NintendoWare.ToolDevelopmentKit;
    using NintendoWare.ToolDevelopmentKit.Reflection;

    public class DomBuilder
    {
        private AccessorManager accessorManager;

        // Context
        private HashSet<object> stackObjects;

        public DomBuilder()
        {
        }

        public DomBuilder(AccessorManager accessorManager)
        {
            Ensure.Argument.NotNull(accessorManager);
            this.accessorManager = accessorManager;
        }

        public DomElement Build(object obj)
        {
            Ensure.Argument.NotNull(obj);

            if (null == this.accessorManager)
            {
                this.accessorManager = new AccessorManager();
            }

            try
            {
                this.accessorManager.TypeFilter += OnTypeFilter;
                this.stackObjects = new HashSet<object>();

                return this.BuildInternal(obj);
            }
            finally
            {
                this.accessorManager.TypeFilter -= OnTypeFilter;
                this.stackObjects = null;
            }
        }

        private DomElement BuildInternal(object obj)
        {
            Assertion.Argument.NotNull(obj);

            // オブジェクト出力可能な場合は DomElement を作成しません。
            if (ObjectWriter.IsTypeSupported(obj.GetType()))
            {
                return null;
            }

            ClassObject classObject = this.accessorManager.CreateMemberObject(obj);

            if (null == classObject)
            {
                throw new Exception("unsupported class type.");
            }

            // class or struct 以外の場合はDomElementを作成しません。
            if (!(classObject.Accessor.Info.IsClass || classObject.Accessor.Info.IsStruct()))
            {
                return null;
            }

            // class の場合は循環参照のチェックを行います。
            if (classObject.Accessor.Info.IsClass && this.stackObjects.Contains(obj))
            {
                throw new Exception("internal error : obj exist in stackObjects.");
            }

            this.stackObjects.Add(obj);

            DomElement element = new DomElement(obj);
            BuildDomElement(element, classObject);

            this.stackObjects.Remove(obj);

            ExtractDomAttributes(element, classObject);

            return element;
        }

        //-----------------------------------------------------------------
        // DOM 要素の作成
        //-----------------------------------------------------------------

        private void BuildDomElement(DomElement element, ClassObject classObject)
        {
            Assertion.Argument.NotNull(element);
            Assertion.Argument.NotNull(classObject);

            // 基本クラスを先にビルドします。
            if (classObject.Base != null)
            {
                BuildDomElement(element, classObject.Base);
            }

            DomElementAttribute elementAttr = classObject.Accessor.GetAttribute<DomElementAttribute>();
            if (elementAttr != null)
            {
                element.Tags = elementAttr.Tags;
            }

            BuildDomElementFields(element, classObject);
        }

        private void BuildDomElementFields(DomElement element, ClassObject classObject)
        {
            Assertion.Argument.NotNull(element);
            Assertion.Argument.NotNull(classObject);

            foreach (MemberObject memberObject in classObject.MemberObjects)
            {
                if (!(memberObject is PropertyObject)) { continue; }

                PropertyObject propertyObject = memberObject as PropertyObject;
                if (propertyObject == null) { continue; }

                // 無効フィールド属性があればスキップします。
                if (propertyObject.Accessor.HasAttribute<DomIgnoreFieldAttribute>())
                {
                    continue;
                }

                // オーバーライドプロパティの場合は、
                // 基本クラスで既に処理済なのでスキップします。
                if (propertyObject.Accessor.IsOverride)
                {
                    continue;
                }

                DomFieldAttribute fieldAttr = propertyObject.Accessor.GetAttribute<DomFieldAttribute>();
                DomElementField field = null;

                // 列挙可能な場合は、新しくDOM要素を作成し、それ以下にDOMフィールドを展開します。
                if (propertyObject.Accessor.IsEnumerable)
                {
                    field = CreateDomElementFieldExpandedFields(propertyObject);
                }
                else
                {
                    field = CreateDomElementField(fieldAttr, propertyObject);
                }

                if (field == null)
                {
                    continue;
                }

                field.Name = CreateFieldName(element, fieldAttr);

                element.Children.Add(field);
            }
        }

        //-----------------------------------------------------------------
        // DOM フィールドの作成
        //-----------------------------------------------------------------

        private DomElementField CreateDomElementField(
            DomFieldBaseAttribute attr, PropertyObject propertyObject)
        {
            return CreateDomElementField(
                attr,
                propertyObject,
                this.GetPropertyValue(propertyObject));
        }

        private DomElementField CreateDomElementField(
            DomFieldBaseAttribute attr, PropertyObject propertyObject, object value)
        {
            Assertion.Argument.NotNull(propertyObject);

            DomElementField field = CreateDomElementFieldInstance(attr, propertyObject, value);
            if (field == null) { return null; }

            ExtractDomAttributes(field, propertyObject);

            return field;
        }

        private DomElementField CreateDomElementFieldExpandedFields(PropertyObject propertyObject)
        {
            Assertion.Argument.NotNull(propertyObject);

            DomElementField field =
                new DomElementField(this.CreateDomElementExpandedFields(propertyObject));

            ExtractDomAttributes(field, propertyObject);

            return field;
        }

        private DomElement CreateDomElementExpandedFields(PropertyObject propertyObject)
        {
            Debug.Assert(null != propertyObject, "invalid argument");

            EnumeratedFieldAttribute enumFieldAttr =
                propertyObject.Accessor.GetAttribute<EnumeratedFieldAttribute>();


            // 列挙されたDOMフィールドの親となるDOM要素を作成します。
            DomElement fieldElement = new DomElement(this.GetPropertyValue(propertyObject));

            // フィールドの値を列挙して追加します。
            foreach (object value in propertyObject.ExtractValues())
            {
                if (null == value) { continue; }

                DomElementField field = CreateDomElementField(enumFieldAttr, propertyObject, value);
                if (field == null) { continue; }

                field.Name = CreateEnumFieldName(fieldElement, enumFieldAttr, value);

                fieldElement.Children.Add(field);
            }

            return fieldElement;
        }

        private DomElementField CreateDomElementFieldInstance(
            DomFieldBaseAttribute attr, PropertyObject propertyObject, object value)
        {
            Assertion.Argument.NotNull(propertyObject);

            // 参照フィールド、またはオプションフィールドの場合は null を許容します。
            // それ以外の場合は、 null を許容しません。
            if (IsObjectReference(propertyObject))
            {
                return new DomElementField(value);
            }
            if (null == value)
            {
                if (null != attr && attr.IsOptional)
                {
                    return null;
                }

                throw new ArgumentException(
                    string.Format("{0}.Value must not be null.", propertyObject.Accessor.Name)
                    );
            }

            if (this.IsDomElementTarget(propertyObject, value))
            {
                DomElement fieldElement = this.BuildInternal(value);

                if (null != fieldElement)
                {
                    return new DomElementField(fieldElement);
                }
            }

            return new DomElementField(value);
        }

        /// <summary>
        /// DOM要素の対象かどうかを調べます。
        /// </summary>
        /// <remarks>
        /// クラス、構造体、インターフェイスは対象とし、再帰的にDOM要素としてツリー展開します。
        /// ただし、文字列、ストリームの場合は除きます。
        /// </remarks>
        /// <param name="propertyObject">プロパティ値を指定します。</param>
        /// <param name="value">
        /// 値を指定します。
        /// propertyObjectが列挙子の場合は、その列挙値を指定します。
        /// </param>
        /// <returns>DOM要素の対象である場合は true、そうでない場合は false を返します。</returns>
        private bool IsDomElementTarget(PropertyObject propertyObject, object value)
        {
            Assertion.Argument.NotNull(propertyObject);
            Assertion.Argument.NotNull(value);

            if (value is string || value is Stream)
            {
                return false;
            }

            return propertyObject.Accessor.ElementValueType.IsClass ||
                propertyObject.Accessor.ElementValueType.IsStruct() ||
                propertyObject.Accessor.ElementValueType.IsInterface;
        }

        //-----------------------------------------------------------------
        // DOM属性の抽出
        //-----------------------------------------------------------------

        private void ExtractDomAttributes(DomObject domObject, MemberObject memberObject)
        {
            Assertion.Argument.NotNull(domObject);
            Assertion.Argument.NotNull(memberObject);

            foreach (Attribute attr in memberObject.Accessor.Attributes)
            {
                domObject.Attributes.Add(attr);
            }
        }

        //-----------------------------------------------------------------
        // その他のユーティリティメソッド
        //-----------------------------------------------------------------

        private bool IsObjectReference(PropertyObject propertyObject)
        {
            Assertion.Argument.NotNull(propertyObject);
            return propertyObject.Accessor.HasAttribute<DomObjectReferenceAttribute>();
        }

        private string CreateFieldName(DomElement parent, DomFieldAttribute attr)
        {
            Assertion.Argument.NotNull(parent);

            if (attr == null)
            {
                return parent.Children.Count.ToString();
            }

            if (attr.FieldNameProvider.Name.Length == 0)
            {
                return parent.Children.Count.ToString();
            }

            return attr.FieldNameProvider.Name;
        }

        private string CreateEnumFieldName(
            DomElement parent, EnumeratedFieldAttribute attr, object value)
        {
            Assertion.Argument.NotNull(parent);

            if (attr == null)
            {
                return parent.Children.Count.ToString();
            }

            string name = attr.FieldNameProvider.GetNameFromValue(parent, value);
            if (name.Length == 0)
            {
                return parent.Children.Count.ToString();
            }

            return name;
        }

        /// <summary>
        /// プロパティオブジェクトの値を取得します。
        /// このメソッドは、TargetInvocationException を catch して
        /// InnerException を throw しなおします。
        /// </summary>
        /// <param name="propertyObject">プロパティオブジェクトを指定します。</param>
        /// <returns>プロパティオブジェクトの値を返します。</returns>
        private object GetPropertyValue(PropertyObject propertyObject)
        {
            Assertion.Argument.NotNull(propertyObject);

            try
            {
                return propertyObject.Value;
            }
            catch (TargetInvocationException exception)
            {
                if (exception.InnerException != null)
                {
                    throw exception.InnerException;
                }

                throw exception;
            }
        }

        //-----------------------------------------------------------------
        // イベントハンドラ
        //-----------------------------------------------------------------

        private void OnTypeFilter(object sender, TypeFilterEventArgs e)
        {
            Ensure.Argument.NotNull(sender);
            Ensure.Argument.NotNull(e);

            // System 名前空間のオブジェクトは対象にできません。
            if (e.TargetType.FullName.StartsWith("System."))
            {
                e.Cancel = true;
                return;
            }
        }
    }
}
