﻿namespace Opal.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Collections.Specialized;
    using System.Diagnostics;
    using System.Linq;
    using System.Linq.Expressions;
    using System.Runtime.CompilerServices;
    using System.Text;
    using System.Threading;
    using System.Threading.Tasks;
    using Opal.ComponentModel;
    using Opal.Events;
    using Opal.ViewModels;

    /// <summary>
    /// 値記憶可能なコレクションビューモデルクラスです。
    /// </summary>
    /// <typeparam name="T">コレクション対象のテンプレートの型です。</typeparam>
    public class RecordableCollectionViewModel<T> : ObservableCollection<T>, INotifyPropertyValueChanged, IDisposable
    {
        private readonly List<INotifyPropertyValueChanged> properties =
            new List<INotifyPropertyValueChanged>();

        private readonly object syncRoot = new object();
        private int initCount = 0;
        private bool isValueChanged = false;
        private bool disposed = false;

        /// <summary>
        /// デストラクタです。
        /// </summary>
        ~RecordableCollectionViewModel()
        {
            //// Dispose前に実行された場合はアサート
            Debug.Assert(!this.disposed);
        }

        /// <summary>
        /// プロパティ値変更時のイベントです。
        /// </summary>
        internal event EventHandler<NotifyPropertyValueChangedEventArgs> PropertyValueChanged;

        /// <summary>
        /// 値が変更されたか判定します。
        /// </summary>
        public bool IsValueChanged
        {
            get
            {
                return this.isValueChanged;
            }

            protected set
            {
                this.SetProperty(ref this.isValueChanged, value, () => this.UpdateState());
            }
        }

        /// <summary>
        /// オブジェクトを破棄します。
        /// </summary>
        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// 初期カウントを設定します。
        /// </summary>
        /// <param name="initCount">設定する初期カウントです。</param>
        public void SetInitCount(int initCount)
        {
            this.initCount = initCount;
        }

        /// <summary>
        /// 値が変更されたか状態を更新します。
        /// </summary>
        public void UpdateState()
        {
            if (this.PropertyValueChanged != null)
            {
                this.PropertyValueChanged(this, new NotifyPropertyValueChangedEventArgs(this.IsValueChanged));
            }
        }

        /// <summary>
        /// オブジェクト破棄の内部処理です。継承した先で固有の処理を実装します。
        /// </summary>
        protected virtual void DisposeInternal()
        {
            lock (this.syncRoot)
            {
                foreach (var property in this.properties)
                {
                    var notifyProperty = property as ObservableValueViewModel;
                    if (notifyProperty != null)
                    {
                        notifyProperty.PropertyValueChanged -= this.CallPropertyValueChanged;
                    }
                }

                this.Clear();
            }
        }

        /// <summary>
        /// オブジェクトを破棄します。
        /// </summary>
        /// <param name="disposing">true時に破棄処理を行います。</param>
        protected void Dispose(bool disposing)
        {
            if (this.disposed)
            {
                return;
            }

            if (disposing)
            {
                this.DisposeInternal();
            }

            this.disposed = true;
        }

        /// <summary>
        /// プロパティ変更イベントの発動です。
        /// </summary>
        /// <param name="propertyName">変更したプロパティの名前です。</param>
        protected virtual void RaisePropertyChanged([CallerMemberName] string propertyName = null)
        {
            var args = EventArgsFactory.GetPropertyChangedEventArgs(propertyName);
            this.OnPropertyChanged(args);
        }

        /// <summary>
        /// プロパティ変更イベントの発動です。
        /// </summary>
        /// <typeparam name="TProperty">プロパティの型です。</typeparam>
        /// <param name="propertyNameExpression">変更したプロパティの名前の式です。</param>
        protected virtual void RaisePropertyChanged<TProperty>(
            Expression<Func<TProperty>> propertyNameExpression)
        {
            Debug.Assert(propertyNameExpression != null);

            MemberExpression memberExpression =
                propertyNameExpression.Body as MemberExpression;
            Debug.Assert(memberExpression != null);

            this.RaisePropertyChanged(memberExpression.Member.Name);
        }

        /// <summary>
        /// プロパティを設定します。
        /// </summary>
        /// <typeparam name="TProperty">設定対象のプロパティの型です。</typeparam>
        /// <param name="item">設定対象です。</param>
        /// <param name="value">設定する値です。</param>
        /// <param name="propertyName">設定するプロパティ名です。</param>
        protected void SetProperty<TProperty>(ref TProperty item, TProperty value, [CallerMemberName] string propertyName = null)
        {
            if (!EqualityComparer<TProperty>.Default.Equals(item, value))
            {
                item = value;
                this.RaisePropertyChanged(propertyName);
            }
        }

        /// <summary>
        /// プロパティを強制設定します。
        /// </summary>
        /// <typeparam name="TProperty">設定対象のプロパティの型です。</typeparam>
        /// <param name="item">設定対象です。</param>
        /// <param name="value">設定する値です。</param>
        /// <param name="propertyName">設定するプロパティ名です。</param>
        protected void SetPropertyForce<TProperty>(ref TProperty item, TProperty value, [CallerMemberName] string propertyName = null)
        {
            item = value;
            this.RaisePropertyChanged(propertyName);
        }

        /// <summary>
        /// プロパティを設定します。
        /// </summary>
        /// <typeparam name="TProperty">設定対象のプロパティの型です。</typeparam>
        /// <param name="item">設定対象です。</param>
        /// <param name="value">設定する値です。</param>
        /// <param name="callback">設定後に実行されるコールバックです。</param>
        /// <param name="propertyName">設定するプロパティ名です。</param>
        protected void SetProperty<TProperty>(ref TProperty item, TProperty value, Action callback, [CallerMemberName] string propertyName = null)
        {
            Debug.Assert(callback != null);

            if (!EqualityComparer<TProperty>.Default.Equals(item, value))
            {
                item = value;
                this.RaisePropertyChanged(propertyName);
                callback();
            }
        }

        /// <summary>
        /// プロパティを強制設定します。
        /// </summary>
        /// <typeparam name="TProperty">設定対象のプロパティの型です。</typeparam>
        /// <param name="item">設定対象です。</param>
        /// <param name="value">設定する値です。</param>
        /// <param name="callback">設定後に実行されるコールバックです。</param>
        /// <param name="propertyName">設定するプロパティ名です。</param>
        protected void SetPropertyForce<TProperty>(ref TProperty item, TProperty value, Action callback, [CallerMemberName] string propertyName = null)
        {
            Debug.Assert(callback != null);

            item = value;
            this.RaisePropertyChanged(propertyName);
            callback();
        }

        /// <summary>
        /// コレクションの変更時に実行されます。
        /// </summary>
        /// <param name="e">イベント引数クラスです。</param>
        protected override void OnCollectionChanged(NotifyCollectionChangedEventArgs e)
        {
            this.IsValueChanged = !(this.initCount == this.Count);

            base.OnCollectionChanged(e);
        }

        private void CallPropertyValueChanged(object sender, NotifyPropertyValueChangedEventArgs e)
        {
            if (e.IsValueChanged)
            {
                this.IsValueChanged |= e.IsValueChanged;
            }
            else
            {
                if (this.properties.FindAll(param => param.IsValueChanged == false).Count == this.properties.Count)
                {
                    this.IsValueChanged = false;
                }
            }
        }
    }
}
